% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcmc-distributions.R
\name{MCMC-distributions}
\alias{MCMC-distributions}
\alias{mcmc_hist}
\alias{mcmc_dens}
\alias{mcmc_hist_by_chain}
\alias{mcmc_dens_overlay}
\alias{mcmc_dens_chains}
\alias{mcmc_dens_chains_data}
\alias{mcmc_violin}
\title{Histograms and kernel density plots of MCMC draws}
\usage{
mcmc_hist(x, pars = character(), regex_pars = character(),
  transformations = list(), facet_args = list(), ..., binwidth = NULL,
  freq = TRUE)

mcmc_dens(x, pars = character(), regex_pars = character(),
  transformations = list(), facet_args = list(), ..., trim = FALSE)

mcmc_hist_by_chain(x, pars = character(), regex_pars = character(),
  transformations = list(), facet_args = list(), ..., binwidth = NULL,
  freq = TRUE)

mcmc_dens_overlay(x, pars = character(), regex_pars = character(),
  transformations = list(), facet_args = list(), color_chains = TRUE, ...,
  trim = FALSE)

mcmc_dens_chains(x, pars = character(), regex_pars = character(),
  transformations = list(), color_chains = TRUE, ..., bw = NULL,
  adjust = NULL, kernel = NULL, n_dens = NULL)

mcmc_dens_chains_data(x, pars = character(), regex_pars = character(),
  transformations = list(), ..., bw = NULL, adjust = NULL,
  kernel = NULL, n_dens = NULL)

mcmc_violin(x, pars = character(), regex_pars = character(),
  transformations = list(), facet_args = list(), ..., probs = c(0.1, 0.5,
  0.9))
}
\arguments{
\item{x}{A 3-D array, matrix, list of matrices, or data frame of MCMC draws.
The \link{MCMC-overview} page provides details on how to specify each these
allowed inputs.}

\item{pars}{An optional character vector of parameter names. If neither
\code{pars} nor \code{regex_pars} is specified then the default is to use
\emph{all} parameters.}

\item{regex_pars}{An optional \link[=grep]{regular expression} to use for
parameter selection. Can be specified instead of \code{pars} or in addition
to \code{pars}.}

\item{transformations}{Optionally, transformations to apply to parameters
before plotting. If \code{transformations} is a function or a single string
naming a function then that function will be used to transform all
parameters. To apply transformations to particular parameters, the
\code{transformations} argument can be a named list with length equal to
the number of parameters to be transformed. Currently only univariate
transformations of scalar parameters can be specified (multivariate
transformations will be implemented in a future release). If
\code{transformations} is a list, the name of each list element should be a
parameter name and the content of each list element should be a function
(or any item to match as a function via \code{\link{match.fun}}, e.g. a
string naming a function). If a function is specified by its name as a
string (e.g. \code{"log"}), then it can be used to construct a new
parameter label for the appropriate parameter (e.g. \code{"log(sigma)"}).
If a function itself is specified (e.g. \code{log} or \code{function(x)
log(x)}) then \code{"t"} is used in the new parameter label to indicate
that the parameter is transformed (e.g. \code{"t(sigma)"}).}

\item{facet_args}{A named list of arguments (other than \code{facets}) passed
to \code{\link[ggplot2]{facet_wrap}} or \code{\link[ggplot2]{facet_grid}}
to control faceting.}

\item{...}{Currently ignored.}

\item{binwidth}{An optional value used as the \code{binwidth} argument to
\code{\link[ggplot2]{geom_histogram}} to override the default binwidth.}

\item{freq}{For histograms, \code{freq=TRUE} (the default) puts count on the
y-axis. Setting \code{freq=FALSE} puts density on the y-axis. (For many
plots the y-axis text is off by default. To view the count or density
labels on the y-axis see the \code{\link{yaxis_text}} convenience
function.)}

\item{trim}{A logical scalar passed to \code{\link[ggplot2]{geom_density}}.}

\item{color_chains}{option for whether to separately color chains.}

\item{bw, adjust, kernel, n_dens}{Optional arguments passed to
\code{\link[stats]{density}} to override default kernel density estimation
parameters. \code{n_dens} defaults to 1024.}

\item{probs}{A numeric vector passed to \code{\link[ggplot2]{geom_violin}}'s
\code{draw_quantiles} argument to specify at which quantiles to draw
horizontal lines. Set to \code{NULL} to remove the lines.}
}
\value{
A ggplot object that can be further customized using the
  \pkg{ggplot2} package.
}
\description{
Various types of histograms and kernel density plots of MCMC draws. See the
\strong{Plot Descriptions} section, below, for details.
}
\section{Plot Descriptions}{

\describe{
  \item{\code{mcmc_hist}}{
   Histograms of posterior draws with all chains merged.
  }
  \item{\code{mcmc_dens}}{
   Kernel density plots of posterior draws with all chains merged.
  }
  \item{\code{mcmc_hist_by_chain}}{
   Histograms of posterior draws with chains separated via faceting.
  }
  \item{\code{mcmc_dens_overlay}}{
   Kernel density plots of posterior draws with chains separated but
   overlaid on a single plot.
  }
  \item{\code{mcmc_violin}}{
   The density estimate of each chain is plotted as a violin with
   horizontal lines at notable quantiles.
  }
  \item{\code{mcmc_dens_chains}}{
   Ridgeline kernel density plots of posterior draws with chains separated
   but overlaid on a single plot. In \code{mcmc_dens_overlay} parameters
   appear in separate facets; in \code{mcmc_dens_chains} they appear in the
   same panel and can overlap vertically.
  }
}
}

\examples{
set.seed(9262017)
# some parameter draws to use for demonstration
x <- example_mcmc_draws()
dim(x)
dimnames(x)

##################
### Histograms ###
##################

# histograms of all parameters
color_scheme_set("brightblue")
mcmc_hist(x)

# histograms of some parameters
color_scheme_set("pink")
mcmc_hist(x, pars = c("alpha", "beta[2]"))
\donttest{
mcmc_hist(x, pars = "sigma", regex_pars = "beta")
}
# example of using 'transformations' argument to plot log(sigma),
# and parsing facet labels (e.g. to get greek letters for parameters)
mcmc_hist(x, transformations = list(sigma = "log"),
          facet_args = list(labeller = ggplot2::label_parsed)) +
          facet_text(size = 15)
\donttest{
# instead of list(sigma = "log"), you could specify the transformation as
# list(sigma = log) or list(sigma = function(x) log(x)), but then the
# label for the transformed sigma is 't(sigma)' instead of 'log(sigma)'
mcmc_hist(x, transformations = list(sigma = log))

# separate histograms by chain
color_scheme_set("pink")
mcmc_hist_by_chain(x, regex_pars = "beta")
}

#################
### Densities ###
#################

mcmc_dens(x, pars = c("sigma", "beta[2]"),
          facet_args = list(nrow = 2))
\donttest{
# separate and overlay chains
color_scheme_set("mix-teal-pink")
mcmc_dens_overlay(x, pars = c("sigma", "beta[2]"),
                  facet_args = list(nrow = 2)) +
                  facet_text(size = 14)
x2 <- example_mcmc_draws(params = 6)
mcmc_dens_chains(x2, pars = c("beta[1]", "beta[2]", "beta[3]"))
}
# separate chains as violin plots
color_scheme_set("green")
mcmc_violin(x) + panel_bg(color = "gray20", size = 2, fill = "gray30")

}
\seealso{
Other MCMC: \code{\link{MCMC-combos}},
  \code{\link{MCMC-diagnostics}},
  \code{\link{MCMC-intervals}}, \code{\link{MCMC-nuts}},
  \code{\link{MCMC-overview}}, \code{\link{MCMC-parcoord}},
  \code{\link{MCMC-recover}},
  \code{\link{MCMC-scatterplots}},
  \code{\link{MCMC-traces}}
}
