% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scores.grid.notime.R
\name{scores.grid.notime}
\alias{scores.grid.notime}
\title{Scores for gridded reference data that do not have a varying time dimension}
\usage{
scores.grid.notime(long.name, nc.mod, nc.ref, mod.id, ref.id,
  unit.conv.mod, unit.conv.ref, variable.unit, score.weights = c(1, 2, 1,
  1, 1), outlier.factor = 1000, irregular = FALSE,
  my.projection = "+proj=ob_tran +o_proj=longlat +o_lon_p=83. +o_lat_p=42.5 +lon_0=263.",
  numCores = 2, period = c("1980-01", "2017-12"), timeInt = "month",
  outputDir = FALSE, variable.name = FALSE)
}
\arguments{
\item{long.name}{A string that gives the full name of the variable, e.g. 'Gross primary productivity'}

\item{nc.mod}{A string that gives the path and name of the netcdf file that contains the model output, e.g. '/home/model_gpp.nc'}

\item{nc.ref}{A string that gives the path and name of the netcdf file that contains the reference data output, e.g. '/home/reference_gpp.nc'}

\item{mod.id}{A string that identifies the source of the reference data set, e.g. 'CanESM2'}

\item{ref.id}{A string that identifies the source of the reference data set, e.g. 'MODIS'}

\item{unit.conv.mod}{A number that is used as a factor to convert the unit of the model data, e.g. 86400}

\item{unit.conv.ref}{A number that is used as a factor to convert the unit of the reference data, e.g. 86400}

\item{variable.unit}{A string that gives the final units using LaTeX notation, e.g. 'gC m$^{-2}$ day$^{-1}$'}

\item{score.weights}{R object that gives the weights of each score (\eqn{S_{bias}}, \eqn{S_{rmse}}, \eqn{S_{phase}}, \eqn{S_{iav}}, \eqn{S_{dist}})
that are used for computing the overall score, e.g. c(1,2,1,1,1)}

\item{outlier.factor}{A number that is used to define outliers, e.g. 10.
Plotting raster objects that contain extreme outliers lead to figures where
most grid cells are presented by a single color since the color legend covers
the entire range of values. To avoid this, the user may define outliers that
will be masked out and marked with a red dot. Outliers are all values that
exceed the interquartile range multiplied by the outlier factor defined here.}

\item{irregular}{Logical. If TRUE the data are converted from an irregular to a regular grid. Default is FALSE.}

\item{my.projection}{A string that defines the projection of the irregular grid}

\item{numCores}{An integer that defines the number of cores, e.g. 2}

\item{period}{An R object that gives the period over which to average the model data, e.g. c('1980-01', '2017-12')}

\item{timeInt}{A string that gives the time interval of the model data, e.g. 'month' or 'year'}

\item{outputDir}{A string that gives the output directory, e.g. '/home/project/study'. The output will only be written if the user specifies an output directory.}

\item{variable.name}{A string with the variable name, e.g. 'GPP'. If FALSE, the variable name stored in the NetCDF file will be used instead. Default is FALSE.}
}
\value{
(1) A list that contains three elements. The first element is a
raster stack with model data
(mean, \eqn{mod.mean}),
reference data
(mean, \eqn{ref.mean}),
and the corresponding bias
(bias, \eqn{bias}). The second and third element of the list are spatial
point data frames that give the model and reference outliers, respectively.
The content of the list can be plotted using \link{plotGrid}.

(2) NetCDF files for each of the statistical variables listed above.
(3) Three text files: (i) score values and (ii) score inputs averaged across
the entire grid, and (iii) score values for each individual grid cell.
}
\description{
This function compares model output against remote-sensing
based reference data that do not vary in time. The performance of a model is
expressed through a score that ranges from zero to one, where increasing values
imply better performance.
Contrary to the function \link{scores.grid.time}, only two scores are computed
(bias score \eqn{S_{bias}} and spatial distribution score, \eqn{S_{dist}}) since the reference data do
not vary with time. Contrary to \link{scores.grid.time}, the bias is relative to the absolute reference mean
value rather than the reference standard deviation. Again, this is because the reference data do
not vary with time:

\eqn{(i) \ bias(\lambda, \phi)=\overline{v_{mod}}(\lambda, \phi)-\overline{v_{ref}}(\lambda, \phi)}

\eqn{(ii) \ \varepsilon_{bias}=|bias(\lambda, \phi)|/|\overline{v_{ref}}(\lambda, \phi)|}

\eqn{(iii) \ s_{bias}(\lambda, \phi)=e^{-\varepsilon_{bias}(\lambda, \phi)}}

\eqn{(iv) \ S_{bias}=\overline{\overline{s_{bias}}}}
}
\examples{
library(amber)
library(classInt)
library(doParallel)
library(foreach)
library(Hmisc)
library(latex2exp)
library(ncdf4)
library(parallel)
library(raster)
library(rgdal)
library(rgeos)
library(scico)
library(sp)
library(stats)
library(utils)
library(viridis)
library(xtable)

# (1) Global plots on a regular grid
long.name <- 'Soil Carbon'
nc.mod <- system.file('extdata/modelRegular', 'cSoil_monthly.nc', package = 'amber')
nc.ref <- system.file('extdata/referenceRegular', 'soilc_HWSD_128x64.nc', package = 'amber')
mod.id <- 'CLASSIC' # define a model experiment ID
ref.id <- 'HWSD' # give reference dataset a name
unit.conv.mod <- 1 # optional unit conversion for model data
unit.conv.ref <- 1 # optional unit conversion for reference data
variable.unit <- 'kgC m$^{-2}$' # unit after conversion (LaTeX notation)

# Short version using default settings:
plot.me <- scores.grid.notime(long.name, nc.mod, nc.ref, mod.id, ref.id,
unit.conv.mod, unit.conv.ref, variable.unit)
plotGrid(long.name, plot.me)

\donttest{
# (2) Regional plots on a rotated grid
long.name <- 'Soil Carbon'
nc.mod <- system.file('extdata/modelRotated', 'cSoil_monthly.nc', package = 'amber')
nc.ref <- system.file('extdata/referenceRotated', 'soilc_HWSD_rotated.nc', package = 'amber')
mod.id <- 'CLASSIC' # define a model experiment ID
ref.id <- 'HWSD' # give reference dataset a name
unit.conv.mod <- 1 # optional unit conversion for model data
unit.conv.ref <- 1 # optional unit conversion for reference data
variable.unit <- 'kgC m$^{-2}$' # unit after conversion (LaTeX notation)
score.weights <- c(1,2,1,1,1) # score weights of S_bias, S_rmse, S_phase, S_iav, S_dist
outlier.factor <- 1000
irregular <- TRUE
my.projection <- '+proj=ob_tran +o_proj=longlat +o_lon_p=83. +o_lat_p=42.5 +lon_0=263.'
numCores <- 2
period <- c('1980-01', '2017-12') # period over which to average the model data

plot.me <- scores.grid.notime(long.name, nc.mod, nc.ref, mod.id, ref.id, unit.conv.mod,
unit.conv.ref, variable.unit, score.weights, outlier.factor, irregular, my.projection,
numCores, period)

# Plot results:
irregular <- TRUE # data is on an irregular grid
my.projection <- '+proj=ob_tran +o_proj=longlat +o_lon_p=83. +o_lat_p=42.5 +lon_0=263.'
# shp.filename <- system.file('extdata/ne_50m_admin_0_countries/ne_50m_admin_0_countries.shp',
#  package = 'amber')
shp.filename <- system.file("extdata/ne_110m_land/ne_110m_land.shp", package = "amber")
my.xlim <- c(-171, 0) # longitude range that you wish to plot
my.ylim <- c(32, 78) # latitude range that you wish to plot
plot.width <- 7 # plot width
plot.height <- 3.8 # plot height

plotGrid(long.name, plot.me, irregular, my.projection,
shp.filename, my.xlim, my.ylim, plot.width, plot.height)
} #donttest
}
