#' Matrix of the product between the pairwise comparison value and pj/pi
#'
#' @author Frankie Cho
#'
#' @description Consider the comparison matrix where element \eqn{a_{ij}} contains the pairwise comparison between the attributes i and j. The weights of the matrix was constructed as in `agg.indpref` using the Perron eigenvector where \eqn{p_{i}} and \eqn{p_{j}} are the weights of the \eqn{i^{th}} and the \eqn{j^{th}} element respectively. `ahp.error` constructs a matrix \eqn{\epsilon_{ij} = a_{ij}p_{j}/p_{i}}.
#'
#' @param ahpmat A list of pairwise comparison matrices of each decision maker generated by `ahp.mat`.
#' @param atts a list of attributes in the correct order
#' @param reciprocal whether to remove all numbers lower than 1 and put all numbers above 1 in the upper triangular matrix. Useful for visualizing the inconsistency rapidly.
#'
#' @return A list of matrices containing \eqn{\epsilon_{ij} = a_{ij}p_{j}/p_{i}} for each decision-maker, with elements from the lower triangle set as NA automatically (since it is essentially equal to the element in the upper triangle).
#'
#'@include ahp_mat.R
#'
#' @examples
#'
#'
#' atts <- c('cult', 'fam', 'house', 'jobs', 'trans')
#'
#' data(city200)
#' 
#' cityahp <-  ahp.mat(city200, atts, negconvert = TRUE)
#' ahp.error(cityahp, atts)
#'
#'
#'@references
#'
#'\insertRef{Saaty2004}{ahpsurvey}
#'
#'@include ahp_indpref.R
#'
#'@export

ahp.error <- function(ahpmat, atts, reciprocal = FALSE) {
    indpref.df <- ahp.indpref(ahpmat, atts, method = "eigen")
    respmat <- ahpmat
    conserror <- list()
    for (ind in 1:length(ahpmat)) {
        indpref <- t(indpref.df[1, ])
        rownames(indpref) <- NULL
        indpref <- as.vector(indpref)
        
        ## Matrix where perfect consistency holds
        pjpi <- indpref %*% (t(indpref))^-1
        
        .conserror <- respmat[[ind]] * t(pjpi)
        
        ## Setting the column names of the inconsistency matrix
        colnames(.conserror) <- colnames(pjpi) <- atts
        rownames(.conserror) <- rownames(pjpi) <- atts
        
        if (reciprocal == TRUE){
          for (i in 1:length(atts)){
            for (j in 1:length(atts)){
              if (.conserror[i,j] < 1){
                .conserror[i,j] <- .conserror[j,i]
              }
            }
          }
          .conserror[lower.tri(.conserror)] <- NA
        }
        
        conserror[[ind]] <- .conserror
    }
    conserror
}
