% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_var_dthcaus.R
\name{derive_var_dthcaus}
\alias{derive_var_dthcaus}
\alias{dthcaus_source}
\title{Derive Death Cause}
\usage{
derive_var_dthcaus(
  dataset,
  ...,
  source_datasets,
  subject_keys = vars(STUDYID, USUBJID)
)

dthcaus_source(
  dataset_name,
  filter,
  date,
  order = NULL,
  mode = "first",
  dthcaus,
  traceability_vars = NULL
)
}
\arguments{
\item{dataset}{Input dataset.

The variables specified by \code{subject_keys} are required.}

\item{...}{Objects of class "dthcaus_source" created by \code{\link[=dthcaus_source]{dthcaus_source()}}.}

\item{source_datasets}{A named \code{list} containing datasets in which to search for the
death cause}

\item{subject_keys}{Variables to uniquely identify a subject

A list of quosures where the expressions are symbols as returned by
\code{vars()} is expected.}

\item{dataset_name}{The name of the dataset, i.e. a string, used to search for
the death cause.}

\item{filter}{An expression used for filtering \code{dataset}.}

\item{date}{A date or datetime variable to be used for sorting \code{dataset}.}

\item{order}{Sort order

Additional variables to be used for sorting the \code{dataset} which is ordered by the
\code{date} and \code{order}. Can be used to avoid duplicate record warning.

\emph{Default}: \code{NULL}

\emph{Permitted Values}: list of variables or \verb{desc(<variable>)} function calls
created by \code{vars()}, e.g., \code{vars(ADT, desc(AVAL))} or \code{NULL}}

\item{mode}{One of \code{"first"} or \code{"last"}.
Either the \code{"first"} or \code{"last"} observation is preserved from the \code{dataset}
which is ordered by \code{date}.}

\item{dthcaus}{A variable name or a string literal --- if a variable name, e.g., \code{AEDECOD},
it is the variable in the source dataset to be used to assign values to
\code{DTHCAUS}; if a string literal, e.g. \code{"Adverse Event"}, it is the fixed value
to be assigned to \code{DTHCAUS}.}

\item{traceability_vars}{A named list returned by \code{\link[=vars]{vars()}} listing the traceability variables,
e.g. \code{vars(DTHDOM = "DS", DTHSEQ = DSSEQ)}.
The left-hand side (names of the list elements) gives the names of the traceability variables
in the returned dataset.
The right-hand side (values of the list elements) gives the values of the traceability variables
in the returned dataset.
These can be either strings or symbols referring to existing variables.}
}
\value{
\code{derive_var_dthcaus()} returns the input dataset with \code{DTHCAUS} variable added.

\code{dthcaus_source()} returns an object of class "dthcaus_source".
}
\description{
Derive death cause (\code{DTHCAUS}) and add traceability variables if required.
}
\details{
This function derives \code{DTHCAUS} along with the user-defined traceability
variables, if required. If a subject has death info from multiple sources,
the one from the source with the earliest death date will be used. If dates are
equivalent, the first source will be kept, so the user should provide the inputs in
the preferred order.
}
\section{Functions}{
\itemize{
\item \code{dthcaus_source}: Create objects of class "dthcaus_source"
}}

\examples{
library(tibble)
library(dplyr)
library(lubridate)

adsl <- tribble(
  ~STUDYID,  ~USUBJID,
  "STUDY01", "PAT01",
  "STUDY01", "PAT02",
  "STUDY01", "PAT03"
)
ae <- tribble(
  ~STUDYID,  ~USUBJID, ~AESEQ, ~AEDECOD,       ~AEOUT,  ~AEDTHDTC,
  "STUDY01", "PAT01",  12,     "SUDDEN DEATH", "FATAL", "2021-04-04"
) \%>\%
  mutate(
    AEDTHDT = ymd(AEDTHDTC)
  )
ds <- tribble(
  ~STUDYID, ~USUBJID, ~DSSEQ, ~DSDECOD, ~DSTERM, ~DSSTDTC,
  "STUDY01", "PAT02", 1, "INFORMED CONSENT OBTAINED", "INFORMED CONSENT OBTAINED", "2021-04-03",
  "STUDY01", "PAT02", 2, "RANDOMIZATION", "RANDOMIZATION", "2021-04-11",
  "STUDY01", "PAT02", 3, "DEATH", "DEATH DUE TO PROGRESSION OF DISEASE", "2022-02-01",
  "STUDY01", "PAT03", 1, "DEATH", "POST STUDY REPORTING OF DEATH", "2022-03-03"
) \%>\%
  mutate(
    DSSTDT = ymd(DSSTDTC)
  )

# Derive `DTHCAUS` only - for on-study deaths only
src_ae <- dthcaus_source(
  dataset_name = "ae",
  filter = AEOUT == "FATAL",
  date = AEDTHDT,
  mode = "first",
  dthcaus = AEDECOD
)

src_ds <- dthcaus_source(
  dataset_name = "ds",
  filter = DSDECOD == "DEATH" & grepl("DEATH DUE TO", DSTERM),
  date = DSSTDT,
  mode = "first",
  dthcaus = DSTERM
)

derive_var_dthcaus(adsl, src_ae, src_ds, source_datasets = list(ae = ae, ds = ds))

# Derive `DTHCAUS` and add traceability variables - for on-study deaths only
src_ae <- dthcaus_source(
  dataset_name = "ae",
  filter = AEOUT == "FATAL",
  date = AEDTHDT,
  mode = "first",
  dthcaus = AEDECOD,
  traceability_vars = vars(DTHDOM = "AE", DTHSEQ = AESEQ)
)

src_ds <- dthcaus_source(
  dataset_name = "ds",
  filter = DSDECOD == "DEATH" & grepl("DEATH DUE TO", DSTERM),
  date = DSSTDT,
  mode = "first",
  dthcaus = DSTERM,
  traceability_vars = vars(DTHDOM = "DS", DTHSEQ = DSSEQ)
)

derive_var_dthcaus(adsl, src_ae, src_ds, source_datasets = list(ae = ae, ds = ds))

# Derive `DTHCAUS` as above - now including post-study deaths with different `DTHCAUS` value
src_ae <- dthcaus_source(
  dataset_name = "ae",
  filter = AEOUT == "FATAL",
  date = AEDTHDT,
  mode = "first",
  dthcaus = AEDECOD,
  traceability_vars = vars(DTHDOM = "AE", DTHSEQ = AESEQ)
)

src_ds <- dthcaus_source(
  dataset_name = "ds",
  filter = DSDECOD == "DEATH" & grepl("DEATH DUE TO", DSTERM),
  date = DSSTDT,
  mode = "first",
  dthcaus = DSTERM,
  traceability_vars = vars(DTHDOM = "DS", DTHSEQ = DSSEQ)
)

src_ds_post <- dthcaus_source(
  dataset_name = "ds",
  filter = DSDECOD == "DEATH" & DSTERM == "POST STUDY REPORTING OF DEATH",
  date = DSSTDT,
  mode = "first",
  dthcaus = "POST STUDY: UNKNOWN CAUSE",
  traceability_vars = vars(DTHDOM = "DS", DTHSEQ = DSSEQ)
)

derive_var_dthcaus(adsl, src_ae, src_ds, src_ds_post, source_datasets = list(ae = ae, ds = ds))
}
\seealso{
ADSL Functions that returns variable appended to dataset: 
\code{\link{derive_var_age_years}()},
\code{\link{derive_var_disposition_status}()},
\code{\link{derive_var_extreme_dtm}()},
\code{\link{derive_var_extreme_dt}()},
\code{\link{derive_vars_aage}()},
\code{\link{derive_vars_disposition_reason}()}

Source Specifications: 
\code{\link{assert_db_requirements}()},
\code{\link{assert_terms}()},
\code{\link{assert_valid_queries}()},
\code{\link{censor_source}()},
\code{\link{date_source}()},
\code{\link{death_event}},
\code{\link{event_source}()},
\code{\link{extend_source_datasets}()},
\code{\link{filter_date_sources}()},
\code{\link{format.sdg_select}()},
\code{\link{format.smq_select}()},
\code{\link{list_tte_source_objects}()},
\code{\link{params}()},
\code{\link{query}()},
\code{\link{sdg_select}()},
\code{\link{smq_select}()},
\code{\link{tte_source}()},
\code{\link{validate_query}()},
\code{\link{validate_sdg_select}()},
\code{\link{validate_smq_select}()}
}
\author{
Shimeng Huang, Samia Kabi, Thomas Neitmann, Tamara Senior
}
\concept{der_adsl}
\concept{source_specifications}
\keyword{der_adsl}
\keyword{source_specifications}
