% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_date_vars.R
\name{derive_vars_dt}
\alias{derive_vars_dt}
\title{Derive/Impute a Date from a Date Character Vector}
\usage{
derive_vars_dt(
  dataset,
  new_vars_prefix,
  dtc,
  date_imputation = NULL,
  flag_imputation = "auto",
  min_dates = NULL,
  max_dates = NULL,
  preserve = FALSE
)
}
\arguments{
\item{dataset}{Input dataset.

The date character vector (\code{dtc}) must be present.}

\item{new_vars_prefix}{Prefix used for the output variable(s).

A character scalar is expected. For the date variable "DT" is appended to
the specified prefix and for the date imputation flag "DTF". I.e., for
\code{new_vars_prefix = "AST"} the variables \code{ASTDT} and \code{ASTDTF} are created.}

\item{dtc}{The \code{'--DTC'} date to impute

A character date is expected in a format like \code{yyyy-mm-dd} or
\code{yyyy-mm-ddThh:mm:ss}. If the year part is not recorded (missing date), no
imputation is performed.}

\item{date_imputation}{The value to impute the day/month when a datepart is
missing.

If \code{NULL}: no date imputation is performed and partial dates are returned as
missing.

Otherwise, a character value is expected, either as a
\itemize{
\item format with month and day specified as \code{"mm-dd"}: e.g. \code{"06-15"} for the 15th
of June,
\item or as a keyword: \code{"FIRST"}, \code{"MID"}, \code{"LAST"} to impute to the first/mid/last
day/month.
}

Default is \code{NULL}.}

\item{flag_imputation}{Whether the date imputation flag must also be derived.

If \code{"auto"} is specified, the date imputation flag is derived if the
\code{date_imputation} parameter is not null.

\emph{Default}: \code{"auto"}

\emph{Permitted Values}: \code{"auto"}, \code{"date"} or \code{"none"}}

\item{min_dates}{Minimum dates

A list of dates is expected. It is ensured that the imputed date is not
before any of the specified dates, e.g., that the imputed adverse event start
date is not before the first treatment date. Only dates which are in the
range of possible dates of the \code{dtc} value are considered. The possible dates
are defined by the missing parts of the \code{dtc} date (see example below). This
ensures that the non-missing parts of the \code{dtc} date are not changed.
A date or date-time object is expected.
For example

\if{html}{\out{<div class="sourceCode r">}}\preformatted{impute_dtc(
"2020-11",
min_dates = list(
 ymd_hms("2020-12-06T12:12:12"),
 ymd_hms("2020-11-11T11:11:11")
),
date_imputation = "first"
)
}\if{html}{\out{</div>}}

returns \code{"2020-11-11T11:11:11"} because the possible dates for \code{"2020-11"}
range from \code{"2020-11-01T00:00:00"} to \code{"2020-11-30T23:59:59"}. Therefore
\code{"2020-12-06T12:12:12"} is ignored. Returning \code{"2020-12-06T12:12:12"} would
have changed the month although it is not missing (in the \code{dtc} date).}

\item{max_dates}{Maximum dates

A list of dates is expected. It is ensured that the imputed date is not after
any of the specified dates, e.g., that the imputed date is not after the data
cut off date. Only dates which are in the range of possible dates are
considered. A date or date-time object is expected.}

\item{preserve}{Preserve day if month is missing and day is present

For example \code{"2019---07"} would return \verb{"2019-06-07} if \code{preserve = TRUE}
(and \code{date_imputation = "MID"}).

Permitted Values: \code{TRUE}, \code{FALSE}

Default: \code{FALSE}}
}
\value{
The input dataset with the date \code{'--DT'} (and the date imputation flag \code{'--DTF'}
if requested) added.
}
\description{
Derive a date (\code{'--DT'}) from a date character vector (\verb{'--DTC}').
The date can be imputed (see \code{date_imputation} parameter)
and the date imputation flag ('\verb{--DTF'}) can be added.
}
\details{
The presence of a \code{'--DTF'} variable is checked and if it already exists in the input dataset,
a warning is issued and \code{'--DTF'} will be overwritten.
}
\examples{
library(lubridate)

mhdt <- tibble::tribble(
  ~MHSTDTC,
  "2019-07-18T15:25:40",
  "2019-07-18T15:25",
  "2019-07-18",
  "2019-02",
  "2019",
  "2019---07",
  ""
)

# Create ASTDT and ASTDTF
# no imputation for partial date
derive_vars_dt(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC
)

# Create ASTDT and ASTDTF
# Impute partial dates to first day/month
derive_vars_dt(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC,
  date_imputation = "FIRST"
)

# Impute partial dates to 6th of April
derive_vars_dt(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC,
  date_imputation = "04-06"
)

# Create AENDT and AENDTF
# Impute partial dates to last day/month
derive_vars_dt(
  mhdt,
  new_vars_prefix = "AEN",
  dtc = MHSTDTC,
  date_imputation = "LAST"
)

# Create BIRTHDT
# Impute partial dates to 15th of June. No DTF
derive_vars_dt(
  mhdt,
  new_vars_prefix = "BIRTH",
  dtc = MHSTDTC,
  date_imputation = "MID",
  flag_imputation = "none"
)

# Impute AE start date to the first date and ensure that the imputed date
# is not before the treatment start date
adae <- tibble::tribble(
  ~AESTDTC, ~TRTSDTM,
  "2020-12", ymd_hms("2020-12-06T12:12:12"),
  "2020-11", ymd_hms("2020-12-06T12:12:12")
)

derive_vars_dt(
  adae,
  dtc = AESTDTC,
  new_vars_prefix = "AST",
  date_imputation = "first",
  min_dates = vars(TRTSDTM)
)

# A user imputing dates as middle month/day, i.e. date_imputation = "MID" can
# use preserve argument to "preserve" partial dates.  For example, "2019---07",
# will be displayed as "2019-06-07" rather than 2019-06-15 with preserve = TRUE

derive_vars_dt(
  mhdt,
  new_vars_prefix = "AST",
  dtc = MHSTDTC,
  date_imputation = "MID",
  preserve = TRUE
)
}
\author{
Samia Kabi
}
\keyword{adam}
\keyword{derivation}
\keyword{timing}
