% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/derive_var_trtemfl.R
\name{derive_var_trtemfl}
\alias{derive_var_trtemfl}
\title{Derive Treatment-emergent Flag}
\usage{
derive_var_trtemfl(
  dataset,
  new_var = TRTEMFL,
  start_date = ASTDTM,
  end_date = AENDTM,
  trt_start_date = TRTSDTM,
  trt_end_date = NULL,
  end_window = NULL,
  ignore_time_for_trt_end = TRUE,
  initial_intensity = NULL,
  intensity = NULL
)
}
\arguments{
\item{dataset}{Input dataset

The variables specified by \code{start_date}, \code{end_date}, \code{trt_start_date},
\code{trt_end_date}, \code{initial_intensity}, and \code{intensity} are expected.}

\item{new_var}{New variable}

\item{start_date}{Event start date

\emph{Permitted Values:} A symbol referring to a date or datetime variable of
the input dataset}

\item{end_date}{Event end date

\emph{Permitted Values:} A symbol referring to a date or datetime variable of
the input dataset}

\item{trt_start_date}{Treatment start date

\emph{Permitted Values:} A symbol referring to a date or datetime variable of
the input dataset}

\item{trt_end_date}{Treatment end date

\emph{Permitted Values:} A symbol referring to a date or datetime variable of
the input dataset or \code{NULL}}

\item{end_window}{If the argument is specified, events starting more than the specified
number of days after end of treatment, are not flagged.

\emph{Permitted Values:} A non-negative integer or \code{NULL}}

\item{ignore_time_for_trt_end}{If the argument is set to \code{TRUE}, the time part is ignored for checking if
the event occurred more than \code{end_window} days after end of treatment.

\emph{Permitted Values:} \code{TRUE}, \code{FALSE}}

\item{initial_intensity}{Initial severity/intensity or toxicity

This derivation assumes AE data collection method as single record per AE
with “initial” and “most extreme” severity/intensity recorded separately.

If the argument is specified, events which start before treatment start and
end after treatment start (or are ongoing) and worsened (i.e., the
intensity is greater than the initial intensity), are flagged.

The values of the specified variable must be comparable with the usual
comparison operators. I.e., if the intensity is greater than the initial
intensity \code{initial_intensity < intensity} must evaluate to \code{TRUE}.

\emph{Permitted Values:} A symbol referring to a variable of the input dataset
or \code{NULL}}

\item{intensity}{Severity/intensity or toxicity

If the argument is specified, events which start before treatment start and
end after treatment start (or are ongoing) and worsened (i.e., the
intensity is greater than the initial intensity), are flagged.

The values of the specified variable must be comparable with the usual
comparison operators. I.e., if the intensity is greater than the initial
intensity \code{initial_intensity < intensity} must evaluate to \code{TRUE}.

\emph{Permitted Values:} A symbol referring to a variable of the input dataset
or \code{NULL}}
}
\value{
The input dataset with the variable specified by \code{new_var} added
}
\description{
Derive treatment emergent analysis flag (e.g., \code{TRTEMFL}).
}
\details{
For the derivation of the new variable the following cases are
considered in this order. The first case which applies, defines the value
of the variable.
\itemize{
\item \emph{not treated}: If \code{trt_start_date} is \code{NA}, it is set to \code{NA_character_}.
\item \emph{event before treatment}: If \code{end_date} is before \code{trt_start_date} (and
\code{end_date} is not \code{NA}), it is set to \code{NA_character_}.
\item \emph{no event date}: If \code{start_date} is \code{NA}, it is set to \code{"Y"} as in such
cases it is usually considered more conservative to assume the event was
treatment-emergent.
\item \emph{event started during treatment}:
\itemize{
\item if \code{end_window} is not specified:
if \code{start_date} is on or after \code{trt_start_date}, it is set to \code{"Y"},
\item if \code{end_window} is specified:
if \code{start_date} is on or after \code{trt_start_date} and \code{start_date} is on
or before \code{trt_end_date} + \code{end_window} days, it is set to \code{"Y"},
}
\item \emph{event started before treatment and (possibly) worsened on treatment}:
\itemize{
\item if \code{initial_intensity} and \code{intensity} is specified: if
\code{initial_intensity < intensity} and \code{start_date} is before
\code{trt_start_date} and \code{end_date} is on or after \code{trt_start_date} or
\code{end_date} is \code{NA}, it is set to \code{"Y"}.
}
\item Otherwise it is set to \code{NA_character_}.
}
}
\examples{

library(tibble)
library(dplyr, warn.conflicts = FALSE)
library(lubridate)

adae <- expected <- tribble(
  ~USUBJID, ~ASTDTM,            ~AENDTM,            ~AEITOXGR, ~AETOXGR,
  # before treatment
  "1",      "2021-12-13T20:15", "2021-12-15T12:45", "1",       "1",
  "1",      "2021-12-14T20:15", "2021-12-14T22:00", "1",       "3",
  # starting before treatment and ending during treatment
  "1",      "2021-12-30T20:00", "2022-01-14T11:00", "1",       "3",
  "1",      "2021-12-31T20:15", "2022-01-01T01:23", "1",       "1",
  # starting during treatment
  "1",      "2022-01-01T12:00", "2022-01-02T23:25", "3",       "4",
  # after treatment
  "1",      "2022-05-10T11:00", "2022-05-10T13:05", "2",       "2",
  "1",      "2022-05-11T11:00", "2022-05-11T13:05", "2",       "2",
  # missing dates
  "1",      "",                 "",                 "3",       "4",
  "1",      "2021-12-30T09:00", "",                 "3",       "4",
  "1",      "2021-12-30T11:00", "",                 "3",       "3",
  "1",      "",                 "2022-01-04T09:00", "3",       "4",
  "1",      "",                 "2021-12-24T19:00", "3",       "4",
  "1",      "",                 "2022-06-04T09:00", "3",       "4",
  # without treatment
  "2",      "",                 "2021-12-03T12:00", "1",       "2",
  "2",      "2021-12-01T12:00", "2021-12-03T12:00", "1",       "2",
  "2",      "2021-12-06T18:00", "",                 "1",       "2"
) \%>\%
  mutate(
    ASTDTM = ymd_hm(ASTDTM),
    AENDTM = ymd_hm(AENDTM),
    TRTSDTM = if_else(USUBJID == "1", ymd_hm("2022-01-01T01:01"), ymd_hms("")),
    TRTEDTM = if_else(USUBJID == "1", ymd_hm("2022-04-30T23:59"), ymd_hms(""))
  )

# derive TRTEMFL without considering treatment end and worsening
derive_var_trtemfl(adae) \%>\% select(ASTDTM, AENDTM, TRTSDTM, TRTEMFL)

# derive TRTEM2FL taking treatment end and worsening into account
derive_var_trtemfl(
  adae,
  new_var = TRTEM2FL,
  trt_end_date = TRTEDTM,
  end_window = 10,
  initial_intensity = AEITOXGR,
  intensity = AETOXGR
) \%>\% select(ASTDTM, AENDTM, AEITOXGR, AETOXGR, TRTEM2FL)
}
\seealso{
OCCDS Functions: 
\code{\link{derive_vars_atc}()},
\code{\link{derive_vars_query}()},
\code{\link{get_terms_from_db}()}
}
\concept{der_occds}
\keyword{der_occds}
