% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/floating_catchment_area.R
\name{floating_catchment_area}
\alias{floating_catchment_area}
\title{Floating catchment area accessibility}
\usage{
floating_catchment_area(
  data,
  fca_metric,
  orig_col,
  dest_col,
  population_col,
  opportunity_col,
  decay_function,
  travel_cost_col = "travel_time"
)
}
\arguments{
\item{data}{A \code{data.frame} with a travel cost matrix in long format,
containing the columns with origin ids, destination ids, the travel cost
between each origin-destination pair (this can be either in  terms of
travel time, distance or monetary costs), and number of population and
opportunities located in each origin and destination.}

\item{fca_metric}{A \code{string} indicating which floating catchment area
accessibility measure to use. Options available: \code{"2SFCA"} and \code{"BFCA"}.
More info in the details.}

\item{orig_col}{A \code{string} with the name of the column of origin ids.}

\item{dest_col}{A \code{string} with the name of the column of destination ids.}

\item{population_col}{A \code{string} with the name of the column of origin with
population count.}

\item{opportunity_col}{A \code{string} indicating the name of the column with the
number of opportunities / resources / services to be considered.}

\item{decay_function}{A \code{fuction} that converts travel cost (t_ij) into an
impedance factor used to weight opportunities. For convenience, the package
currently includes the following functions: \code{decay_bineary},
\code{decay_exponential}, \code{decay_exponential} and \code{decay_power}.
See the documentation of each decay function for more details.}

\item{travel_cost_col}{A \code{string} with the name of the column with travel
cost between origins and destinations. It can be either either travel time,
distance or monetary costs. Defaults to \code{travel_ime}.}
}
\value{
A \code{data.table} object.
}
\description{
Calculates accessibility accounting for the competition of resources using one
of the multiple accessibility metrics in the floating catchment area family.
The function currently includes \verb{2SFCA} 2-Step Floating Catchment Area (Luo &
Wang, 2003), and \code{BFCA} Balanced Floating Catchment Area (Paez et al. 2021).
Accessibility can be calculated using multiple alternative decay functions
passed through the \code{decay_function} parameter.
}
\section{2SFCA}{
The 2SFCA measure was the first accessibility metric in the floating
catchment area family. It was originally proposed by Luo & Wang (2003).
}

\section{BFCA}{
The balanced floating catchment area (BFCA) metric calculates accessibility
accounting for competition effects while simultaneously correcting for issues
of inflation of demand and service levels that are present in previous
floating  catchment area measures. BFCA was  was originally proposed by Paez
et al. (2019) and named in Pereira et al. (2021).
\subsection{References:}{
\itemize{
\item Luo, W., & Wang, F. (2003). Measures of spatial accessibility to health
care in a GIS environment: synthesis and a case study in the Chicago region.
Environment and planning B: planning and design, 30(6), 865-884. \doi{10.1068/b29120}.
\item Paez, A., Higgins, C. D., & Vivona, S. F. (2019). Demand and level of
service inflation in Floating Catchment Area (FCA) methods. Plos one, 14(6),
e0218773. \doi{10.1371/journal.pone.0218773}
\item Pereira, R. H., Braga, C. K. V., Servo, L. M., Serra, B., Amaral, P.,
Gouveia, N., & Paez, A. (2021). Geographic access to COVID-19 healthcare in
Brazil using a balanced float catchment area approach. Social Science &
Medicine, 273. \doi{10.1016/j.socscimed.2021.113773}
}
}
}

\examples{
library(accessibility)

# load a travel time matrix data in long format
data_path <- system.file("extdata/ttm_bho.rds", package = "accessibility")
ttm <- readRDS(data_path)

# 2SFCA with a step decay function
df <- floating_catchment_area(
       data = ttm,
       fca_metric = '2SFCA',
       orig_col = 'from_id',
       dest_col = 'to_id',
       opportunity_col = 'jobs',
       population_col = 'population',
       decay_function = decay_binary(cutoff = 50),
       travel_cost_col = 'travel_time'
       )
head(df)

# BFCA with an exponential decay function
df <- floating_catchment_area(
       data = ttm,
       fca_metric = 'BFCA',
       orig_col = 'from_id',
       dest_col = 'to_id',
       opportunity_col = 'jobs',
       population_col = 'population',
       decay_function = decay_exponential(decay_value = 0.5),
       travel_cost_col = 'travel_time'
       )
head(df)

}
\concept{Floating catchment area}
