% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xSocialiserGenes.r
\name{xSocialiserGenes}
\alias{xSocialiserGenes}
\title{Function to calculate pair-wise semantic similarity given a list of genes and the ontology in query}
\usage{
xSocialiserGenes(
data,
check.symbol.identity = F,
ontology = c("GOBP", "GOMF", "GOCC", "DO", "HPPA", "HPMI", "HPCM",
"HPMA", "MP"),
measure = c("BM.average", "BM.max", "BM.complete", "average", "max"),
method.term = c("Resnik", "Lin", "Schlicker", "Jiang", "Pesquita"),
rescale = TRUE,
force = TRUE,
fast = TRUE,
parallel = TRUE,
multicores = NULL,
path.mode = c("all_paths", "shortest_paths", "all_shortest_paths"),
true.path.rule = T,
verbose = T,
RData.location = "http://galahad.well.ox.ac.uk/bigdata",
guid = NULL
)
}
\arguments{
\item{data}{an input vector containing gene symbols}

\item{check.symbol.identity}{logical to indicate whether to match the
input data via Synonyms for those unmatchable by official gene symbols.
By default, it sets to false}

\item{ontology}{the ontology supported currently. It can be "GOBP" for
Gene Ontology Biological Process, "GOMF" for Gene Ontology Molecular
Function, "GOCC" for Gene Ontology Cellular Component, "DO" for Disease
Ontology, "HPPA" for Human Phenotype Phenotypic Abnormality, "HPMI" for
Human Phenotype Mode of Inheritance, "HPCM" for Human Phenotype
Clinical Modifier, "HPMA" for Human Phenotype Mortality Aging, "MP" for
Mammalian Phenotype}

\item{measure}{the measure used to derive semantic similarity between
genes/SNPs from semantic similarity between terms. Take the semantic
similartity between SNPs as an example. It can be "average" for average
similarity between any two terms (one from SNP 1, the other from SNP
2), "max" for the maximum similarity between any two terms,
"BM.average" for best-matching (BM) based average similarity (i.e. for
each term of either SNP, first calculate maximum similarity to any term
in the other SNP, then take average of maximum similarity; the final
BM-based average similiary is the pre-calculated average between two
SNPs in pair), "BM.max" for BM based maximum similarity (i.e. the same
as "BM.average", but the final BM-based maximum similiary is the
maximum of the pre-calculated average between two SNPs in pair),
"BM.complete" for BM-based complete-linkage similarity (inspired by
complete-linkage concept: the least of any maximum similarity between a
term of one SNP and a term of the other SNP). When comparing BM-based
similarity between SNPs, "BM.average" and "BM.max" are sensitive to the
number of terms involved; instead, "BM.complete" is much robust in this
aspect. By default, it uses "BM.average"}

\item{method.term}{the method used to measure semantic similarity
between terms. It can be "Resnik" for information content (IC) of most
informative common ancestor (MICA) (see
\url{http://dl.acm.org/citation.cfm?id=1625914}), "Lin" for 2*IC at
MICA divided by the sum of IC at pairs of terms, "Schlicker" for
weighted version of 'Lin' by the 1-prob(MICA) (see
\url{http://www.ncbi.nlm.nih.gov/pubmed/16776819}), "Jiang" for 1 -
difference between the sum of IC at pairs of terms and 2*IC at MICA
(see \url{https://arxiv.org/pdf/cmp-lg/9709008.pdf}), "Pesquita" for
graph information content similarity related to Tanimoto-Jacard index
(ie. summed information content of common ancestors divided by summed
information content of all ancestors of term1 and term2 (see
\url{http://www.ncbi.nlm.nih.gov/pubmed/18460186}))}

\item{rescale}{logical to indicate whether the resulting values are
rescaled to the range [0,1]. By default, it sets to true}

\item{force}{logical to indicate whether the only most specific terms
(for each SNP) will be used. By default, it sets to true. It is always
advisable to use this since it is computationally fast but without
compromising accuracy (considering the fact that true-path-rule has
been applied when running \code{\link{xDAGanno}})}

\item{fast}{logical to indicate whether a vectorised fast computation
is used. By default, it sets to true. It is always advisable to use
this vectorised fast computation; since the conventional computation is
just used for understanding scripts}

\item{parallel}{logical to indicate whether parallel computation with
multicores is used. By default, it sets to true, but not necessarily
does so. It will depend on whether these two packages "foreach" and
"doParallel" have been installed}

\item{multicores}{an integer to specify how many cores will be
registered as the multicore parallel backend to the 'foreach' package.
If NULL, it will use a half of cores available in a user's computer.
This option only works when parallel computation is enabled}

\item{path.mode}{the mode of paths induced by vertices/nodes with input
annotation data. It can be "all_paths" for all possible paths to the
root, "shortest_paths" for only one path to the root (for each node in
query), "all_shortest_paths" for all shortest paths to the root (i.e.
for each node, find all shortest paths with the equal lengths)}

\item{true.path.rule}{logical to indicate whether the true-path rule
should be applied to propagate annotations. By default, it sets to
true}

\item{verbose}{logical to indicate whether the messages will be
displayed in the screen. By default, it sets to false for no display}

\item{RData.location}{the characters to tell the location of built-in
RData files. See \code{\link{xRDataLoader}} for details}

\item{guid}{a valid (5-character) Global Unique IDentifier for an OSF
project. See \code{\link{xRDataLoader}} for details}
}
\value{
It returns an object of class "igraph", with nodes for input genes and
edges for pair-wise semantic similarity between them. Also added graph
attribute is 'dag' storing the annotated ontology DAG used. If no
similarity is calculuated, it returns NULL.
}
\description{
\code{xSocialiserGenes} is supposed to calculate pair-wise semantic
similarity between a list of input genes and the ontology in query. It
returns an object of class "igraph", a network representation of
socialized genes. Now it supports enrichment analysis using a wide
variety of ontologies such as Gene Ontology and Phenotype Ontologies.
It first calculates semantic similarity between terms and then derives
semantic similarity from term-term semantic similarity. Parallel
computing is also supported.
}
\note{
For the mode "shortest_paths", the induced subgraph is the most
concise, and thus informative for visualisation when there are many
nodes in query, while the mode "all_paths" results in the complete
subgraph.
}
\examples{
\dontrun{
# Load the XGR package and specify the location of built-in data
library(XGR)
RData.location <- "http://galahad.well.ox.ac.uk/bigdata/"

# Gene-based similarity analysis using Mammalian Phenotype Ontology (MP)
# a) provide the input Genes of interest (eg 100 randomly chosen human genes)
## load human genes
org.Hs.eg <- xRDataLoader(RData='org.Hs.eg',
RData.location=RData.location)
data <- as.character(sample(org.Hs.eg$gene_info$Symbol, 100))
data

# b) perform similarity analysis
sim <- xSocialiserGenes(data=data, ontology="MP",
RData.location=RData.location)

# c) save similarity results to the file called 'MP_similarity.txt'
output <- igraph::get.data.frame(sim, what="edges")
utils::write.table(output, file="MP_similarity.txt", sep="\t",
row.names=FALSE)

# d) visualise the gene network
## extract edge weight (with 2-digit precision)
x <- signif(as.numeric(E(sim)$weight), digits=2)
## rescale into an interval [1,4] as edge width
edge.width <- 1 + (x-min(x))/(max(x)-min(x))*3
## do visualisation
xVisNet(g=sim, vertex.shape="sphere", edge.width=edge.width,
edge.label=x, edge.label.cex=0.7)
}
}
\seealso{
\code{\link{xSocialiser}}
}
