% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/function_auto_decompose.R
\name{auto_decompose}
\alias{auto_decompose}
\title{Automatic Decomposition Function}
\usage{
auto_decompose(
  time_series,
  suspected_periods = c(12, 6, 4, 3),
  proportion_of_variance_type = c("leave_out_first", "total"),
  max_proportion_of_variance = 0.995,
  log_ratio_cutoff = 0.2,
  window_length = "auto",
  num_trend_components = 2
)
}
\arguments{
\item{time_series}{A vector which stores the time series of interest in the log scale.}

\item{suspected_periods}{A vector which stores the suspected periods in the descending order of importance. The default option is c(12,6,4,3), corresponding to 12, 6, 4, and 3 months.}

\item{proportion_of_variance_type}{A character string specifying the option for choosing the maximum number of eigenvalues based on the proportion of total variance explained. If "leave_out_first" is chosen, then the contribution made by the first eigenvector is ignored; otherwise, if "total" is chosen, then the contribution made by all the eigenvectors is considered.}

\item{max_proportion_of_variance}{A numeric specifying the proportion of total variance explained using the method specified in proportion_of_variance_type. The default option is 0.995.}

\item{log_ratio_cutoff}{A numeric specifying the threshold for the deviation between the estimated period and candidate periods in suspected_periods. The default option is 0.2, which means that, if the absolute log ratio between the estimated and candidate period is within 0.2 (approximately a 20\% difference), then the estimated period is deemed equal to the candidate period.}

\item{window_length}{A character string or positive integer specifying the window length for the SSA estimation. If "auto" is chosen, then the algorithm automatically selects the window length by taking a multiple of 12 which does not exceed half the length of time_series. The default option is "auto".}

\item{num_trend_components}{A positive integer specifying the number of eigenvectors to be chosen for describing the trend in SSA. The default option is 2.}
}
\value{
\item{reconstruction}{A list containing important information about the reconstructed time series. In particular, it contains the reconstructed main trend component, overall trend component, seasonal component for each period specified in suspected_periods, and overall seasonal component.}
\item{grouping}{A matrix containing information about the locations of the eigenvalue groups for each period in suspected_periods and trend component. The locations are indicated by '1'.}
\item{window_length}{A numeric indicating the window length.}
\item{ts_ssa}{An ssa object storing the singular spectrum analysis decomposition.}
}
\description{
Automatically decomposes a time series using singular spectrum analysis. See package \link[Rssa]{Rssa} for details on singular spectrum analysis.
}
\examples{
data("park_visitation")

### Decompose national parks service visitor counts and flickr photo user-days

# parameters ---------------------------------------------
suspected_periods <- c(12,6,4,3)
proportion_of_variance_type = "leave_out_first"
max_proportion_of_variance <- 0.995
log_ratio_cutoff <- 0.2

# load data ----------------------------------------------

park <- "YELL" #for Yellowstone National Park

nps_ts <- ts(park_visitation[park_visitation$park == park,]$nps, start = 2005, freq = 12)
nps_ts <- log(nps_ts)

pud_ts <- ts(park_visitation[park_visitation$park == park,]$pud, start = 2005, freq = 12)
pud_ts <- log(pud_ts)

# decompose time series and plot decompositions -----------
decomp_pud <- auto_decompose(pud_ts,
                                     suspected_periods,
                                     proportion_of_variance_type = proportion_of_variance_type,
                                     max_proportion_of_variance,
                                     log_ratio_cutoff)
plot(decomp_pud)

decomp_nps <- auto_decompose(nps_ts,suspected_periods,
                                       proportion_of_variance_type = proportion_of_variance_type,
                                     max_proportion_of_variance,log_ratio_cutoff)

plot(decomp_nps)

}
