## SimInf, a framework for stochastic disease spread simulations
## Copyright (C) 2015 - 2017  Stefan Engblom
## Copyright (C) 2015 - 2017  Stefan Widgren
##
## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program.  If not, see <http://www.gnu.org/licenses/>.

##' Create a DESCRIPTION file for the package skeleton
##' @importFrom utils packageVersion
##' @noRd
create_DESCRIPTION_file <- function(path, name, author, maintainer,
                                    email, license)
{
    lines <- c(paste0("Package: ", name),
               "Type: Package",
               paste0("Title: Model ('", name, "') Interface to the 'SimInf' Package"),
               "Version: 1.0",
               paste0("Author: ", author),
               paste0("Maintainer: ", maintainer, " <", email, ">"),
               "Description: Generated by 'SimInf'. Provides an interface",
               paste0("    to the 'SimInf' package for the '", name, "' model."),
               paste0("License: ", license),
               "NeedsCompilation: yes",
               paste0("Depends: SimInf(>= ", packageVersion("SimInf"), ")"),
               "Imports: methods",
               "LinkingTo: SimInf",
               "Collate:",
               "    'model.R'")

    writeLines(lines, con = file.path(path, "DESCRIPTION"))

    invisible(NULL)
}

##' Create a NAMESPACE file for the package skeleton
##' @noRd
create_NAMESPACE_file <- function(path, name)
{
    lines <- c(paste0("export(", name, ")"),
               paste0("exportClasses(", name, ")"),
               "exportMethods(run)",
               "import(SimInf)",
               "import(methods)",
               paste0("useDynLib(", name, ", .registration=TRUE)"))

    writeLines(lines, con = file.path(path, "NAMESPACE"))

    invisible(NULL)
}

create_model_C_file <- function(path, model, name)
{
    writeLines(C_code(model, name), con = file.path(path, "src", "model.c"))
    invisible(NULL)
}

create_model_R_class <- function(name)
{
    c(paste0("##' Class \\code{", name, "}"),
      "##'",
      paste0("##' Class to handle the \\code{", name, "} \\code{SimInf_model}."),
      "##' @export",
      paste0("setClass(\"", name, "\", contains = \"SimInf_model\")"))
}

##' @importFrom utils capture.output
##' @noRd
create_model_R_object <- function(model, name)
{
    rows <- paste0(rownames(model@S), collapse = "\", \"")
    parameters <- paste0(names(model@gdata), collapse = "\", \"")

    ## Dependency graph
    G <- capture.output(dput(as.matrix(model@G)))
    G <- c(paste0("G <- ", G[1]), G[-1])
    G <- paste0("    ", G)

    ## State change matrix
    S <- capture.output(dput(as.matrix(model@S)))
    S <- c(paste0("S <- ", S[1]), S[-1])
    S <- paste0("    ", S)

    ## Select matrix
    E <- capture.output(dput(as.matrix(model@events@E)))
    E <- c(paste0("E <- ", E[1]), E[-1])
    E <- paste0("    ", E)

    ## Shift matrix
    N <- capture.output(dput(as.matrix(model@events@N)))
    N <- c(paste0("N <- ", N[1]), N[-1])
    N <- paste0("    ", N)

    c("##' Create a model for the SimInf framework",
      "##'",
      "##' Create a model to be used by the SimInf framework.",
      "##' @param u0 A data.frame with the initial state in each node.",
      "##' @param tspan A vector (length >= 2) of increasing time points",
      "##'     where the state of each node is to be returned.",
      "##' @param events A data.frame with scheduled events.",
      "##' @param gdata A named numeric vector with rate-constants for the",
      "##'     model.",
      "##' @import SimInf",
      "##' @import methods",
      "##' @export",
      "##' @examples",
      "##' ## Please add example(s) how to use the model",
      paste0(name, " <- function(u0 = NULL, tspan = NULL, events = NULL, gdata = NULL) {"),
      paste0("    compartments <- c(\"", rows, "\")"),
      paste0("    parameters <- c(\"", parameters, "\")"),
      "",
      "    ## Check u0",
      "    if (is.null(u0))",
      "        stop(\"'u0' must be specified.\")",
      "    if (!is.data.frame(u0))",
      "        u0 <- as.data.frame(u0)",
      "    if (!all(compartments %in% names(u0)))",
      "        stop(\"Missing columns in u0\")",
      "    u0 <- u0[, compartments, drop = FALSE]",
      "",
      "    ## Check gdata",
      "    if (is.null(gdata))",
      "        stop(\"'gdata' must be specified.\")",
      "    if (!is.numeric(gdata))",
      "        stop(\"'gdata' must be a named numeric vector.\")",
      "    if (!all(parameters %in% names(gdata)))",
      "        stop(\"Missing parameters in 'gdata'\")",
      "    gdata <- gdata[parameters]",
      "",
      G,
      "",
      S,
      "",
      E,
      "",
      N,
      "",
      "    model <- SimInf_model(G = G, S = S, E = E, N = N, tspan = tspan,",
      "                          events = events, u0 = u0, gdata = gdata)",
      "",
      paste0("    as(model, \"", name, "\")"),
      "}")
}

create_model_run_fn <- function(name)
{
    c("##' Run the model",
      "##'",
      "##' @rdname run-methods",
      "##' @param model The model to run.",
      "##' @param threads Number of threads. Default is NULL, i.e. to use all",
      "##'     available processors.",
      "##' @param solver Which numerical solver to utilize. Default is NULL,",
      "##'     i.e., use the default numerical solver in SimInf.",
      "##' @return SimInf_model with result from simulation.",
      "##' @export",
      paste0("##' @useDynLib ", name, ", .registration=TRUE"),
      "setMethod(\"run\",",
      paste0("    signature(model = \"", name, "\"),"),
      "    function(model, threads = NULL, solver = NULL)",
      "    {",
      "        methods::validObject(model)",
      paste0("       .Call(SimInf_model_run, model, threads, solver, PACKAGE = \"",
             name, "\")"),
      "    })")
}

##' @importFrom utils packageVersion
##' @noRd
create_model_R_file <- function(path, model, name)
{
    lines <- c(sprintf("## Generated by SimInf (v%s) %s */",
                       packageVersion("SimInf"),
                       format(Sys.time(), "%Y-%m-%d %H:%M")),
               "",
               create_model_R_class(name),
               "",
               create_model_R_object(model, name),
               "",
               create_model_run_fn(name))

    writeLines(lines, con = file.path(path, "R", "model.R"))

    invisible(NULL)
}

create_model_class_man_file <- function(path, name)
{
    lines <- c("\\docType{class}",
               paste0("\\name{", name, "-class}"),
               paste0("\\alias{", name, "-class}"),
               paste0("\\title{Class \\code{\"", name, "\"}}"),
               "\\description{",
               paste0("Class to handle the ", name, " \\code{SimInf_model}."),
               "}")

    writeLines(lines, con = file.path(path, "man", paste0(name, "-class.Rd")))

    invisible(NULL)
}

create_model_man_file <- function(path, name)
{
    lines <- c(paste0("\\name{", name, "}"),
               paste0("\\alias{", name, "}"),
               "\\title{Create a model for the SimInf framework}",
               "\\usage{",
               paste0(name, "(u0 = NULL, tspan = NULL, events = NULL, gdata = NULL)"),
               "}",
               "\\arguments{",
               "\\item{u0}{A \\code{data.frame} with the initial state in each node.}",
               "",
               "\\item{tspan}{A vector (length >= 2) of increasing time points",
               "where the state of each node is to be returned.}",
               "\\item{events}{A data.frame with scheduled events.}",
               "\\item{gdata}{A named numeric vector with rate-constants for the",
               "model.}",
               "}",
               "\\description{",
               "Create a model to be used by the SimInf framework.",
               "}",
               "\\examples{",
               "## Please add example(s) how to use the model",
               "}")

    writeLines(lines, con = file.path(path, "man", paste0(name, ".Rd")))

    invisible(NULL)
}

create_model_run_man_file <- function(path, name)
{
    lines <- c("\\docType{methods}",
               paste0("\\name{run,", name, "-method}"),
               paste0("\\alias{run,", name, "-method}"),
               "\\title{Run the model}",
               "\\usage{",
               paste0("\\S4method{run}{", name, "}(model, threads = NULL, solver = NULL)"),
               "}",
               "\\arguments{",
               "\\item{model}{The model to run.}",
               "",
               "\\item{threads}{Number of threads. Default is NULL, i.e. to use all",
               "available processors.}",
               "",
               "\\item{solver}{Which numerical solver to utilize. Default is NULL, i.e.",
               "use the default numerical solver in SimInf.}",
               "}",
               "\\value{",
               "SimInf_model with result from simulation.",
               "}",
               "\\description{",
               "Run the model",
               "}")

    writeLines(lines, con = file.path(path, "man", "run-methods.Rd"))

    invisible(NULL)
}

##' Create a package skeleton from a \code{SimInf_model}
##'
##' Describe your model in a logical way in R, then \code{mparse}
##' creates a \code{\linkS4class{SimInf_model}} object with your model
##' definition that can be installed as an add-on R package.
##' @param model The \code{model} \code{\linkS4class{SimInf_model}}
##'     object with your model to create the package skeleton from.
##' @param name Character string: the package name and directory name
##'     for your package.
##' @param path Path to put the package directory in. Default is '.'
##'     i.e. the current directory.
##' @param author Author of the package.
##' @param email Email of the package maintainer.
##' @param maintainer Maintainer of the package.
##' @param license License of the package. Default is 'GPL-3'.
##' @return invisible \code{NULL}.
##' @export
##' @references Read the \emph{Writing R Extensions} manual for more
##'     details.
##'
##' Once you have created a \emph{source} package you need to install
##' it: see the \emph{R Installation and Administration} manual,
##' \code{\link{INSTALL}} and \code{\link{install.packages}}.
package_skeleton <- function(model, name = NULL, path = ".",
                             author = NULL, email = NULL,
                             maintainer = NULL, license = "GPL-3")
{
    ## Check model argument
    if (missing(model))
        stop("Missing 'model' argument")
    if (!is(model, "SimInf_model"))
        stop("'model' argument is not a 'SimInf_model' object")

    stopifnot(!is.null(name), is.character(name), length(name) == 1,
              nchar(name) > 0)
    stopifnot(!is.null(path), is.character(path), length(path) == 1,
              nchar(path) > 0)
    path <- file.path(path, name)
    if (!is.na(file.info(path)$size))
        stop(paste0("'", path, "' already exists"))

    if (is.null(author))
        author <- "Your Name"
    stopifnot(is.character(author), length(author) == 1, nchar(author) > 0)

    if (is.null(email))
        email <- "your@email.com"
    stopifnot(is.character(email), length(email) == 1, nchar(email) > 0)

    if (is.null(maintainer))
        maintainer <- author
    stopifnot(is.character(maintainer), length(maintainer) == 1, nchar(maintainer) > 0)

    ## Create folder structure
    message("Creating directories ...", domain = NA)
    dir.create(path, recursive = TRUE)
    dir.create(file.path(path, "man"))
    dir.create(file.path(path, "src"))
    dir.create(file.path(path, "R"))

    ## Create files
    message("Creating DESCRIPTION ...", domain = NA)
    create_DESCRIPTION_file(path, name, author, maintainer, email,
                            license)
    message("Creating NAMESPACE ...", domain = NA)
    create_NAMESPACE_file(path, name)
    message("Creating C file ...", domain = NA)
    create_model_C_file(path, model, name)
    message("Creating R file ...", domain = NA)
    create_model_R_file(path, model, name)
    message("Creating help files ...", domain = NA)
    create_model_man_file(path, name)
    create_model_class_man_file(path, name)
    create_model_run_man_file(path, name)

    invisible(NULL)
}
