\newcommand{\CRANpkg}{\href{http://CRAN.R-project.org/package=#1}{\pkg{#1}}}

\name{snssde3d}
\alias{snssde3d}
\alias{snssde3d.default}
\alias{summary.snssde3d}
\alias{print.snssde3d}
\alias{time.snssde3d}
\alias{mean.snssde3d}
\alias{median.snssde3d}
\alias{quantile.snssde3d}
\alias{kurtosis.snssde3d}
\alias{skewness.snssde3d}
\alias{moment.snssde3d}
\alias{bconfint.snssde3d}
\alias{plot.snssde3d}
\alias{points.snssde3d}
\alias{lines.snssde3d}
\alias{plot3D.snssde3d}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{Simulation of 3-Dim Stochastic Differential Equation}
\description{The (S3) generic function \code{snssde3d} of simulation of solutions to 3-dim stochastic differential equations of Ito or Stratonovich type, with different methods.
}
\usage{
snssde3d(N, \dots)
\method{snssde3d}{default}(N = 1000, M =1, x0 = 0, y0 = 0, z0 = 0, t0 = 0, T = 1, Dt, 
   driftx, diffx, drifty, diffy, driftz, diffz, alpha = 0.5, mu = 0.5, 
   type = c("ito", "str"), method = c("euler", "milstein","predcorr", 
   "smilstein", "taylor", "heun", "rk1", "rk2", "rk3"), \dots)
						   
						   
\method{summary}{snssde3d}(object, \dots)
\method{time}{snssde3d}(x, \dots)
\method{mean}{snssde3d}(x, \dots)
\method{median}{snssde3d}(x, \dots)
\method{quantile}{snssde3d}(x, \dots)
\method{kurtosis}{snssde3d}(x, \dots)
\method{skewness}{snssde3d}(x, \dots)
\method{moment}{snssde3d}(x, order = 2, \dots)
\method{bconfint}{snssde3d}(x, level=0.95, \dots)
\method{plot}{snssde3d}(x, \dots)
\method{lines}{snssde3d}(x, \dots)
\method{points}{snssde3d}(x, \dots)
\method{plot3D}{snssde3d}(x, display = c("persp","rgl"), \dots)
}
%- maybe also 'usage' for other xs documented here.
\arguments{
  \item{N}{number of simulation steps.}
  \item{M}{number of trajectories.}
  \item{x0, y0, z0}{initial value of the process \eqn{X_{t}}{X(t)}, \eqn{Y_{t}}{Y(t)} and \eqn{Z_{t}}{Z(t)} at time \code{t0}.}
  \item{t0}{initial time.}
  \item{T}{final time.} 
  \item{Dt}{time step of the simulation (discretization). If it is \code{\link[base]{missing}} a default \eqn{\Delta t = \frac{T-t_{0}}{N}}{Dt = (T-t0)/N}.}
  \item{driftx, drifty, driftz}{drift coefficient: an \code{\link[base]{expression}} of four variables \code{t}, \code{x}, \code{y} and \code{z} for process \eqn{X_t}{X(t)}, \eqn{Y_t}{Y(t)} and \eqn{Z_t}{Z(t)}.}
  \item{diffx, diffy, diffz}{diffusion coefficient: an \code{\link[base]{expression}} of four variables \code{t}, \code{x}, \code{y} and \code{z} for process \eqn{X_t}{X(t)}, \eqn{Y_t}{Y(t)} and \eqn{Z_t}{Z(t)}.}
  \item{alpha, mu }{weight of the predictor-corrector scheme; the default \code{alpha = 0.5} and \code{mu = 0.5}.}
  \item{type}{if \code{type="ito"} simulation sde of Ito type, else \code{type="str"} simulation sde of Stratonovich type; the default \code{type="ito"}.}
  \item{method}{numerical methods of simulation, the default \code{method = "euler"}.}
  \item{x, object}{an object inheriting from class \code{"snssde3d"}.}
  \item{order}{order of moment.}
  \item{level}{the confidence level required.} 
  \item{display}{\code{"persp"} perspective or \code{"rgl"} plots.} 
  \item{\dots}{further arguments for (non-default) methods.}
}
\details{
The function \code{snssde3d} returns a \code{mts} x of length \code{N+1}; i.e. solution of the 3-dim sde \eqn{(X_{t},Y_{t},Z_{t})}{(X(t),Y(t),Z(t))}
of Ito or Stratonovich types; If \code{Dt} is not specified, then the best discretization \eqn{\Delta t = \frac{T-t_{0}}{N}}{Dt = (T-t0)/N}. 

The 3-dim Ito stochastic differential equation is:
\deqn{dX(t) = a(t,X(t),Y(t),Z(t)) dt + b(t,X(t),Y(t),Z(t)) dW_{1}(t)}{dX(t) = a(t,X(t),Y(t),Z(t))*dt + b(t,X(t),Y(t),Z(t))*dW1(t)}
\deqn{dY(t) = a(t,X(t),Y(t),Z(t)) dt + b(t,X(t),Y(t),Z(t)) dW_{2}(t)}{dY(t) = a(t,X(t),Y(t),Z(t))*dt + b(t,X(t),Y(t),Z(t))*dW2(t)}
\deqn{dZ(t) = a(t,X(t),Y(t),Z(t)) dt + b(t,X(t),Y(t),Z(t)) dW_{3}(t)}{dZ(t) = a(t,X(t),Y(t),Z(t))*dt + b(t,X(t),Y(t),Z(t))*dW3(t)}
3-dim Stratonovich sde :
\deqn{dX(t) = a(t,X(t),Y(t),Z(t)) dt + b(t,X(t),Y(t),Z(t)) \circ dW_{1}(t)}{dX(t) = a(t,X(t),Y(t),Z(t))*dt + b(t,X(t),Y(t),Z(t)) o dW1(t)}
\deqn{dY(t) = a(t,X(t),Y(t),Z(t)) dt + b(t,X(t),Y(t),Z(t)) \circ dW_{2}(t)}{dY(t) = a(t,X(t),Y(t),Z(t))*dt + b(t,X(t),Y(t),Z(t)) o dW2(t)}
\deqn{dZ(t) = a(t,X(t),Y(t),Z(t)) dt + b(t,X(t),Y(t),Z(t)) \circ dW_{3}(t)}{dZ(t) = a(t,X(t),Y(t),Z(t))*dt + b(t,X(t),Y(t),Z(t)) o dW3(t)}
\eqn{W_{1}(t), W_{2}(t), W_{3}(t)}{W1(t), W2(t) and W3(t)} three standard Brownian motion independent.

The methods of approximation are classified according to their different properties. Mainly two criteria of optimality are used in the literature: the strong 
and the weak (orders of) convergence. The \code{method} of simulation can be one among: \code{Euler-Maruyama Order 0.5}, \code{Milstein Order 1}, \code{Milstein Second-Order}, 
\code{Predictor-Corrector method}, \code{Ito-Taylor Order 1.5}, \code{Heun Order 2} and \code{Runge-Kutta Order 1, 2 and 3}. 

For more details see \code{vignette("SDEs")}.
}
\value{
\code{snssde3d} returns an object inheriting from \code{\link[base]{class}} \code{"snssde3d"}.
\item{X, Y, Z }{an invisible \code{mts} (3-dim) object (X(t),Y(t),Z(t)).}
\item{driftx, drifty, driftz }{drift coefficient of X(t), Y(t) and Z(t).}
\item{diffx, diffy, diffz }{diffusion coefficient of X(t), Y(t) and Z(t).}
\item{type }{type of sde.}
\item{method }{the numerical method used.}
}
\references{
  Friedman, A. (1975). 
  \emph{Stochastic differential equations and applications}.
  Volume 1, ACADEMIC PRESS.
  
  Henderson, D. and Plaschko,P. (2006). 
  \emph{Stochastic differential equations in science and engineering}.
  World Scientific.
  
  Allen, E. (2007).
  \emph{Modeling with Ito stochastic differential equations}. 
  Springer-Verlag.

  Jedrzejewski, F. (2009). 
  \emph{Modeles aleatoires et physique probabiliste}. 
  Springer-Verlag.
  
  Iacus, S.M. (2008). 
  \emph{Simulation and inference for stochastic differential equations: with R examples}. 
  Springer-Verlag, New York.
  
  Kloeden, P.E, and Platen, E. (1989). 
  A survey of numerical methods for stochastic differential equations. 
  \emph{Stochastic Hydrology and Hydraulics}, \bold{3}, 155--178.

  Kloeden, P.E, and Platen, E. (1991a).
  Relations between multiple ito and stratonovich integrals.
  \emph{Stochastic Analysis and Applications}, \bold{9}(3), 311--321.

  Kloeden, P.E, and Platen, E. (1991b).
  Stratonovich and ito stochastic taylor expansions. 
  \emph{Mathematische Nachrichten}, \bold{151}, 33--50.

  Kloeden, P.E, and Platen, E. (1995).
  \emph{Numerical Solution of Stochastic Differential Equations}. 
  Springer-Verlag, New York.

  Oksendal, B. (2000).
  \emph{Stochastic Differential Equations: An Introduction with Applications}.
  5th edn. Springer-Verlag, Berlin.

  Platen, E. (1980).
  Weak convergence of approximations of ito integral equations. 
  \emph{Z Angew Math Mech}. \bold{60}, 609--614.

  Platen, E. and Bruti-Liberati, N. (2010).
  \emph{Numerical Solution of Stochastic Differential Equations with Jumps in Finance.} 
  Springer-Verlag, New York

  Saito, Y, and Mitsui, T. (1993).
  Simulation of Stochastic Differential Equations. 
  \emph{The Annals of the Institute of Statistical Mathematics}, \bold{3}, 419--432.
}
\author{
A.C. Guidoum, K. Boukhetala.
}
\seealso{
\code{\link{snssde1d}} and \code{\link{snssde2d}} for 1- and 2-dim sde.

\code{\link[sde]{sde.sim}} in package \CRANpkg{sde}. \code{simulate} in package "yuima". 
}
\examples{

## Example 1: Ito sde
## dX(t) = 4*(-1-X(t))*Y(t) dt + 0.2 * dW1(t) 
## dY(t) = 4*(1-Y(t)) *X(t) dt + 0.2 * dW2(t) 
## dZ(t) = 4*(1-Z(t)) *Y(t) dt + 0.2 * dW3(t)        
## W1(t), W2(t) and W3(t) three independent Brownian motion
set.seed(1234)

fx <- expression(4*(-1-x)*y)
gx <- expression(0.2)
fy <- expression(4*(1-y)*x)
gy <- expression(0.2)
fz <- expression(4*(1-z)*y)
gz <- expression(0.2)

res <- snssde3d(x0=2,y0=-2,z0=-2,driftx=fx,diffx=gx,drifty=fy,diffy=gy,
                driftz=fz,diffz=gz,N=1000,M=50)
res
summary(res)
dev.new()
plot(res,pos=2)
dev.new()
plot(res,union = FALSE)
dev.new()
plot3D(res,display="persp") ## in space (O,X,Y,Z)

## Example 2: Stratonovich sde
## dX(t) = Y(t)* dt           
## dY(t) = (4*( 1-X(t)^2 )* Y(t) - X(t))* dt + 0.2 o dW2(t)
## dZ(t) = (4*( 1-X(t)^2 )* Z(t) - X(t))* dt + 0.2 o dW3(t)
set.seed(1234)

fx <- expression( y )
gx <- expression( 0 )
fy <- expression( (4*( 1-x^2 )* y - x) )
gy <- expression( 0.2)
fz <- expression( (4*( 1-x^2 )* z - x) )
gz <- expression( 0.2)

res <- snssde3d(driftx=fx,diffx=gx,drifty=fy,diffy=gy,driftz=fz,diffz=gz,
                ,N=5000,T=50,type="str")
res
dev.new()
plot(res,pos=2)
dev.new()
plot(res,union = FALSE)
dev.new()
plot3D(res,display="persp") ## in space (O,X,Y,Z)
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{sde}
\keyword{ts}
\keyword{mts}
