library(SentimentAnalysis)
context("Testing sentiment dictionaries to store polarity words")

test_that("wordlist works correctly", {
  wl <- c("uncertain", "unsure")
  d <- SentimentDictionary(wl)
  expect_is(d, "SentimentDictionaryWordlist")
  expect_equivalent(d$wordlist, wl)
  
  expect_equivalent(numEntries(d), 2)
  expect_error(numPositiveEntries(d), "Type not supported")
  expect_error(numNegativeEntries(d), "Type not supported")
  
  expect_output(summary(d), "word list")
  expect_output(print(d), "word list")
  expect_output(print(d),  wl[1])
  
  write(d, "test.dict")
  d <- read("test.dict")
  unlink("test.dict")
  expect_is(d, "SentimentDictionaryWordlist")
  expect_equivalent(d$wordlist, wl)
})

test_that("binary dictionary works correctly", {
  pos <- c("increase", "rise")
  neg <- c("fall", "drop", "short")
  d <- SentimentDictionary(pos, neg)
  expect_is(d, "SentimentDictionaryBinary")
  expect_equivalent(d$positive, pos)
  expect_equivalent(d$negative, neg)
  
  expect_equivalent(numEntries(d), 5)
  expect_equivalent(numPositiveEntries(d), 2)
  expect_equivalent(numNegativeEntries(d), 3)
  
  expect_output(summary(d), "binary")
  expect_output(summary(d), "Positive entries")
  expect_output(summary(d), "40%")
  expect_output(print(d), "binary")
  expect_output(print(d),  paste0("+ ", pos[1]))
  expect_output(print(d),  paste0("- ", neg[1]))
  
  write(d, "test.dict")
  d <- read("test.dict")
  unlink("test.dict")
  expect_is(d, "SentimentDictionaryBinary")
  expect_equivalent(d$positive, pos)
  expect_equivalent(d$negative, neg)
})

test_that("weighted dictionary works correctly", {
  words <- c("increase", "rise", "neutral", "fall", "drop", "short")
  scores <- c(+2, +1, 0, -1, -2, -0.5)
  idf <- 1:6
  d <- SentimentDictionary(words, scores, idf)
  expect_is(d, "SentimentDictionaryWeighted")
  expect_equivalent(d$words, words)
  expect_equivalent(d$scores, scores)
  expect_equivalent(d$intercept, 0)
  expect_equivalent(d$idf, idf)  
  
  expect_equivalent(numEntries(d), 6)
  expect_equivalent(numPositiveEntries(d), 2)
  expect_equivalent(numNegativeEntries(d), 3)
  
  expect_output(summary(d), "weighted")
  expect_output(summary(d), "Average score:")
  expect_output(summary(d), "50%")
  expect_output(summary(d), "Standard deviation: 1.42")
  expect_output(print(d), "weighted")
  expect_output(print(d),  "-2.00 drop")
  expect_output(print(d),  " 0.00 neutral")
  
  write(d, "test.dict")
  d <- read("test.dict")
  unlink("test.dict")
  expect_is(d, "SentimentDictionaryWeighted")
  expect_equivalent(d$words, words)
  expect_equivalent(d$scores, scores)
  expect_equivalent(d$intercept, 0)
  expect_equivalent(d$idf, idf)  
  
  words <- c("increase", "rise", "neutral", "fall", "drop", "short")
  scores <- c(+2, +1, 0, -1, -2, -0.5)
  idf <- 1:6
  d <- SentimentDictionary(words, scores, idf, intercept=3)
  expect_is(d, "SentimentDictionaryWeighted")
  expect_equivalent(d$words, words)
  expect_equivalent(d$scores, scores)
  expect_equivalent(d$intercept, 3)
  expect_equivalent(d$idf, idf)  
  
  expect_output(print(d), "Intercept: 3")
  
  write(d, "test.dict")
  d <- read("test.dict")
  unlink("test.dict")
  expect_is(d, "SentimentDictionaryWeighted")
  expect_equivalent(d$words, words)
  expect_equivalent(d$scores, scores)
  expect_equivalent(d$intercept, 3)
  expect_equivalent(d$idf, idf)  
  
  expect_error(SentimentDictionary(letters, 1:25, 1:26))
  expect_error(SentimentDictionary(letters, 1:26, 1:25))
})

test_that("wordlist works correctly", {
  d1 <- SentimentDictionary(c("a", "a", "b", "c", "d"))
  d2 <- SentimentDictionary(c("d", "e"))
  cmp <- compareDictionaries(d1, d2)
  expect_equivalent(cmp$totalUniqueWords, 5)
  expect_equivalent(cmp$totalSameWords, 1)
  expect_equivalent(cmp$ratioSameWords, 0.2)
  
  d1 <- SentimentDictionary(c("a", "a", "b", "c", "d", "m"), c("x", "y", "z"))
  d2 <- SentimentDictionary(c("d", "e"), c("m", "w", "x"))
  cmp <- compareDictionaries(d1, d2)
  expect_equivalent(cmp$totalUniqueWords, 10)
  expect_equivalent(cmp$totalSameWords, 3)
  expect_equivalent(cmp$ratioSameWords, 0.3)
  expect_equivalent(cmp$numWordsEqualClass, 2)
  expect_equivalent(cmp$ratioWordsEqualClass, 0.2)
  expect_equivalent(cmp$numWordsDifferentClass, 1)
  expect_equivalent(cmp$ratioWordsDifferentClass, 0.1)
  
  d1 <- SentimentDictionary(c("a", "b", "c", "d", "m", "x", "y", "z"), c(1, 1, 1, 1, 1, -1, -1, -1), rep(NA, 8))
  d2 <- SentimentDictionary(c("d", "e", "m", "w", "x"), c(1, 0, -1, -1, -1), rep(NA, 5))
  cmp <- compareDictionaries(d1, d2)
  expect_equivalent(cmp$totalUniqueWords, 10)
  expect_equivalent(cmp$totalSameWords, 3)
  expect_equivalent(cmp$ratioSameWords, 0.3)
  expect_equivalent(cmp$numWordsEqualClass, 2)
  expect_equivalent(cmp$ratioWordsEqualClass, 0.2)
  expect_equivalent(cmp$numWordsDifferentClass, 1)
  expect_equivalent(cmp$ratioWordsDifferentClass, 0.1)
  expect_equivalent(cmp$correlation, 0.5)
})

