% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sensitivity.R
\name{sensitivity}
\alias{sensitivity}
\title{Sensitivities of a Stressed Model}
\usage{
sensitivity(object, xCol = "all", wCol = "all", type = c("Gamma",
  "Kolmogorov", "Wasserstein", "all"), f = NULL, k = NULL)
}
\arguments{
\item{object}{A \code{SWIM} object.}

\item{xCol}{Numeric or character vector, (names of) the columns 
of the underlying data of the \code{object} 
(\code{default = "all"}). If \code{xCol = NULL}, only 
the transformed data \code{f(x)} is considered.}

\item{wCol}{Vector, the columns of the scenario weights 
of the \code{object} corresponding to different 
stresses (\code{default = "all"}).}

\item{type}{Character, one of \code{"Gamma", "Kolmogorov", 
"Wasserstein", "all"}.}

\item{f}{A function, or list of functions, that, applied to 
\code{x}, constitute the transformation of the data 
for which the sensitivity is calculated.}

\item{k}{A vector or list of vectors, same length as \code{f},
indicating which columns of \code{x} each function 
in \code{f} operates on.\cr
When \code{f} is a list, \code{k[[i]]} corresponds 
to the input variables of \code{f[[i]]}.}
}
\value{
A data.frame containing the sensitivity measures of the 
    stressed model with rows corresponding to different random   
    variables. The first two rows specify the \code{stress} and
    \code{type} of the sensitivity measure.
}
\description{
Provides different sensitivity measures that compare the stressed 
    and the baseline model.
}
\details{
Provides sensitivity measures that compare the stressed and 
    the baseline model. Implemented sensitivity measures:
    \enumerate{
    \item 
      \code{Gamma}, the \emph{Reverse Sensitivity Measure}, defined 
      for a random variable \code{Y} and scenario weights \code{w} by 
      \deqn{Gamma = ( E(Y * w) - E(Y) ) / c,}
      where \code{c} is a normalisation constant such that 
      \code{|Gamma| <= 1}, see
      \insertCite{Pesenti2019reverse}{SWIM}. Loosely speaking, the 
      Reverse Sensitivity Measure is the normalised difference 
      between the first moment of the stressed and the baseline 
      distributions of \code{Y}. 
    
    \item
      \code{Kolmogorov}, the Kolmogorov distance, defined for 
      distribution functions \code{F,G} by 
      \deqn{Kolmogorov = sup |F(x) - G(x)|.}
      Note that the Kolmogorov distance of one stress is the same for 
      all inputs. Should be used to compare different stresses not 
      individual components.   
    
    \item
      \code{Wasserstein}, the Wasserstein distance of order 1, defined
      for two distribution functions \code{F,G} by 
      \deqn{Wasserstein = \int | F(x) - G(x)| dx.} 
    }
    
    If \code{f} and \code{k} are provided, the sensitivity of the 
    transformed data is returned.
}
\examples{
     
## example with a stress on VaR
set.seed(0)
x <- as.data.frame(cbind(
  "log-normal" = rlnorm(1000), 
  "gamma" = rgamma(1000, shape = 2)))
res1 <- stress(type = "VaR", x = x, 
  alpha = c(0.9, 0.95), q_ratio = 1.05)
  
sensitivity(res1, wCol = 1, type = "all") 
## sensitivity of log-transformed data 
sensitivity(res1, wCol = 1, type = "all", 
  f = list(function(x)log(x), function(x)log(x)), k = list(1,2)) 
  
## Consider the portfolio Y = X1 + X2 + X3 + X4 + X5,  
## where (X1, X2, X3, X4, X5) are correlated normally 
## distributed with equal mean and different standard deviations,
## see the README for further details.

\donttest{
set.seed(0)
SD <- c(70, 45, 50, 60, 75)
Corr <- matrix(rep(0.5, 5^2), nrow = 5) + diag(rep(1 - 0.5, 5))
if (!requireNamespace("mvtnorm", quietly = TRUE))
   stop("Package \\"mvtnorm\\" needed for this function 
   to work. Please install it.")
x <- mvtnorm::rmvnorm(10^5, 
   mean =  rep(100, 5), 
   sigma = (SD \%*\% t(SD)) * Corr) 
data <- data.frame(rowSums(x), x)
names(data) <- c("Y", "X1", "X2", "X3", "X4", "X5")
rev.stress <- stress(type = "VaR", x = data, 
   alpha = c(0.75, 0.9), q_ratio = 1.1, k = 1)

sensitivity(rev.stress, type = "all") 
## sensitivity to sub-portfolios X1 + X2 and X3 + X4
sensitivity(rev.stress, xCol = NULL, type = "Gamma", 
  f = rep(list(function(x)x[1] + x[2]), 2), k = list(c(2,3), c(4,5))) 
plot_sensitivity(rev.stress, xCol = 2:6, type = "Gamma")     
importance_rank(rev.stress, xCol = 2:6, type = "Gamma")
}
     
}
\references{
\insertRef{Pesenti2019reverse}{SWIM}
}
\seealso{
See \code{\link{importance_rank}} for ranking of random
    variables according to their sensitivities,  
    \code{\link{plot_sensitivity}} for plotting 
    sensitivity measures and \code{\link{summary}} for 
    summary statistics of a stressed model.
}
