% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/svem_random_table.R
\name{svem_random_table_from_model}
\alias{svem_random_table_from_model}
\title{Generate a Random Prediction Table from a Fitted SVEMnet Model (no refit)}
\usage{
svem_random_table_from_model(
  object,
  n = 1000,
  mixture_groups = NULL,
  debias = FALSE
)
}
\arguments{
\item{object}{A fitted \code{svem_model} returned by \code{SVEMnet()}.
The object must contain \code{$sampling_schema}, which is
created by the updated \code{SVEMnet()}.}

\item{n}{Number of random points to generate (default \code{1000}).}

\item{mixture_groups}{Optional list of mixture factor groups. Each element is
a list with components:
\itemize{
\item \code{vars}: character vector of mixture variable names (must be in the model).
\item \code{lower}: numeric vector of lower bounds (same length as \code{vars}).
\item \code{upper}: numeric vector of upper bounds (same length as \code{vars}).
\item \code{total}: scalar specifying the group total (e.g., 1.0).
}
If omitted, all variables are sampled independently using the cached schema.}

\item{debias}{Logical; if \code{TRUE}, applies the calibration from
\code{object$debias_fit} (if available) to the mean prediction (default \code{FALSE}).}
}
\value{
A data frame with sampled predictors and a predicted response column.
The response column name matches the left-hand side of \code{object$formula}.
}
\description{
Samples the original predictor factor space using ranges and levels cached
inside a fitted \code{svem_model} (from \code{SVEMnet()}) and computes
predictions at those points. Optional mixture groups let you sample
composition variables on a (possibly truncated) simplex via a Dirichlet draw.
No refitting is performed.
}
\details{
This function uses:
\itemize{
\item \code{object$sampling_schema$num_ranges} for uniform sampling of numeric variables.
\item \code{object$sampling_schema$factor_levels} for categorical sampling.
\item \code{object$terms}, \code{object$xlevels}, and \code{object$contrasts}
(via \code{predict.svem_model}) to encode the model matrix consistently.
}
Mixture groups are handled by drawing Dirichlet weights and mapping them to the
truncated simplex defined by \code{lower}, \code{upper}, and \code{total}; proposals
violating upper bounds are rejected (with oversampling to keep it efficient).
}
\examples{
\donttest{
set.seed(1)
n <- 40
X1 <- runif(n); X2 <- runif(n)
A <- runif(n); B <- runif(n); C <- pmax(0, 1 - A - B)  # simple mixture-ish
F <- factor(sample(c("lo","hi"), n, TRUE))
y <- 1 + 2*X1 - X2 + 3*A + 1.5*B + 0.5*C + (F=="hi") + rnorm(n, 0, 0.3)
d <- data.frame(y, X1, X2, A, B, C, F)

fit <- SVEMnet(y ~ X1 + X2 + A + B + C + F, d, nBoot = 50, glmnet_alpha = 1)

# No mixture constraints (independent sampling using cached ranges/levels)
tbl1 <- svem_random_table_from_model(fit, n = 50)
head(tbl1)

# With mixture constraints for A,B,C that sum to 1 and have bounds
mix <- list(list(vars = c("A","B","C"),
                 lower = c(0.1, 0.1, 0.1),
                 upper = c(0.7, 0.7, 0.7),
                 total = 1.0))
tbl2 <- svem_random_table_from_model(fit, n = 50, mixture_groups = mix)
head(tbl2)
}

}
\seealso{
\code{\link{SVEMnet}}, \code{\link{predict.svem_model}},
\code{\link{svem_significance_test}}, \code{\link{svem_significance_test_parallel}}
}
