% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/point2poly_tess.R
\name{point2poly_tess}
\alias{point2poly_tess}
\title{Point-to-polygon interpolation, tessellation method}
\usage{
point2poly_tess(
  pointz,
  polyz,
  poly_id,
  char_methodz = "aw",
  methodz = "aw",
  pop_raster = NULL,
  varz = NULL,
  pycno_varz = NULL,
  char_varz = NULL,
  char_assign = "biggest_overlap",
  funz = function(x, w) {
     stats::weighted.mean(x, w, na.rm = TRUE)
 },
  return_tess = FALSE,
  seed = 1
)
}
\arguments{
\item{pointz}{Source points layer. \code{sf} object.}

\item{polyz}{Destination polygon layer. Must have identical CRS to \code{pointz}. \code{sf} object.}

\item{poly_id}{Name of unique ID column for destination polygon layer. Character string.}

\item{char_methodz}{Interpolation method(s) for character strings. Could be either of "aw" (areal weighting, default) or "pw" (population weighting). See "details". Character string.}

\item{methodz}{Interpolation method(s) for numeric covariates. Could be either of "aw" (areal weighting, default) and/or "pw" (population weighting). See "details". Character string or vector of character strings.}

\item{pop_raster}{Population raster to be used for population weighting, Must be supplied if \code{methodz="pw"}. Must have identical CRS to \code{poly_from}. \code{raster} or \code{SpatRaster} object.}

\item{varz}{Names of numeric variable(s) to be interpolated from source polygon layer to destination polygons. Character string or list of character strings.}

\item{pycno_varz}{Names of spatially extensive numeric variables for which the pycnophylactic (mass-preserving) property should be preserved. Character string or vector of character strings.}

\item{char_varz}{Names of character string variables to be interpolated from source polygon layer to destination polygons. Character string or vector of character strings.}

\item{char_assign}{Assignment rule to be used for variables specified in \code{char_varz}. Could be either "biggest_overlap" (default) or "all_overlap". See "details". Character string or vector of character strings.}

\item{funz}{Aggregation function to be applied to variables specified in \code{varz}. Must take as an input a numeric vector \code{x} and vector of weights \code{w}. Function or list of functions.}

\item{return_tess}{Return Voronoi polygons, in addition to destinaton polygon layer? Default is \code{FALSE}. Logical.}

\item{seed}{Seed for generation of random numbers. Default is 1. Numeric.}
}
\value{
If \code{return_tess=FALSE}, returns a \code{sf} polygon object, with variables from \code{pointz} interpolated to the geometries of \code{polyz}.

If \code{return_tess=TRUE}, returns a list, containing
\itemize{
 \item{"result". }{The destination polygon layer. \code{sf} object.}
 \item{"tess". }{The (intermediate) Voronoi tessellation polygon layer. \code{sf} object.}
 }
}
\description{
Function for interpolating values from a source point layer to a destination polygon layer, using Voronoi tessellation and area/population weights.
}
\details{
This function interpolates point data to polygons with a two-step process. In the first step (tessellation), each point is assigned a Voronoi cell, drawn such that (a) the distance from its borders to the focal point is less than or equal to the distance to any other point, and (b) no gaps between cells remain. The second step (interpolation) performs a polygon-in-polygon interpolation, using the Voronoi cells as source polygons.

Currently supported integration methods in the second step (\code{methodz}) include:
\itemize{
 \item{Areal weighting ("aw"). }{Values from \code{poly_from} weighted in proportion to relative area of spatial overlap between source features and geometries of \code{poly_to}.}
 \item{Population weighting ("pw"). }{Values from \code{poly_from} weighted in proportion to relative population sizes in areas of spatial overlap between source features and geometries of \code{poly_to}. This routine uses a third layer (supplied in \code{pop_raster}) to calculate the weights.}
}
When a list of variables are supplied and one methods argument specified, then the chosen method will be applied to all variables.

When a list of of variables are supplied and multiple methods arguments specified, then weighting methods will be applied in a pairwise order. For example, specifying \code{varz = list(c("to1","pvs1_margin"), c("vv1"))} and \code{methodz = c('aw', 'pw')} will apply areal weighting to the the first set of variables (to1 and pvs1_margin) and population weighing to the second set (vv1).

Interpolation procedures are handled somewhat differently for numeric and character string variables. For numeric variables supplied in \code{varz}, "aw" and/or "pw" weights are passed to the function specified in \code{funz}. If different sets of numeric variables are to be aggregated with different functions, both \code{varz} and \code{funz} should be specified as lists (see examples below).

For character string (and any other) variables supplied in \code{char_varz}, "aw" and/or "pw" weights are passed to the assignment rule(s) specified in \code{char_assign}. Note that the \code{char_varz} argument may include numerical variables, but \code{varz} cannot include character string variables.

Currently supported assignment rules for character strings (\code{char_assign}) include:
\itemize{
 \item{"biggest_overlap". }{For each variable in \code{char_varz}, the features in \code{poly_to} are assigned a single value from overlapping \code{poly_from} features, corresponding to the intersection with largest area and/or population weight.}
 \item{"all_overlap". }{For each variable in \code{char_varz}, the features in \code{poly_to} are assigned all values from overlapping \code{poly_from} features, ranked by area and/or population weights (largest-to-smallest) of intersections.}
}
It is possible to pass multiple arguments to \code{char_assign} (e.g. \code{char_assign=c("biggest_overlap","all_overlap")}), in which case the function will calculate both, and append the resulting columns to the output.
}
\examples{
# Interpolation of a single variable, with area weights
\dontrun{
data(hex_05_deu)
data(clea_deu2009_pt)
out_1 <- point2poly_tess(pointz = clea_deu2009_pt,
                             polyz = hex_05_deu,
                             poly_id = "HEX_ID",
                             varz = "to1")
plot(out_1["to1_aw"])
}

# Extract and inspect tessellation polygons
\dontrun{
out_2 <- point2poly_tess(pointz = clea_deu2009_pt,
                             polyz = hex_05_deu,
                             poly_id = "HEX_ID",
                             varz = "to1",
                             return_tess = TRUE)
plot(out_2$tess["to1"])
plot(out_2$result["to1_aw"])
}

# Interpolation of multiple variables, with area and population weights
\dontrun{
data(gpw4_deu2010)
out_3 <- point2poly_tess(pointz = clea_deu2009_pt,
                         polyz = hex_05_deu,
                         poly_id = "HEX_ID",
                         methodz = c("aw","pw"),
                         varz = list(
                           c("to1","pvs1_margin"),
                           c("vv1")
                         ),
                         pycno_varz = "vv1",
                         funz = list(
                           function(x,w){stats::weighted.mean(x,w)},
                           function(x,w){sum(x*w)}
                           ),
                         char_varz = c("incumb_pty_n","win1_pty_n"),
                         pop_raster = gpw4_deu2010)
plot(out_3["vv1_pw"])
}
}
