% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stopt_SA.R
\name{stopt.SA}
\alias{stopt.SA}
\title{Optimization over Stiefel Manifold with Simulated Annealing}
\usage{
stopt.SA(
  func,
  size,
  n.start = 10,
  stepsize = 0.1,
  maxiter = 100,
  cooling = c("exponential", 10, 0.9),
  init.val = NULL,
  print.progress = FALSE
)
}
\arguments{
\item{func}{a function to be \emph{minimized}.}

\item{size}{a length-2 vector containing dimension information \eqn{(p,r)}.}

\item{n.start}{number of runs, i.e., algorithm is executed \code{n.start} times.}

\item{stepsize}{size of random walk on each component.}

\item{maxiter}{maximum number of iterations for each run.}

\item{cooling}{triplet for cooling schedule. See the section for the usage.}

\item{init.val}{if \code{NULL}, starts from a random point. Otherwise, a Stiefel matrix of size \eqn{(p,r)} should be provided for fixed starting point.}

\item{print.progress}{a logical; \code{TRUE} to show}
}
\value{
a named list containing: \describe{
\item{cost}{minimized function value.}
\item{solution}{a \eqn{(p\times r)} matrix that attains the \code{cost}.}
\item{accfreq}{frequency of acceptance moves.}
}
}
\description{
Simulated Annealing is a black-box, derivative-free optimization algorithm 
that iterates via stochastic search in the neighborhood of current position.
}
\examples{
## Optimization for eigen-decomposition
#  Let's find top-3 eigenvalues 
set.seed(121)                         # set seed
A = cov(matrix(rnorm(100*5), ncol=5)) # define covariance
myfunc <- function(p){                # cost function
  return(sum(-diag(t(p)\%*\%A\%*\%p)))
} 

#  Solve the optimization problem
Aout = stopt.SA(myfunc, size=c(5,3), n.start=40, maxiter=500)

#  Compute 3 Eigenvalues
#  1. use computed basis
abase   = Aout$solution
eig3sol = sort(diag(t(abase)\%*\%A\%*\%abase), decreasing=TRUE)

#  2. use 'eigen' function
eig3dec = sort(eigen(A)$values, decreasing=TRUE)[1:3]

\donttest{
#   Visualize
opar <- par(no.readonly=TRUE)
yran = c(min(min(eig3sol),min(eig3dec))*0.95,
         max(max(eig3sol),max(eig3dec))*1.05)
plot(1:3, eig3sol, type="b", col="red",  pch=19, ylim=yran,
     xlab="index", ylab="eigenvalue", main="compare top 3 eigenvalues")
lines(1:3, eig3dec, type="b", col="blue", pch=19)
legend(1, 1, legend=c("optimization","decomposition"), col=c("red","blue"),
       lty=rep(1,2), pch=19)
par(opar)
}

}
