/*
 * This file is part of ReporteRs
 * Copyright (c) 2014, David Gohel All rights reserved.
 * This program is licensed under the GNU GENERAL PUBLIC LICENSE V3.
 * You may obtain a copy of the License at :
 * http://www.gnu.org/licenses/gpl.html
 */

package org.lysis.reporters.formats;

import java.math.BigInteger;
import java.util.LinkedHashMap;

import org.docx4j.dml.CTTextNoBullet;
import org.docx4j.dml.CTTextParagraphProperties;
import org.docx4j.dml.CTTextSpacing;
import org.docx4j.dml.CTTextSpacingPoint;
import org.docx4j.dml.STTextAlignType;
import org.docx4j.wml.CTShd;
import org.docx4j.wml.Jc;
import org.docx4j.wml.JcEnumeration;
import org.docx4j.wml.PPr;
import org.docx4j.wml.PPrBase;
import org.docx4j.wml.PPrBase.NumPr;
import org.docx4j.wml.PPrBase.NumPr.Ilvl;
import org.docx4j.wml.PPrBase.NumPr.NumId;
import org.docx4j.wml.PPrBase.PBdr;
import org.lysis.reporters.lists.LevelDefinition;
import org.lysis.reporters.lists.NumberingDefinition;

public class ParProperties {
	private String textalign;
	private int paddingbottom;
	private int paddingtop;
	private int paddingleft;
	private int paddingright;

	private int marginleft;
	private int marginright;
	private int margintop;
	private int marginbottom;

	private String liststyle;
	private int level;

	private BorderProperties borderBottom;
	private BorderProperties borderLeft;
	private BorderProperties borderTop;
	private BorderProperties borderRight;
	private String shading_color;

	private long ordered_num_id = -1;
	private long unordered_num_id = -1;
	


	public ParProperties(String textalign, int paddingbottom, int paddingtop,
			int paddingleft, int paddingright, 
			int marginbottom, int margintop,
			int marginleft, int marginright, 
			String liststyle, int level,
			BorderProperties borderBottom, BorderProperties borderLeft, 
			BorderProperties borderTop, BorderProperties borderRight
			) {
		super();
		this.borderBottom = borderBottom;
		this.borderLeft = borderLeft;
		this.borderTop = borderTop;
		this.borderRight = borderRight;
		this.textalign = textalign;
		this.paddingbottom = paddingbottom;
		this.paddingtop = paddingtop;
		this.paddingleft = paddingleft;
		this.paddingright = paddingright;
		this.liststyle = liststyle;
		this.level = level;
		this.marginleft = marginleft;
		this.marginright = marginright;
		this.margintop = margintop;
		this.marginbottom = marginbottom;
	}

	
	public void setOrderedNumid(long id){
		ordered_num_id = id;
	}
	
	public void setUnorderedNumid(long id){
		unordered_num_id = id;
	}
	
	public String getTextalign() {
		return textalign;
	}

	public String getListStyle() {
		return liststyle;
	}
	public int getLevel() {
		return level;
	}
	public int getPaddingbottom() {
		return paddingbottom;
	}

	public int getPaddingtop() {
		return paddingtop;
	}

	public int getPaddingleft() {
		return paddingleft;
	}

	public int getPaddingright() {
		return paddingright;
	}
	public void setShadingColor(String shadingColor){
		shading_color = shadingColor;
	}
	public void setPaddingLeft(int padding ){
		paddingleft = padding;
	}
	
	public void setMarginLeft(int margin ){
		marginleft = margin;
	}
	public void setMarginRight(int margin ){
		marginright = margin;
	}
	public void setMarginTop(int margin ){
		margintop = margin;
	}
	public void setMarginBottom(int margin ){
		marginbottom = margin;
	}

	public String getShadingColor() {
		return shading_color;
	}

	public boolean hasShadingColor() {
		return shading_color != null;
	}

	public ParProperties getClone(){
		ParProperties parProp = new ParProperties( textalign,
				paddingbottom, paddingtop,
				paddingleft, paddingright, 
				marginbottom,margintop,marginleft,marginright,
				liststyle, level, 
				borderBottom,
				borderLeft, 
				borderTop,
				borderRight);
		if( shading_color != null) parProp.setShadingColor(this.shading_color);
		parProp.setOrderedNumid(ordered_num_id);
		parProp.setUnorderedNumid(unordered_num_id);
		
		return( parProp );
	}
	
	public PPr get_docx_pp(){
		PPr parProperties = new PPr();
		
		Jc alignment = new Jc();
		if( textalign.equals("left")) alignment.setVal(JcEnumeration.LEFT);
		else if( textalign.equals("center")) alignment.setVal(JcEnumeration.CENTER);
		else if( textalign.equals("right")) alignment.setVal(JcEnumeration.RIGHT);
		else if( textalign.equals("justify")) alignment.setVal(JcEnumeration.BOTH);
        parProperties.setJc(alignment); 
        PPrBase.Spacing space = new PPrBase.Spacing();
        space.setBefore(BigInteger.valueOf((long)paddingtop*20));
        space.setAfter(BigInteger.valueOf((long)paddingbottom*20));
        parProperties.setSpacing(space);
        
		if( shading_color!=null){
			CTShd shading = new CTShd();
			shading.setFill(shading_color);
			parProperties.setShd(shading);
		}
		
        PBdr parBorder = new PBdr();
        if( borderLeft.hasBorder() )
        	parBorder.setLeft(borderLeft.get_docx_bd());
        if( borderBottom.hasBorder() )
	        parBorder.setBottom(borderBottom.get_docx_bd());
        if( borderRight.hasBorder() )
	        parBorder.setRight(borderRight.get_docx_bd());
        if( borderTop.hasBorder() )
        	parBorder.setTop(borderTop.get_docx_bd());
		parProperties.setPBdr(parBorder );

        if( ( getListStyle().equals("ordered") || 
        			getListStyle().equals("unordered") ) && 
        		ordered_num_id > 0 ){
        	
        	Ilvl ilvl = new Ilvl();
        	ilvl.setVal( BigInteger.valueOf( getLevel()-1 ) );
        	NumId numid = new NumId();
        	if( getListStyle().equals("ordered") ){
	    		numid.setVal( BigInteger.valueOf( ordered_num_id ) );   		
        	} else if( getListStyle().equals("unordered") ){
	    		numid.setVal( BigInteger.valueOf( unordered_num_id ) );
        	}
        	NumPr numPr =  new NumPr();
    		numPr.setIlvl(ilvl);
    		numPr.setNumId(numid);
    		parProperties.setNumPr(numPr);
        } else if( getListStyle().equals("blockquote") ){
            PPrBase.Ind padding = new PPrBase.Ind();
            padding.setLeft(BigInteger.valueOf( (long)paddingleft * 20 * (getLevel()+1) ));
            padding.setRight(BigInteger.valueOf( (long)paddingright * 20 * (getLevel()+1) ));
            parProperties.setInd(padding);
       } else {
            PPrBase.Ind padding = new PPrBase.Ind();
            padding.setLeft(BigInteger.valueOf((long)paddingleft*20 ));
            padding.setRight(BigInteger.valueOf((long)paddingright*20 ));
            parProperties.setInd(padding);
        }

        return parProperties;
	}
	public CTTextParagraphProperties get_pptx_pp(){
		CTTextParagraphProperties parProperties = new CTTextParagraphProperties();
		
		if( textalign.equals("left")) parProperties.setAlgn(STTextAlignType.L);
		else if( textalign.equals("center")) parProperties.setAlgn(STTextAlignType.CTR);
		else if( textalign.equals("right")) parProperties.setAlgn(STTextAlignType.R);
		else if( textalign.equals("justify")) parProperties.setAlgn(STTextAlignType.JUST);

		CTTextSpacing spaceafter = new CTTextSpacing();
		CTTextSpacingPoint pointafter = new CTTextSpacingPoint();
		pointafter.setVal(paddingbottom*100);
		spaceafter.setSpcPts(pointafter);
		parProperties.setSpcAft(spaceafter);
		
		CTTextSpacing spacebefore = new CTTextSpacing();
		CTTextSpacingPoint pointbefore = new CTTextSpacingPoint();
		pointbefore.setVal(paddingtop*100);
		spacebefore.setSpcPts(pointbefore);
		parProperties.setSpcBef(spacebefore);
	
		parProperties.setMarR(paddingright * 12700);
		parProperties.setMarL(level*paddingleft*12700);
    	parProperties.setIndent(0);
		parProperties.setBuNone(new CTTextNoBullet());

        return parProperties;
	}

	public CTTextParagraphProperties get_pptx_pp(NumberingDefinition nb, int startNumbering){
		
		CTTextParagraphProperties parProperties = new CTTextParagraphProperties();
		
		if( textalign.equals("left")) parProperties.setAlgn(STTextAlignType.L);
		else if( textalign.equals("center")) parProperties.setAlgn(STTextAlignType.CTR);
		else if( textalign.equals("right")) parProperties.setAlgn(STTextAlignType.R);
		else if( textalign.equals("justify")) parProperties.setAlgn(STTextAlignType.JUST);

		CTTextSpacing spaceafter = new CTTextSpacing();
		CTTextSpacingPoint pointafter = new CTTextSpacingPoint();
		pointafter.setVal(paddingbottom*100);
		spaceafter.setSpcPts(pointafter);
		parProperties.setSpcAft(spaceafter);
		
		CTTextSpacing spacebefore = new CTTextSpacing();
		CTTextSpacingPoint pointbefore = new CTTextSpacingPoint();
		pointbefore.setVal(paddingtop*100);
		spacebefore.setSpcPts(pointbefore);
		parProperties.setSpcBef(spacebefore);
		parProperties.setMarR(paddingright * 12700);

		if( getListStyle().equals("ordered") || getListStyle().equals("unordered") ){
			LevelDefinition ld ;
			if( getListStyle().equals("ordered") )
				ld = nb.getOrdered().getLevelDefintion(level);
			else ld = nb.getUnordered().getLevelDefintion(level);
			ld.set_pptx_scheme(parProperties, startNumbering);
		} else {
    		parProperties.setMarL(level*paddingleft*12700);
        	parProperties.setIndent(0);
    		parProperties.setBuNone(new CTTextNoBullet());
        }

        return parProperties;

	}


	public LinkedHashMap<String, String> get_html_pp() {
		LinkedHashMap<String, String> parProperties = new LinkedHashMap<String, String>();

		if( textalign.equals("left")) parProperties.put("text-align", "left");	
		else if( textalign.equals("center")) parProperties.put("text-align", "center");	
		else if( textalign.equals("right")) parProperties.put("text-align", "right");	
		else if( textalign.equals("justify")) parProperties.put("text-align", "justify");	
		else parProperties.put("text-align", "left");
		
		parProperties.put("padding-top", paddingtop+"pt");	
		parProperties.put("padding-bottom", paddingbottom+"pt");	
		parProperties.put("padding-right", paddingright+"pt");	
		parProperties.put("padding-left", paddingleft+"pt");	
		
		if( shading_color!=null){
			parProperties.put("background-color", shading_color);
		}
		
		if( marginleft > 0 ){
			parProperties.put("margin-left", marginleft + "pt");	
		}
		if( marginright > 0 ){
			parProperties.put("margin-right", marginright + "pt");	
		}
		if( margintop > 0 ){
			parProperties.put("margin-top", margintop + "pt");	
		}
		if( marginbottom > 0 ){
			parProperties.put("margin-bottom", marginbottom + "pt");	
		}
		
		if( borderBottom.hasBorder() ){
			parProperties.put("border-bottom-color", borderBottom.getBorderColor() );	
			parProperties.put("border-bottom-style", borderBottom.getBorderStyle() );	
			parProperties.put("border-bottom-width", borderBottom.getBorderWidth() + "px" );
		}
		if( borderTop.hasBorder() ){
			parProperties.put("border-top-color", borderTop.getBorderColor() );	
			parProperties.put("border-top-style", borderTop.getBorderStyle() );	
			parProperties.put("border-top-width", borderTop.getBorderWidth() + "px" );	
		}
		if( borderRight.hasBorder() ){
			parProperties.put("border-right-color", borderRight.getBorderColor() );	
			parProperties.put("border-right-style", borderRight.getBorderStyle() );	
			parProperties.put("border-right-width", borderRight.getBorderWidth() + "px" );
		}
		if( borderLeft.hasBorder() ){
			parProperties.put("border-left-color", borderLeft.getBorderColor() );	
			parProperties.put("border-left-style", borderLeft.getBorderStyle() );	
			parProperties.put("border-left-width", borderLeft.getBorderWidth() + "px" );	
		}
        return parProperties;

	}
	
}
