##
## Function part of Rcssplot package
##
## Parser or Rcss grammar
## Call RcssParser() - the others are helper functions)
##
## Author: Tomasz Konopka
## 

##
## Function parses a text file assuming it is in css format.
## Function transfers css information into css object 
##
## file - file on disk, or vector of files on disk
##
## returns - a rcss object augmented by settings from the file(s)
##
RcssParser <- function(file) {
  lex <- RcssLexer(file)
  parsetree <- RcssMakeParseTree(lex)
  parsetree
}




## ################################################################
## Helper functions

## Wrapper to produce generic errors
RcssParseError <- function(lextab, n, parsefunction, expecting) {
  stopCF(paste0(parsefunction,": \n",
             "   expecting: ", expecting,"\n",
             "   got: ", lextab[n,"token"],"\n"))
}




## ################################################################
## Functions for top-level manipulations of parsed tokens
## and their conversion into an rcss object

## Converts a table of lexed tokens into a tree with
## slectors and declarations 
## 
## lextab - a data.frame with lexed tokens
##
RcssMakeParseTree <- function(lextab) {
  
  parsetree <- list()
  n <- 1
  
  ## Start Parsing until "n" is within the lex table
  while (n <= nrow(lextab)) {
    ruleset <- RcssParseRuleSet(lextab, n)
    parsetree[[length(parsetree)+1]] <- ruleset$RuleSet
    n <- ruleset$n
  }
  
  parsetree
}




## ################################################################
## Functions that parse at individual states


##
## Parse one rule set (Selectors { Declarations })
##
## lextab - data frame with tokens
## n - current row in the data frame
##
##
RcssParseRuleSet <- function(lextab, n) {
  selectors = RcssParseSelectorSet(lextab, n)
  declarations = RcssParseDeclarationSet(lextab, selectors$n)
  list(n = declarations$n,
       RuleSet = list(SelectorSet = selectors$SelectorSet,
                      DeclarationSet = declarations$DeclarationSet))
}


##
## Parse one selector (IDENT [ class]* | class)
## 
## lextab - data frame with tokens
## n - the current row in the data frame
##
## return - new n after parsed selectors, list of selectors
RcssParseSelector <- function(lextab, n) {

  ## the output here will be an object name followed by classes
  ans <- c("")

  ## parse the initial IDENT object
  if (lextab[n, "token"] == ".") {
    ## this is ok too  
  } else if (lextab[n, "type"] == "IDENT") {
    ans[1] <- lextab[n, "token"]
    n <- n + 1
  } else {
    RcssParseError(lextab, n, "RcssParseSelector", ".")
  }
  
  ## after the first ident, can have classes
  while (n <= nrow(lextab) & lextab[n,"token"] == ".") {
    n <- n + 1
    if (lextab[n,"type"] == "IDENT") {
      ##ans[length(ans)+1] = paste0(lextab[n, "token"]) ## ****
      ans[length(ans)+1] = lextab[n, "token"]
      n <- n +1
    } else {
      RcssParseError(lextab, n, "RcssParseSelector", "IDENT")
    }
  }

  ## at this point, collected all IDENT and classes
  list(n = n, Selector = ans)
}


##
## Parse a set of selectors (i.e. selector [ ',' selector ]*)
##
## lextab - data frame with tokens
## n - the current row in the data frame 
##
RcssParseSelectorSet <- function(lextab, n) {

  ans <- list()
  
  ## get the first selector and advance counter
  sel <- RcssParseSelector(lextab, n)
  ans[[1]] <- sel$Selector;
  n <- as.integer(sel$n);
  
  ## now parse other selectors as long as they are separated by ','
  while (n <= nrow(lextab) & lextab[n,"token"] == ",") {
    sel = RcssParseSelector(lextab, n + 1)
    ans[[length(ans) + 1]] = sel$Selector
    n <- sel$n
  }

  list(n = n, SelectorSet = ans)
}


##
## Parse a set of declarations
## (expects to start with a '{' and end with a '}')
##
## lextab - data frame with tokens
## n - the current row in the data.frame to process
##
RcssParseDeclarationSet <- function(lextab, n) {

  ## check the parsing starts with an open brace
  if (lextab[n,"token"] != "{") {
    RcssParseError(lextab, n, "RcssParseDeclarationSet", "{");
  }
  
  n <- n + 1;
  ans <- list();

  ## keep reading declarations until hit a }
  while(n <= nrow(lextab) & lextab[n, "token"] != "}") {
    ## parse a property/expr pair
    exprprop = RcssParseDeclaration(lextab, n)
    if (is.null(exprprop$Expr)) {
      ans[length(ans)+1] = list(NULL)
    } else {
      ans[[length(ans) + 1]] <- exprprop$Expr    
    }
    names(ans)[length(ans)] = exprprop$Property
    
    ## advance the n counter over this property/expr pair
    n <- exprprop$n
  }

  ## when the loop ends, the current state is a "}". Move over, finish.
  n <- n + 1
  list(n = n, DeclarationSet = ans)
}


## helper to parse special strings into R primitives
##
## tok - one string
## return - an R primitive for that string, "NULL" gives an empt list
parseIDENT <- function(tok) {
  if (tok == "NULL") {
    return(list())
  } else if (tok %in% c("TRUE", "FALSE", "NA")) {
    return(as.logical(tok))
  } 
  tok
}


##
## Parse one declaration (expects an IDENT)
##
RcssParseDeclaration <- function(lextab, n) {
  
  ## get name of the property, then move over the property
  property <- RcssParseProperty(lextab, n)
  n <- property$n
  property <- property$Property
  
  ## move over the ':'
  if (lextab[n,"token"] != ":") {
    RcssParseError(lextab, n, "RcssParseDeclaration", ":");
  }
  n <- n + 1

  ## keep reading the declarations until hit a ';' or a '}'
  expr <- c()
  exprlen <- 0
  while (n <= nrow(lextab) & !lextab[n,"token"] %in% c(";", "}")) {
    
    ## allowed tokens are IDENT, NUMBER, HEXCOLOR, STRING
    if (lextab[n, "type"] %in% c("IDENT", "STRING", "HEXCOLOR")) {
      expr[exprlen + 1] <- parseIDENT(lextab[n, "token"])
    } else if (lextab[n, "type"]=="NUMBER") {
      expr[exprlen+1] <- as.numeric(lextab[n,"token"])
    } else {
      RcssParseError(lextab, n, "RcssParseDeclaration",
                     "IDENT|NUMBER|HEXCOLOR|STRING")
    }
    
    n <- n + 1
    exprlen <- exprlen + 1
  }

  ## if current token is an end of declaration, skip over it
  if (lextab[n, "token"] == ";") {
    n <- n + 1
  }

  ## when the user leaves the expression blank, assume that means ""
  ## e.g. this can be used to set xlab=""
  if (exprlen==0) {
    expr <- ""
  }

  ## return an object with the property/expr pair, but also
  ## the number of the next non-trivial token
  list(n = n, Property = property, Expr = expr)
}


## parse the name of a property
## This is usually one token,
## but if a property has dots (e.g. cex.axis) this function
## will concatenate the text together
RcssParseProperty <- function(lextab, n) {

  if (lextab[n,"type"] != "IDENT") {
    RcssParseError(lextab, n, "RcssParseProperty", "IDENT");
  }

  ## deal with the expected case (one token)
  property <- lextab[n, "token"]
  n <- n + 1

  ## deal with extension of property when the next tokens are ".IDENT"
  while (n < nrow(lextab) &
         lextab[n,"type"] == "TERMINAL" & lextab[n, "token"] == ".") {
    
    if (lextab[n + 1, "type"] == "IDENT") {
      property <- c(property, ".", lextab[n + 1, "token"])
    }
    
    n <- n + 2;
  }
  
  ## build a unified property
  property <- paste(property, collapse = "")
  
  list(n = n, Property = property)
}

