OpenRSurvey <- function() {
  # Activates the main GUI for RSurvey.

  # Additional functions (subroutines)

  # Close GUI

  CloseGUI <- function() {
    tclServiceMode(FALSE)
    if (as.integer(tclvalue(tt.done.var)) != 0)
      return()
    CloseDevices()

    geo <- unlist(strsplit(as.character(tkwm.geometry(tt)), "\\+"))
    Data("win.loc", paste("+", as.integer(geo[2]),
                          "+", as.integer(geo[3]), sep=""))

    tclvalue(tt.done.var) <- 1
    tkdestroy(tt)
    tclServiceMode(TRUE)
  }

  # Open binary project file

  OpenProj <- function() {
    f <- GetFile(cmd="Open", exts="rda", win.title="Open Project File",
                 parent=tt)
    if (is.null(f))
      return()
    if (ClearObjs() == "cancel")
      return()

    project <- NULL
    load(file=f$path)
    Data(replace.all=project)
    Data("proj.file", f$path)

    SetCsi()
    SetVars()
  }

  # Save binary project file

  SaveProj <- function() {
    if (!is.null(Data("proj.file"))) {
      if (file.access(Data("proj.file"), mode = 0) != 0)
          Data("proj.file", NULL)
    }
    if (is.null(Data("proj.file"))) {
      f <- GetFile(cmd="Save As", exts="rda", win.title="Save Project As",
                   defaultextension="rda", parent=tt)
      if (!is.null(f)) {
        Data("proj.file", f$path)
        pth <- paste(head(unlist(strsplit(f$path, "/")), -1), collapse="/")
        Data("default.dir", pth)
      }
    }
    if (!is.null(Data("proj.file"))) {
      csi <- Data("csi")
      Data("csi", NULL)

      project <- Data()
      save(project, file=Data("proj.file"), compress=TRUE)

      Data("csi", csi)
    }
  }

  # Save a new binary project file

  SaveProjAs <- function() {
    Data("proj.file", NULL)
    SaveProj()
    tkfocus(tt)
  }

  # Clear objects

  ClearObjs <- function() {
    msg <- "Save the existing project?"
    if (is.null(Data("proj.file")))
      ans <- "no"
    else
      ans <- as.character(tkmessageBox(icon="question", message=msg,
                                       title="Warning", type="yesnocancel",
                                       parent=tt))
    if (ans == "cancel") {
      return(ans)
    } else if (ans == "yes") {
      SaveProj()
    }
    Data(clear.proj=TRUE)
    SetVars()
    ans
  }

  # Import survey data

  CallImportData <- function() {
    ImportData(tt)
    SetVars()
  }

  # Set button state

  ButtonState <- function(vars) {
    s <- "normal"
    if (is.null(vars$x) | is.null(vars$y))
      s <- "disabled"
    tkconfigure(frame2.but.1.1, state=s)

    s <- "normal"
    if (is.null(vars$x) | is.null(vars$y) | is.null(vars$z))
      s <- "disabled"
    tkconfigure(frame2.but.1.2, state=s)
    tkconfigure(frame2.but.2.2, state=s)

    s <- "normal"
    if (is.null(vars$z) | is.null(vars$t))
      s <- "disabled"
    tkconfigure(frame2.but.2.1, state=s)
  }

  # Set variables

  SetVars <- function() {
    tkset(frame1.box.1.2, "")
    tkset(frame1.box.2.2, "")
    tkset(frame1.box.3.2, "")
    tkset(frame1.box.4.2, "")
    tkset(frame1.box.5.2, "")
    tkset(frame1.box.6.2, "")

    cols <- Data("cols")
    vars <- Data("vars")

    if (is.null(cols) | is.null(vars)) {
      tkconfigure(frame1.box.1.2, value="")
      tkconfigure(frame1.box.2.2, value="")
      tkconfigure(frame1.box.3.2, value="")
      tkconfigure(frame1.box.4.2, value="")
      tkconfigure(frame1.box.5.2, value="")
      tkconfigure(frame1.box.6.2, value="")
      ButtonState(vars)
      if (is.null(cols))
        return()
    }

    ids <- sapply(cols, function(i) i$id)
    classes <- sapply(cols, function(i) i$class)

    idxs.t <- which(classes %in% c("POSIXct"))
    idxs.n <- which(classes %in% c("numeric", "integer"))

    vals.t <- c("", ids[idxs.t])
    vals.n <- c("", ids[idxs.n])

    tkconfigure(frame1.box.1.2, value=vals.n)
    tkconfigure(frame1.box.2.2, value=vals.n)
    tkconfigure(frame1.box.3.2, value=vals.n)
    tkconfigure(frame1.box.4.2, value=vals.t)
    tkconfigure(frame1.box.5.2, value=vals.n)
    tkconfigure(frame1.box.6.2, value=vals.n)

    if (!is.null(vars$x))
      tcl(frame1.box.1.2, "current", which(vars$x  == idxs.n))
    if (!is.null(vars$y))
      tcl(frame1.box.2.2, "current", which(vars$y  == idxs.n))
    if (!is.null(vars$z))
      tcl(frame1.box.3.2, "current", which(vars$z  == idxs.n))
    if (!is.null(vars$t))
      tcl(frame1.box.4.2, "current", which(vars$t  == idxs.t))
    if (!is.null(vars$vx))
      tcl(frame1.box.5.2, "current", which(vars$vx == idxs.n))
    if (!is.null(vars$vy))
      tcl(frame1.box.6.2, "current", which(vars$vy == idxs.n))

    ButtonState(vars)
  }

  # Refresh variables

  RefreshVars <- function(item) {
    cols <- Data("cols")

    col.classes <- sapply(cols, function(i) i$class)

    idxs.t <- which(col.classes %in% c("POSIXct"))
    idxs.n <- which(col.classes %in% c("numeric", "integer"))

    idx.x  <- as.integer(tcl(frame1.box.1.2, "current"))
    idx.y  <- as.integer(tcl(frame1.box.2.2, "current"))
    idx.z  <- as.integer(tcl(frame1.box.3.2, "current"))
    idx.t  <- as.integer(tcl(frame1.box.4.2, "current"))
    idx.vx <- as.integer(tcl(frame1.box.5.2, "current"))
    idx.vy <- as.integer(tcl(frame1.box.6.2, "current"))

    vars <- list()

    if (idx.x > 0)
      vars$x <- idxs.n[idx.x]
    if (idx.y > 0)
      vars$y <- idxs.n[idx.y]
    if (idx.z > 0)
      vars$z <- idxs.n[idx.z]
    if (idx.t > 0)
      vars$t <- idxs.t[idx.t]
    if (idx.vx > 0)
      vars$vx <- idxs.n[idx.vx]
    if (idx.vy > 0)
      vars$vy <- idxs.n[idx.vy]

    if (!identical(vars, Data("vars"))) {
      Data("vars", vars)
      Data("data.pts", NULL)
      Data("data.grd", NULL)
    }
    ButtonState(vars)
  }

  # Manage data

  CallManageData <- function() {
    ManageData(Data("cols"), Data("vars"), tt)
    SetVars()
    tkfocus(tt)
  }

  # Export data

  CallExportData <- function(file.type) {
    if (is.null(Data("data.raw")))
      return()

    is.coordinate <- !is.null(Data("vars")$x) & !is.null(Data("vars")$y)
    if (!is.coordinate & file.type %in% c("shape", "grid"))
      stop("Spatial coordinates missing")

    if (file.type == "grid") {
      CallProcessData(interpolate=TRUE)
      WriteFile(file.type="grid")
    } else {
      CallProcessData()
      col.ids <- sapply(Data("cols"), function(i) i$id)
      ExportData(col.ids, file.type=file.type, parent=tt)
    }

    tkfocus(tt)
  }

  # Close graphic devices

  CloseDevices <- function() {
    graphics.off()
    while (rgl.cur() != 0)
      rgl.close()
  }

  # Save R graphic devices

  SaveRDevice <- function() {
    if (is.null(dev.list()))
      return()
    exts <- c("eps", "png", "jpg", "jpeg", "pdf", "bmp", "tif", "tiff")
    f <- GetFile(cmd="Save As", exts=exts, win.title="Save R Graphic As",
                 defaultextension="eps", parent=tt)
    if (is.null(f))
      return()
    savePlot(filename=f$path, type=f$ext)
  }

  # Save RGL graphic devices

  SaveRGLDevice <- function() {
    if (rgl.cur() == 0)
      return()
    f <- GetFile(cmd="Save As", exts=c("png", "eps", "pdf"),
                 win.title="Save RGL Graphic As", defaultextension="png",
                 parent=tt)
    if (is.null(f))
      return()

    if (f$ext == "png")
      rgl.snapshot(filename=f$path, fmt=f$ext)
    else
      rgl.postscript(filename=f$path, fmt=f$ext)
  }

  # About package

  AboutPackage <- function() {
    if ("package:RSurvey" %in% search())
      path <- system.file("DESCRIPTION", package="RSurvey")
    else
      path <- file.path(getwd(), "DESCRIPTION")
    msg <- paste(readLines(path, n=-1L), collapse="\n")
    tkmessageBox(icon="info", message=msg, title="About", parent=tt)
  }

  # Manage polygons

  CallManagePolygons <- function() {
    ManagePolygons(Data("poly"), parent=tt)
  }

  # Set polygon range and limit

  CallSetPolygonLimits <- function() {
    pdata.old <- Data("poly.data")
    pcrop.old <- Data("poly.crop")

    ans <- SetPolygonLimits(names(Data("poly")), pdata.old, pcrop.old, tt)

    if (!is.null(ans)) {
      if (!identical(ans$poly.data, pdata.old)) {
        Data("poly.data", ans$poly.data)
        Data("data.pts", NULL)
        Data("data.grd", NULL)
      }
      if (!identical(ans$poly.crop, pcrop.old)) {
        Data("poly.crop", ans$poly.crop)
        Data("data.grd", NULL)
      }
    }
    tkfocus(tt)
  }

  # Construct polygon

  ConstructPolygon <- function(type) {
    if (is.null(Data("data.source")))
      return()
    msg <- paste("After the plot has been created, use the mouse to identify",
                 "the vertices of the polygon. The identification process can",
                 "be terminated by clicking the second button and selecting",
                 "[Stop] from the menu, or from the [Stop] menu on the",
                 "graphics window.", sep="\n")
    tkmessageBox(icon="info", message=msg, title="Build Polygon", type="ok",
                 parent=tt)
    CallPlot2d(type=type, build.poly=TRUE)
    tkfocus(tt)
  }

  # Autocrop polygon

  CallAutocropPolygon <- function() {
    if (is.null(Data("data.source")))
      return()
    CallProcessData()

    d <- Data("data.pts")

    xlab <- Data("cols")[[Data("vars")$x]]$id
    ylab <- Data("cols")[[Data("vars")$y]]$id
    zlab <- Data("cols")[[Data("vars")$z]]$id

    asp     <- Data("asp.yx")
    csi     <- Data("csi")
    width   <- Data("width")
    nlevels <- Data("nlevels")
    cex.pts <- Data("cex.pts")
    rkey    <- Data("rkey")

    ply.new <- AutocropPolygon(d, tt, xlab=xlab, ylab=ylab, zlab=zlab,
                               asp=asp, csi=csi, width=width, nlevels=nlevels,
                               cex.pts=cex.pts, rkey=rkey)

    if (inherits(ply.new, "gpc.poly")) {
      ply <- list()
      if (!is.null(Data("poly")))
        ply <- Data("poly")
      ply.name <- NamePolygon(old=names(ply))
      ply[[ply.name]] <- ply.new
      Data("poly", ply)
      Data("poly.crop", ply.name)
      Data("data.grd", NULL)
    }
    tkfocus(tt)
  }

  # Name polygon

  NamePolygon <- function(old=NULL, nam=NA){
    if (is.na(nam))
      nam <- "New Polygon"
    idx <- 1
    chk <- nam
    while (chk %in% old) {
      chk <- paste(nam, " (", idx, ")", sep="")
      idx <- idx + 1
    }
    chk
  }

  # Plot temporal data

  CallPlotTimeSeries <- function() {
    CallProcessData()
    if (is.null(Data("data.pts")))
      return()

    dat          <- Data("data.pts")
    lim          <- Data("lim.axes")
    cols         <- Data("cols")
    vars         <- Data("vars")

    ylab <- cols[[vars$z]]$id

    axis.side <- 1:2
    if (Data("show.2.axes"))
      axis.side <- 1:4

    tkconfigure(tt, cursor="watch")
    PlotTimeSeries(x=dat$t, y=dat$z, xlim=lim$t, ylim=lim$z, ylab=ylab,
                   tgap=Data("tgap"), width=Data("width"),
                   cex.pts=Data("cex.pts"), axis.side=axis.side,
                   minor.ticks=Data("minor.ticks"),
                   ticks.inside=Data("ticks.inside"),
                   rm.pnt.line=Data("rm.pnt.line"))
    tkconfigure(tt, cursor="arrow")
    tkfocus(tt)
  }

  # Plot point or 2d surface data

  CallPlot2d <- function(type, build.poly=FALSE) {
    if (type == "p")
      CallProcessData()
    else
      CallProcessData(interpolate=TRUE)

    if (is.null(Data("data.grd")) && type %in% c("g", "l")) {
      return()
    } else if (is.null(Data("data.pts"))) {
      return()
    }

    ply <- if (type == "p") Data("poly.data") else Data("poly.crop")
    if (!is.null(ply))
      ply <- Data("poly")[[ply]]

    show.poly   <- Data("show.poly") && inherits(ply, "gpc.poly")
    show.lines  <- type %in% c("l", "g") && Data("show.lines")
    show.points <- type %in% c("l", "g") && Data("show.points")

    axis.side <- 1:2
    if (Data("show.2.axes"))
      axis.side <- 1:4

    nlevels <- Data("nlevels")
    cols    <- Data("cols")
    vars    <- Data("vars")

    xlab <- cols[[vars$x]]$id
    ylab <- cols[[vars$y]]$id
    zlab <- if (is.null(vars$z)) NULL else cols[[vars$z]]$id

    if (type == "p") {
      dat <- Data("data.pts")
    } else if (type %in% c("l", "g")) {
      dat <- Data("data.grd")
    }

    if (type == "g") {
      x.midpoint <- dat$x[1:(length(dat$x) - 1)] + diff(dat$x) / 2
      y.midpoint <- dat$y[1:(length(dat$y) - 1)] + diff(dat$y) / 2
      xran <- range(x.midpoint, finite=TRUE)
      yran <- range(y.midpoint, finite=TRUE)
    } else {
      xran <- range(dat$x, finite=TRUE)
      yran <- range(dat$y, finite=TRUE)
    }

    # Adjust axes limits for polygon

    lim <- Data("lim.axes")

    xlim <- lim$x
    if (is.null(xlim))
      xlim <- c(NA, NA)
    ylim <- lim$y
    if (is.null(ylim))
      ylim <- c(NA, NA)

    if (show.poly) {
      bbx <- bby <- NULL
      bb <- get.bbox(ply)

      if (!is.na(xlim[1]))
        bb$x[1] <- xlim[1]
      if (!is.na(xlim[2]))
        bb$x[2] <- xlim[2]
      if (!is.na(ylim[1]))
        bb$y[1] <- ylim[1]
      if (!is.na(ylim[2]))
        bb$y[2] <- ylim[2]

      xy <- cbind(x=c(bb$x, rev(bb$x)), y=c(bb$y[c(1,1)], bb$y[c(2,2)]))
      bb <- get.bbox(intersect(ply, as(xy, "gpc.poly")))
      bbx <- range(bb$x)
      bby <- range(bb$y)
      bbx <- extendrange(bbx, f=0.02)
      bby <- extendrange(bby, f=0.02)
      if (is.na(xlim[1]) && bbx[1] < xran[1])
        lim$x[1] <- bbx[1]
      if (is.na(xlim[2]) && bbx[2] > xran[2])
        lim$x[2] <- bbx[2]
      if (is.na(ylim[1]) && bby[1] < yran[1])
        lim$y[1] <- bby[1]
      if (is.na(ylim[2]) && bby[2] > yran[2])
        lim$y[2] <- bby[2]
    }

    tkconfigure(tt, cursor="watch")
    Plot2d(dat, type=type, xlim=lim$x, ylim=lim$y, zlim=lim$z,
           xlab=xlab, ylab=ylab, zlab=zlab, asp=Data("asp.yx"),
           csi=Data("csi"), width=Data("width"), nlevels=nlevels,
           cex.pts=Data("cex.pts"), rkey=Data("rkey"),
           color.palette=Data("color.palette"),
           vuni=Data("vuni"), vmax=Data("vmax"),
           vxby=Data("vxby"), vyby=Data("vyby"),
           axis.side=axis.side, minor.ticks=Data("minor.ticks"),
           ticks.inside=Data("ticks.inside"), rm.pnt.line=Data("rm.pnt.line"),
           add.contour.lines=show.lines)

    if (show.poly)
      plot(ply, add=TRUE, poly.args=list(border="black", lty=3))
    if (show.points)
      points(x=Data("data.pts")$x, y=Data("data.pts")$y, pch=19,
             cex=Data("cex.pts") / 2, col="black")
    if (build.poly) {
      v <- locator(type="o", col="black", bg="black", pch=22)
      loc.xy <- cbind(c(v$x, v$x[1]), c(v$y, v$y[1]))
      points(loc.xy, col="black", bg="black", pch=22)
      lines(loc.xy, col="black")

      ply.new <- as(as.data.frame(v), "gpc.poly")
      if (!is.null(ply))
        ply.new <- intersect(ply, ply.new)
      if (area.poly(ply.new) == 0) {
        msg <- "The resulting polygon is invalid."
        tkmessageBox(icon="warning", message=msg, title="Polygon Discarded",
                     parent=tt)
        ply.new <- NULL
      }

      if (inherits(ply.new, "gpc.poly")) {
        ply.list <- if (is.null(Data("poly"))) list() else Data("poly")
        ply.name <- NamePolygon(old=names(ply.list))
        ply.list[[ply.name]] <- ply.new

        if (type == "p") {
          pts <- get.pts(ply.new)
          logic <- rep(TRUE, nrow(dat))
          for (i in seq(along=pts)) {
              is.in <-  point.in.polygon(point.x=dat$x, point.y=dat$y,
                                         pol.x=pts[[i]]$x, pol.y=pts[[i]]$y) > 0
              is.in <- if (pts[[i]]$hole) !is.in else is.in
              logic <- logic & is.in
          }
          if (any(logic)) {
            points(dat$x[logic], dat$y[logic], col="red",
                   cex=Data("cex.pts"), pch=20)
            Data("poly", ply.list)
            Data("poly.data", ply.name)
            Data("data.pts", NULL)
            Data("data.grd", NULL)
          } else {
            msg <- "No data points fall within the given polygon."
            tkmessageBox(icon="warning", message=msg, title="Polygon Discarded",
                         parent=tt)
          }
        } else if (type == "l") {
          cutout <- CutoutPolygon(dat, ply.new)
          if (!is.null(cutout)) {
            Data("poly", ply.list)
            Data("poly.crop", ply.name)
            Data("data.grd", NULL)
          }
        }
      }
    }
    tkconfigure(tt, cursor="arrow")
    tkfocus(tt)
  }

  # Plot 3d surface data

  CallPlot3d <- function() {
    CallProcessData(interpolate=TRUE)

    if (is.null(Data("data.grd")))
      return()

    dat <- Data("data.grd")
    pts <- NULL
    if (Data("show.points"))
      pts <- Data("data.pts")
    lim <- Data("lim.axes")

    tkconfigure(tt, cursor="watch")
    Plot3d(x=dat, px=pts, xlim=lim$x, ylim=lim$y, zlim=lim$z,
           vasp=Data("asp.zx"), hasp=Data("asp.yx"),
           width=Data("width"), cex.pts=Data("cex.pts"),
           nlevels=Data("nlevels"), color.palette=Data("color.palette"))
    tkconfigure(tt, cursor="arrow")
    tkfocus(tt)
  }

  # Open HTML help for R functions

  OpenHTMLHelp <- function() {
    if (!("RSurvey" %in% .packages(all.available=TRUE)))
      stop("requires installed RSurvey package", call.=FALSE)
    if (tools:::httpdPort == 0L)
      tools::startDynamicHelp()
    if (tools:::httpdPort > 0L) {
      url <- paste("http://127.0.0.1:", tools:::httpdPort,
                   "/library/RSurvey/html/00Index.html", sep="")
      browseURL(url, browser=getOption("browser"))
    } else {
      stop("requires the HTTP server to be running", call.=FALSE)
    }
    invisible()
  }

  # Set the height of (default-sized) characters in inches.

  SetCsi <- function() {
    if (is.null(Data("csi"))) {
      x11(pointsize=12)
      Data("csi", par("csi"))
      dev.off()
    }
  }

  # Call view data for state variable data

  CallViewData <- function() {
    CallProcessData()
    if (is.null(Data("data.pts")))
      return()

    tkconfigure(tt, cursor="watch")

    vars <- Data("vars")
    cols <- Data("cols")

    state.vars <- list(x="x-axis", y="y-axis", z="z-axis",
                       t="t-axis", vx="x-vector", vy="y-vector")
    state.vars <- state.vars[names(state.vars) %in% names(vars)]
    state.idxs <- sapply(names(state.vars), function(i) vars[[i]])

    d <- Data("data.pts")[, names(state.vars)]

    fun <- function(i, type) {
      val <- cols[[i]][[type]]
      if (is.null(val)) NA else val
    }
    col.names <- sapply(state.idxs, function(i) fun(i, "name"))
    col.units <- sapply(state.idxs, function(i) fun(i, "unit"))
    col.formats <- sapply(state.idxs, function(i) fun(i, "format"))

    ViewData(d, col.names, col.units, col.formats, parent=tt)

    tkconfigure(tt, cursor="arrow")
    tkfocus(tt)
  }

  # Call process data

  CallProcessData <- function(interpolate=FALSE) {
    if (is.null(Data("data.raw"))) {
      Data("data.pts", NULL)
      Data("data.grd", NULL)
      return()
    }

    tkconfigure(tt, cursor="watch")

    # Process points

    if (is.null(Data("data.pts"))) {
      cols <- Data("cols")
      vars <- Data("vars")

      var.names <- names(vars)

      Eval <- function(v) {
        if (is.null(v)) NULL else EvalFunction(cols[[v]]$fun, cols)
      }
      lst <- lapply(var.names, function(i) Eval(vars[[i]]))
      len <- sapply(lst, function(i) length(i))
      max.len <- max(len)

      d <- as.data.frame(matrix(NA, nrow=max.len, ncol=length(lst)))
      names(d) <- var.names
      for (i in seq(along=lst))
        d[[i]] <- c(lst[[i]], rep(NA, max.len - len[i]))

      lim <- Data("lim.data")

      if (!is.null(vars$x) & !is.null(vars$y)) {
        ply <- Data("poly.data")
        if (!is.null(ply))
          ply <- Data(c("poly", ply))
      } else {
        ply <- NULL
      }

      data.pts <- ProcessData(d, type="p", lim=lim, ply=ply)
      Data("data.pts", data.pts)
      Data("data.grd", NULL)
    }

    if (is.null(Data("data.pts"))) {
      tkconfigure(tt, cursor="arrow")
      return()
    }

    # Process grid

    if (is.null(Data("data.grd")) && interpolate) {
      ply <- Data("poly.crop")
      if (!is.null(ply))
        ply <- Data("poly")[[ply]]

      grid.res <- Data("grid.res")
      grid.mba <- Data("grid.mba")

      data.grd <- ProcessData(Data("data.pts"), type="g", ply=ply,
                              grid.res=grid.res, grid.mba=grid.mba)
      Data("data.grd", data.grd)
    }

    tkconfigure(tt, cursor="arrow")
  }

  # Main program

  # Load required R packages

  LoadPackages()

  # Warn if using Windows OS and running in MDI mode

  if (.Platform$OS.type == "windows" && getIdentification() == "RGui")
    message("\n\n    You are running R in MDI mode which *may* interfere\n",
            "    with the functionality of the graphical user interface.\n",
            "    It is recommended to use R in SDI mode which can be\n",
            "    set in the command line or by clicking in the Menu:\n",
            "    Edit - GUI Preferences: SDI, then Save and restart R.\n\n")

  # Establish working directory

  if ("package:RSurvey" %in% search())
    path <- system.file("RSurvey-ex", package="RSurvey")
  else
    path <- getwd()

  if (is.null(Data("default.dir")))
    Data("default.dir", path)

  if ("package:RSurvey" %in% search())
    image.path <- system.file("images", package="RSurvey")
  else
    image.path <- file.path(path, "inst", "images")

  # Set options

  SetCsi()
  options(digits.secs=3)

  # Assign variables linked to Tk entry widgets

  tt.done.var <- tclVar(0)

  # Package version number

  f <- "DESCRIPTION"
  if ("package:RSurvey" %in% search())
    f <- system.file("DESCRIPTION", package="RSurvey")
  ver <- scan(f, what="character", skip=1, nlines=1, quiet=TRUE)[2]
  Data("ver", paste("RSurvey", ver))

  # Open GUI

  tclServiceMode(FALSE)
  tt <- tktoplevel()
  tkwm.geometry(tt, Data("win.loc"))
  tktitle(tt) <- Data("ver")
  tkwm.resizable(tt, 1, 0)

  # Top menu

  top.menu <- tkmenu(tt, tearoff=0)

  # File menu

  menu.file <- tkmenu(tt, tearoff=0)
  tkadd(top.menu, "cascade", label="File", menu=menu.file, underline=0)

  tkadd(menu.file, "command", label="New project", accelerator="Ctrl+N",
        command=ClearObjs)
  tkadd(menu.file, "command", label="Open project", accelerator="Ctrl+O",
        command=OpenProj)
  tkadd(menu.file, "command", label="Save project", accelerator="Ctrl+S",
        command=SaveProj)
  tkadd(menu.file, "command", label="Save project as",
        accelerator="Shift+Ctrl+S", command=SaveProjAs)

  tkadd(menu.file, "separator")
  tkadd(menu.file, "command", label="Import data",
        command=CallImportData)

  menu.file.export <- tkmenu(tt, tearoff=0)
  tkadd(menu.file.export, "command", label="Text file",
        command=function() CallExportData("text"))
  tkadd(menu.file.export, "command", label="ESRI shapefile",
        command=function() CallExportData("shape"))
  tkadd(menu.file, "cascade", label="Export point data as", menu=menu.file.export)

  tkadd(menu.file, "command", label="Export grid data as",
        command=function() CallExportData("grid"))

  tkadd(menu.file, "separator")
  menu.file.save <- tkmenu(tt, tearoff=0)
  tkadd(menu.file.save, "command", label="R graphic", accelerator="Ctrl+R",
        command=SaveRDevice)
  tkadd(menu.file.save, "command", label="RGL graphic",
        command=SaveRGLDevice)
  tkadd(menu.file, "cascade", label="Save plot from", menu=menu.file.save)

  tkadd(menu.file, "separator")
  tkadd(menu.file, "command", label="Exit",
        command=CloseGUI)

  # Edit menu

  menu.edit <- tkmenu(tt, tearoff=0)
  tkadd(top.menu, "cascade", label="Edit", menu=menu.edit, underline=0)

  tkadd(menu.edit, "command", label="Manage data",
        command=CallManageData)

  tkadd(menu.edit, "command", label="Set data limits",
        command=function() {
          old <- Data("lim.data")
          new <- EditLimits(old, "Data Limits", tt)
          if (!identical(old, new)) {
            Data("lim.data", new)
            Data("data.pts", NULL)
            Data("data.grd", NULL)
          }
        })

  tkadd(menu.edit, "separator")
  tkadd(menu.edit, "command", label="View data",
        command=CallViewData)

  tkadd(menu.edit, "separator")
  tkadd(menu.edit, "command", label="Preferences",
        command=function() {
          SetPreferences(tt)
        })

  # Polygon menu

  menu.poly <- tkmenu(tt, tearoff=0)

  tkadd(top.menu, "cascade", label="Polygon", menu=menu.poly, underline=0)

  tkadd(menu.poly, "command", label="Manage polygons",
        command=CallManagePolygons)
  tkadd(menu.poly, "separator")
  tkadd(menu.poly, "command", label="Set polygon limits",
        command=CallSetPolygonLimits)
  tkadd(menu.poly, "command", label="Clear polygon limits",
        command=function() {
          Data("poly.data", NULL)
          Data("poly.crop", NULL)
          Data("data.pts", NULL)
          Data("data.grd", NULL)
        })
  tkadd(menu.poly, "separator")

  menu.poly.con <- tkmenu(tt, tearoff=0)
  tkadd(menu.poly.con, "command", label="Boundary defining data limits",
        command=function() {
          ConstructPolygon(type="p")
        })
  tkadd(menu.poly.con, "command", label="Crop region for interpolated surface",
        command=function() {
          ConstructPolygon(type="l")
        })
  tkadd(menu.poly, "cascade", label="Build", menu=menu.poly.con)
  tkadd(menu.poly, "command", label="Autocrop region",
        command=CallAutocropPolygon)

  # Plot menu

  menu.plot <- tkmenu(tt, tearoff=0)
  tkadd(top.menu, "cascade", label="Plot", menu=menu.plot, underline=0)

  tkadd(menu.plot, "command", label="Configuration",
        command=function() {
          SetConfiguration(tt)
        })

  tkadd(menu.plot, "command", label="Set axes limits",
        command=function() {
          lim <- EditLimits(Data("lim.axes"), "Axes Limits", tt)
          Data("lim.axes", lim)
        })
  tkadd(menu.plot, "command", label="Set color palette",
        command=function() {
          pal <- ChoosePalette(Data("color.palette"), Data("nlevels"), tt)
          if (!is.null(pal))
            Data("color.palette", pal)
        })

  tkadd(menu.plot, "separator")
  tkadd(menu.plot, "command", label="Close all plots", accelerator="Ctrl+F4",
        command=CloseDevices)

  # Help menu

  menu.help <- tkmenu(tt, tearoff=0)
  tkadd(top.menu, "cascade", label="Help", menu=menu.help, underline=0)
  tkadd(menu.help, "command", label="R functions (html)",
        command=OpenHTMLHelp)
  tkadd(menu.help, "separator")
  tkadd(menu.help, "command", label="About",
        command=AboutPackage)

  if (!("RSurvey" %in% .packages())) {
      tkadd(menu.help, "separator")
      tkadd(menu.help, "command", label="Restore R session",
            command=function() {
              CloseGUI()
              Data("data.pts", NULL)
              Data("data.grd", NULL)
              RestoreSession(file.path(getwd(), "R"), save.objs="Data",
                             fun.call="OpenRSurvey")
            })
  }

  # Finalize top menu

  tkconfigure(tt, menu=top.menu)

  # Frame 0, toolbar with command buttons

  new.var     <- tclVar()
  save.var    <- tclVar()
  import.var  <- tclVar()
  data.var    <- tclVar()
  polygon.var <- tclVar()
  globe.var   <- tclVar()
  config.var  <- tclVar()
  axes.var    <- tclVar()
  help.var    <- tclVar()
  close.var   <- tclVar()

  frame0 <- ttkframe(tt, relief="flat", borderwidth=2)
  tkpack(frame0, side="top", fill="x")

  tkimage.create("photo", new.var, format="GIF",
                 file=file.path(image.path, "new.gif"))
  tkimage.create("photo", save.var, format="GIF",
                 file=file.path(image.path, "save.gif"))
  tkimage.create("photo", import.var, format="GIF",
                 file=file.path(image.path, "import.gif"))
  tkimage.create("photo", data.var, format="GIF",
                 file=file.path(image.path, "data.gif"))
  tkimage.create("photo", polygon.var, format="GIF",
                 file=file.path(image.path, "polygon.gif"))
  tkimage.create("photo", config.var, format="GIF",
                 file=file.path(image.path, "config.gif"))
  tkimage.create("photo", axes.var, format="GIF",
                 file=file.path(image.path, "axes.gif"))
  tkimage.create("photo", help.var, format="GIF",
                 file=file.path(image.path, "help.gif"))
  tkimage.create("photo", close.var, format="GIF",
                 file=file.path(image.path, "close.gif"))

  frame0.but.1  <- tkbutton(frame0, relief="flat", overrelief="raised",
                            borderwidth=1, image=new.var,
                            command=ClearObjs)
  frame0.but.2  <- tkbutton(frame0, relief="flat", overrelief="raised",
                            borderwidth=1, image=save.var,
                            command=SaveProj)
  frame0.but.3  <- tkbutton(frame0, relief="flat", overrelief="raised",
                            borderwidth=1, image=import.var,
                            command=CallImportData)
  frame0.but.4  <- tkbutton(frame0, relief="flat", overrelief="raised",
                            borderwidth=1, image=data.var,
                            command=CallManageData)
  frame0.but.5  <- tkbutton(frame0, relief="flat", overrelief="raised",
                            borderwidth=1, image=polygon.var,
                            command=CallManagePolygons)
  frame0.but.6  <- tkbutton(frame0, relief="flat", overrelief="raised",
                            borderwidth=1, image=config.var,
                            command=function() SetConfiguration(tt))
  frame0.but.7  <- tkbutton(frame0, relief="flat", overrelief="raised",
                            borderwidth=1, image=axes.var,
                            command=function() {
                             lim <- EditLimits(Data("lim.axes"),
                                               "Axes Limits", tt)
                             Data("lim.axes", lim)
                           })
  frame0.but.8  <- tkbutton(frame0, relief="flat", overrelief="raised",
                            borderwidth=1, image=help.var,
                            command=OpenHTMLHelp)
  frame0.but.9  <- tkbutton(frame0, relief="flat", overrelief="raised",
                            borderwidth=1, image=close.var,
                            command=CloseDevices)

  tkpack(frame0.but.1, frame0.but.2, frame0.but.3, frame0.but.4, frame0.but.5,
         frame0.but.6, frame0.but.7, frame0.but.8, frame0.but.9, side="left")

  separator <- ttkseparator(tt, orient="horizontal")
  tkpack(separator, fill="x")

  # Frame 1, variables

  frame1 <- ttklabelframe(tt, relief="flat", borderwidth=5, padding=5,
                          text="State variables")

  frame1.lab.1.1 <- ttklabel(frame1, text="x-axis")
  frame1.lab.2.1 <- ttklabel(frame1, text="y-axis")
  frame1.lab.3.1 <- ttklabel(frame1, text="z-axis")
  frame1.lab.4.1 <- ttklabel(frame1, text="t-axis")
  frame1.lab.5.1 <- ttklabel(frame1, text="x-vector")
  frame1.lab.6.1 <- ttklabel(frame1, text="y-vector")

  frame1.box.1.2 <- ttkcombobox(frame1, state="readonly")
  frame1.box.2.2 <- ttkcombobox(frame1, state="readonly")
  frame1.box.3.2 <- ttkcombobox(frame1, state="readonly")
  frame1.box.4.2 <- ttkcombobox(frame1, state="readonly")
  frame1.box.5.2 <- ttkcombobox(frame1, state="readonly")
  frame1.box.6.2 <- ttkcombobox(frame1, state="readonly")

  tkgrid(frame1.lab.1.1, frame1.box.1.2, pady=c(0, 4))
  tkgrid(frame1.lab.2.1, frame1.box.2.2, pady=c(0, 4))
  tkgrid(frame1.lab.3.1, frame1.box.3.2, pady=c(0, 4))
  tkgrid(frame1.lab.4.1, frame1.box.4.2, pady=c(0, 4))
  tkgrid(frame1.lab.5.1, frame1.box.5.2, pady=c(0, 4))
  tkgrid(frame1.lab.6.1, frame1.box.6.2)

  tkgrid.configure(frame1.lab.1.1, frame1.lab.2.1, frame1.lab.3.1,
                   frame1.lab.4.1, frame1.lab.5.1, frame1.lab.6.1,
                   sticky="e", padx=c(0, 2))

  tkgrid.configure(frame1.box.1.2, frame1.box.2.2, frame1.box.3.2,
                   frame1.box.4.2, frame1.box.5.2, frame1.box.6.2, sticky="we")
  tkgrid.configure(frame1.box.1.2, frame1.box.2.2, frame1.box.3.2,
                   frame1.box.4.2, frame1.box.5.2)

  tkgrid.columnconfigure(frame1, 1, weight=1, minsize=25)

  tkpack(frame1, fill="x", expand=TRUE, ipadx=0, ipady=0, padx=10, pady=5)

  # Frame 2, plotting buttons

  frame2 <- ttklabelframe(tt, relief="flat", borderwidth=5, padding=5,
                          text="Plot types")

  frame2.but.1.1 <- ttkbutton(frame2, width=15, text="Scatter",
                              command=function() {
                                CallPlot2d(type="p")
                              })
  frame2.but.1.2 <- ttkbutton(frame2, width=15, text="2D Surface",
                              command=function() {
                                type <- if (Data("img.contour")) "g" else "l"
                                CallPlot2d(type=type)
                              })
  frame2.but.2.1 <- ttkbutton(frame2, width=15, text="Time Series",
                              command=CallPlotTimeSeries)
  frame2.but.2.2 <- ttkbutton(frame2, width=15, text="3D Surface",
                              command=CallPlot3d)

  tkgrid(frame2.but.1.1, frame2.but.1.2, pady=c(0, 4))
  tkgrid(frame2.but.2.1, frame2.but.2.2, pady=0)

  tkgrid.configure(frame2.but.1.1, frame2.but.2.1, padx=c(0, 4))

  tcl("grid", "anchor", frame2, "center")

  tkpack(frame2, fill="x", ipadx=0, ipady=0, expand=TRUE,
         padx=10, pady=c(0, 10))

  # Set variables

  SetVars()

  # Bind events

  tclServiceMode(TRUE)

  tkbind(tt, "<Destroy>", CloseGUI)

  tkbind(tt, "<Control-n>", ClearObjs)
  tkbind(tt, "<Control-o>", OpenProj)
  tkbind(tt, "<Control-s>", SaveProj)
  tkbind(tt, "<Shift-Control-S>", SaveProjAs)
  tkbind(tt, "<Control-r>", SaveRDevice)
  tkbind(tt, "<Control-F4>", CloseDevices)

  tkbind(frame1.box.1.2, "<<ComboboxSelected>>", RefreshVars)
  tkbind(frame1.box.2.2, "<<ComboboxSelected>>", RefreshVars)
  tkbind(frame1.box.3.2, "<<ComboboxSelected>>", RefreshVars)
  tkbind(frame1.box.4.2, "<<ComboboxSelected>>", RefreshVars)
  tkbind(frame1.box.5.2, "<<ComboboxSelected>>", RefreshVars)
  tkbind(frame1.box.6.2, "<<ComboboxSelected>>", RefreshVars)

  # GUI closure

  tkfocus(force=tt)
  invisible()
}
