% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ALD.R
\name{ALD_RNAmf}
\alias{ALD_RNAmf}
\title{find the next point by ALD criterion}
\usage{
ALD_RNAmf(Xcand = NULL, fit, mc.sample = 1000, cost = NULL,
optim = TRUE, parallel = FALSE, ncore = 1, trace=TRUE)
}
\arguments{
\item{Xcand}{vector or matrix of candidate set which could be added into the current design only used when \code{optim=FALSE}. \code{Xcand} is the set of the points where ALD criterion is evaluated. If \code{Xcand=NULL}, \eqn{100 \times d} number of points from 0 to 1 are generated by Latin hypercube design. Default is \code{NULL}.}

\item{fit}{object of class \code{RNAmf}.}

\item{mc.sample}{a number of mc samples generated for the MC approximation. Default is \code{1000}.}

\item{cost}{vector of the costs for each level of fidelity. If \code{cost=NULL}, total costs at all levels would be 1. \code{cost} is encouraged to have a ascending order of positive value. Default is \code{NULL}.}

\item{optim}{logical indicating whether to optimize AL criterion by \code{optim}'s gradient-based \code{L-BFGS-B} method. If \code{optim=TRUE}, \eqn{5 \times d} starting points are generated by Latin hypercube design for optimization. If \code{optim=FALSE}, AL criterion is optimized on the \code{Xcand}. Default is \code{TRUE}.}

\item{parallel}{logical indicating whether to compute the AL criterion in parallel or not. If \code{parallel=TRUE}, parallel computation is utilized. Default is \code{FALSE}.}

\item{ncore}{a number of core for parallel. It is only used if \code{parallel=TRUE}. Default is 1.}

\item{trace}{logical indicating whether to print the computational time for each step. If \code{trace=TRUE}, the computation time for each step is printed. Default is \code{FALSE}.}
}
\value{
\itemize{
  \item \code{ALD}: list of ALD criterion computed at each point of \code{Xcand} at each level if \code{optim=FALSE}. If \code{optim=TRUE}, \code{ALD} returns \code{NULL}.
  \item \code{cost}: a copy of \code{cost}.
  \item \code{Xcand}: a copy of \code{Xcand}.
  \item \code{chosen}: list of chosen level and point.
  \item \code{time}: a scalar of the time for the computation.
}
}
\description{
The function acquires the new point by the Active learning Decomposition (ALD) criterion.
It calculates the ALD criterion \eqn{\frac{V_l(\bm{x})}{\sum^l_{j=1}C_j}},
where \eqn{V_l(\bm{x})} is the contribution of GP emulator
at each fidelity level \eqn{l} and \eqn{C_j} is the simulation cost at level \eqn{j}.
For details, see Heo and Sung (2024, <\doi{https://doi.org/10.1080/00401706.2024.2376173}>).

A new point is acquired on \code{Xcand}. If \code{Xcand=NULL}, a new point is acquired on unit hypercube \eqn{[0,1]^d}.
}
\examples{
\donttest{
library(lhs)
library(doParallel)
library(foreach)

### simulation costs ###
cost <- c(1, 3)

### 1-d Perdikaris function in Perdikaris, et al. (2017) ###
# low-fidelity function
f1 <- function(x) {
  sin(8 * pi * x)
}

# high-fidelity function
f2 <- function(x) {
  (x - sqrt(2)) * (sin(8 * pi * x))^2
}

### training data ###
n1 <- 13
n2 <- 8

### fix seed to reproduce the result ###
set.seed(1)

### generate initial nested design ###
X <- NestedX(c(n1, n2), 1)
X1 <- X[[1]]
X2 <- X[[2]]

### n1 and n2 might be changed from NestedX ###
### assign n1 and n2 again ###
n1 <- nrow(X1)
n2 <- nrow(X2)

y1 <- f1(X1)
y2 <- f2(X2)

### n=100 uniform test data ###
x <- seq(0, 1, length.out = 100)

### fit an RNAmf ###
fit.RNAmf <- RNAmf_two_level(X1, y1, X2, y2, kernel = "sqex")

### predict ###
predy <- predict(fit.RNAmf, x)$mu
predsig2 <- predict(fit.RNAmf, x)$sig2

### active learning with optim=TRUE ###
ald.RNAmf.optim <- ALD_RNAmf(
  Xcand = x, fit.RNAmf, cost = cost,
  optim = TRUE, parallel = TRUE, ncore = 2
)
print(ald.RNAmf.optim$time) # computation time of optim=TRUE

### active learning with optim=FALSE ###
ald.RNAmf <- ALD_RNAmf(
  Xcand = x, fit.RNAmf, cost = cost,
  optim = FALSE, parallel = TRUE, ncore = 2
)
print(ald.RNAmf$time) # computation time of optim=FALSE

### visualize ALD ###
oldpar <- par(mfrow = c(1, 2))
plot(x, ald.RNAmf$ALD$ALD1,
  type = "l", lty = 2,
  xlab = "x", ylab = "ALD criterion at the low-fidelity level",
  ylim = c(min(c(ald.RNAmf$ALD$ALD1, ald.RNAmf$ALD$ALD2)),
           max(c(ald.RNAmf$ALD$ALD1, ald.RNAmf$ALD$ALD2)))
)
points(ald.RNAmf$chosen$Xnext,
  ald.RNAmf$ALD$ALD1[which(x == drop(ald.RNAmf$chosen$Xnext))],
  pch = 16, cex = 1, col = "red"
)
plot(x, ald.RNAmf$ALD$ALD2,
  type = "l", lty = 2,
  xlab = "x", ylab = "ALD criterion at the high-fidelity level",
  ylim = c(min(c(ald.RNAmf$ALD$ALD1, ald.RNAmf$ALD$ALD2)),
           max(c(ald.RNAmf$ALD$ALD1, ald.RNAmf$ALD$ALD2)))
)
par(oldpar)}

}
