#' @title Creates gray-level run length matrix from RIA image
#' @export
#'
#' @description  Creates gray-level run length matrix (GLRLM) from \emph{RIA_image}.
#' GLRLM assesses the spatial relation of voxels to each other by investigating how many times
#' same value voxels occur next to each other in a given direction. By default the \emph{$modif}
#' image will be used to calculate GLRLMs. If \emph{use_slot} is given, then the data
#' present in \emph{RIA_image$use_slot} will be used for calculations.
#' Results will be saved into the \emph{glrlm} slot. The name of the subslot is determined
#' by the supplied string in \emph{save_name}, or is automatically generated by RIA. \emph{right},
#' \emph{down} and \emph{forward} logicals are used to indicate the direction of the runs.
#'
#' @param RIA_data_in \emph{RIA_image}, created by \code{\link[RIA]{load_dicom}}.
#'
#' @param right logical, indicating whether the direction should go to the right.
#'
#' @param down logical, indicating whether the direction should go down.
#'
#' @param forward logical, indicating whether the direction should in the cross plane.
#'
#' @param use_type string, currently only "single" data processing is supported.
#'
#' @param use_orig logical, indicating to use image present in \emph{RIA_data$orig}.
#' If FALSE, the modified image will be used stored in \emph{RIA_data$modif}.
#'
#' @param use_slot string, name of slot where data wished to be used is. Use if the desired image
#' is not in the \emph{data$orig} or \emph{data$modif} slot of the \emph{RIA_image}. For example,
#' ig the desired dataset is in \emph{RIA_image$dichotomized$ep_4}, then \emph{use_slot} should be
#' \emph{dichotomized$ep_4}. The results are automatically saved. If the results are not saved to
#' the desired slot, then please use \emph{save_name} parameter.
#'
#' @param save_name string, indicating the name of subslot of \emph{$glcm} to save results to.
#' If left empty, then it will be automatically determined based on the
#' last entry of \emph{RIA_image$log$events}.
#'
#' @param verbose_in logical indicating whether to print detailed information.
#' Most prints can also be suppressed using the \code{\link{suppressMessages}} function.
#'
#' @return \emph{RIA_image} containing the GLRLM.
#'
#' @examples \dontrun{
#' #Dichotomize loaded image and then calculate GLRLM matrix of RIA_image$modif
#' RIA_image <- dichotomize(RIA_image, bins_in = c(4, 8), equal_prob = TRUE,
#' use_orig = TRUE, write_orig = FALSE)
#' RIA_image <- glrlm(RIA_image, use_orig = FALSE, verbose_in = TRUE)
#'
#' #Use use_slot parameter to set which image to use
#' RIA_image <- glrlm(RIA_image, use_orig = FALSE, use_slot = "dichotomized$ep_4",
#' right = TRUE, down = TRUE, forward = FALSE)
#' }

glrlm <- function(RIA_data_in, right = TRUE, down = FALSE, forward = FALSE, use_type = "single", use_orig = FALSE, use_slot = NULL, save_name = NULL, verbose_in = TRUE)
{
  data_in_orig <- check_data_in(RIA_data_in, use_type = use_type, use_orig = use_orig, verbose_in = verbose_in)

  if(any(class(data_in_orig) != "list")) data_in_orig <- list(data_in_orig)
  list_names <- names(data_in_orig)

  for (k in 1: length(data_in_orig))
  {
    data_in <-  data_in_orig[[k]]

    if(forward & dim(data_in)[3] == 1) {{stop("WARNING: CANNOT ASSESS Z PLANE OFFSET IF DATA IS 2D!")}}

    data_NA <- as.vector(data_in)
    data_NA <- data_NA[!is.na(data_NA)]
    if(length(data_NA) == 0) {stop("WARNING: SUPPLIED RIA_image DOES NOT CONTAIN ANY DATA!!!")}
    if(length(dim(data_in)) < 2 | length(dim(data_in)) > 3) stop(paste0("DATA LOADED IS ", length(dim(data_in)), " DIMENSIONAL. ONLY 2D AND 3D DATA ARE SUPPORTED!"))


    dim_x <- dim(data_in)[1]
    dim_y <- dim(data_in)[2]
    dim_z <- ifelse(!is.na(dim(data_in)[3]), dim(data_in)[3], 1)

    base_m <- array(NA, dim = c(dim_x+dim_x*down, dim_y+dim_y*right, dim_z+dim_z*forward))
    offset <- array(c(dim_x*down, dim_y*right, dim_z*forward)); offset[offset == 0] <- NA; offset <- min(offset, na.rm = TRUE)

    base_m[1:dim_x, 1:dim_y, 1:dim_z] <- data_in

    data_v <- as.vector(data_in)
    data_v <- data_v[!is.na(data_v)]
    gray_levels <- length(unique(data_v))

    glrlm <- array(NA, c(gray_levels, offset))

    for (i in 1: gray_levels) {

      base_filt_m <- data_in; base_filt_m[base_filt_m != i] <- NA; base_filt_m[base_filt_m == i] <- 1
      base_filt_change_m <- array(NA, dim = c(dim_x+dim_x*down, dim_y+dim_y*right, dim_z+dim_z*forward)); base_filt_change_m[1:dim_x, 1:dim_y, 1:dim_z] <- base_filt_m

      for (j in 1: (offset-1)) {
        shift_m <- array(NA, dim = c(dim_x+dim_x*down, dim_y+dim_y*right, dim_z+dim_z*forward))
        shift_m[(1+j*down):(dim_x+j*down), (1+j*right):(dim_y+j*right), (1+j*forward):(dim_z+j*forward)] <- base_filt_m

        diff_m <- base_filt_change_m - shift_m
        count_diff <- length(diff_m[!is.na(diff_m)])
        glrlm[i,(j+1)] <- count_diff

        base_filt_change_m <- diff_m
      }

      count_gl <- base_filt_m; count_gl <- count_gl[!is.na(count_gl)]; count_gl <- sum(count_gl, na.rm = TRUE)

      for (p in seq(dim(glrlm)[2], 3, -1)) {
        if(glrlm[i, p] > 0) {

          m = 2
          for (q in seq((p-1), 2, -1)) {
            glrlm[i, q] <- glrlm[i, q] - glrlm[i, p]*m
            m = m+1
          }
        }
      }

      glrlm_r_sum <- sum(col(glrlm)[1,]*glrlm[i,], na.rm = TRUE)
      if(glrlm_r_sum != count_gl) {glrlm[i,1] <- (count_gl-glrlm_r_sum)
      } else {glrlm[i,1] <- 0}

    }

  if(use_type == "single") {

    if(any(class(RIA_data_in) == "RIA_image") )
    {
      if(is.null(save_name)) {
        txt <- automatic_name(RIA_data_in, use_orig, use_slot)
        txt <- paste0(txt, "_", as.numeric(right), as.numeric(down), as.numeric(forward))

        RIA_data_in$glrlm[[txt]] <- glrlm

        }
      if(!is.null(save_name)) {RIA_data_in$glrlm[[save_name]] <- glrlm
        }
      }
    }

    if(is.null(save_name)) {txt_name <- txt
    } else {txt_name <- save_name}
    if(verbose_in) {message(" "); message(paste0("GLRLM WAS SUCCESSFULLY ADDED TO '", txt_name, "' SLOT OF RIA_image$glcm")); message(" ") }


  }

  if(any(class(RIA_data_in) == "RIA_image")) {return(RIA_data_in)
  } else {return(glrlm)}


}

