#' @title
#' Extract a single tibble from a REDCapTidieR supertibble
#'
#' @description
#' Take a supertibble generated with \code{read_redcap_tidy()}
#' and return one of its tibbles.
#'
#' @details
#' This function makes it easy to extract a single instrument's data from a
#' REDCapTidieR supertibble.
#'
#' @returns A \code{tibble}
#'
#' @param .data A tibble generated by \code{read_redcap_tidy()}. Required.
#' @param tbl The \code{form_name} to extract. Required.
#'
#' @importFrom checkmate assert_character
#' @importFrom rlang .data
#'
#' @examples
#' # Supply a `read_redcap_tidy()` output to for table extraction
#' sample_data <- tibble::tribble(
#'   ~redcap_form_name,    ~redcap_data,   ~structure,
#'   "super_hero_powers",   list(),         "repeating",
#'   "heroes_information",  list(),         "nonrepeating"
#' )
#'
#'
#' extract_table(sample_data, "heroes_information")
#'
#' @export

extract_table <- function(.data,
                          tbl) {
  # Check tbl is valid ----
  assert_character(all_of(tbl))

  if (length(all_of(tbl)) > 1) {
    stop("Only one table may be supplied.")
  }

  # Extract specified table ----
  out <- extract_tables(.data, tbls = all_of(tbl))[[1]]

  out
}


#' Extract one or more tibbles from a REDCapTidieR supertibble
#'
#' @description
#' Take a supertibble generated with \code{read_redcap_tidy()}
#' and return a named list of tibbles.
#'
#' @details
#' This function makes it easy to extract a multiple instrument's data from a
#' REDCapTidieR supertibble into a named list. Specifying instruments using
#' \code{tidyselect} semantics such as \code{starts_with()}
#' or \code{ends_with()} is supported.
#'
#' @returns A named list of \code{tibble}s
#'
#' @param .data A tibble generated by \code{read_redcap_tidy()}. Required.
#' @param tbls A vector of \code{form_name}s or a tidy selector. Default is
#' \code{everything()}.
#'
#' @importFrom rlang .data enquo
#' @importFrom dplyr select %>%
#' @importFrom tidyselect eval_select everything
#' @importFrom tidyr pivot_wider
#' @importFrom purrr map pluck
#'
#' @examples
#' # Supply a `read_redcap_tidy()` output to for table extraction
#'
#' # Default `everything()`
#' sample_data <- tibble::tribble(
#'    ~redcap_form_name,    ~redcap_data, ~structure,
#'    "super_hero_powers",   list(),       "repeating",
#'    "heroes_information",  list(),       "nonrepeating"
#' )
#'
#' extract_tables(sample_data)
#'
#' extract_tables(sample_data, starts_with("heroes"))
#'
#' @export

extract_tables <- function(.data,
                           tbls = everything()) {

  # Extract specified table ----
  # Pass tbls as an expression for enquosure
  tbls <- enquo(tbls)

  out <- .data %>%
    select(-.data$structure) %>%
    pivot_wider(names_from = .data$redcap_form_name,
                values_from = .data$redcap_data)

  out <- out[eval_select(tbls, data = out)]

  out %>%
    map(.f = ~pluck(.)[[1]])
}
