% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RAMP.R
\name{RAMP}
\alias{RAMP}
\title{Regularization Algorithm under Marginality Principle (RAMP) for high
dimensional generalized quadratic regression.}
\usage{
RAMP(X, y, family = "gaussian", penalty = "LASSO", gamma = 3.7,
  inter = TRUE, hier = "Strong", eps = 1e-15, tune = "EBIC",
  penalty.factor = rep(1, ncol(X)), inter.penalty.factor = 1, lam.list,
  lambda.min.ratio, max.iter = 100, max.num, n.lambda = 100,
  ebic.gamma = 1, refit = TRUE, trace = FALSE)
}
\arguments{
\item{X}{input matrix, of dimension nobs x nvars; each row is an observation
vector.}

\item{y}{response variable, of dimension nobs x 1. continous for
\code{family='gaussian'}, binary for \code{family='binomial'}.}

\item{family}{response type. Default is 'gaussian'. The other choice is 'binomial'
for logistic regression.}

\item{penalty}{Choose from \code{LASSO}, \code{SCAD} and \code{MCP}. Default
is 'LASSO'.}

\item{gamma}{concavity parameter. If missing, the code will use 3.7 for
'SCAD' and 2.7 for 'MCP'.}

\item{inter}{whether to select interaction effects. Default is TRUE.}

\item{hier}{whether to enforce strong or weak heredity. Default is 'Strong'.}

\item{eps}{the precision used to test the convergence. Default is 1e-15.}

\item{tune}{tuning parameter selection method.
'AIC', 'BIC', 'EBIC' and 'GIC' are available options. Default is EBIC.}

\item{penalty.factor}{A multiplicative factor for the penalty applied to each coefficient. If supplied, penalty.factor must be a numeric vector of length equal to the number of columns of X. The purpose of penalty.factor is to apply differential penalization if some coefficients are thought to be more likely than others to be in the model. In particular, penalty.factor can be 0, in which case the coefficient is always in the model without shrinkage.}

\item{inter.penalty.factor}{the penalty factor for interactions effects. Default is 1. 
larger value discourage interaction effects.}

\item{lam.list}{a user supplied \eqn{\lambda} sequence.
typical usage is to have the program compute its own
\code{lambda} sequence based on \code{lambda.min.ratio} and \code{n.lambda}.
 supplying a value of \eqn{\lambda} overrides this.}

\item{lambda.min.ratio}{optional input. smallest value for \code{lambda}, as
a fraction of \code{max.lam}, the (data derived) entry value. the default
depends on the sample size \code{n} relative to the number of variables
\code{p}. if \code{n} > \code{p}, the default is 0.0001. otherwise, the
default is 0.01.}

\item{max.iter}{maximum number of iteration in the computation. Default is
100.}

\item{max.num}{optional input. maximum number of nonzero coefficients.}

\item{n.lambda}{the number of \code{lambda} values. Default is 100.}

\item{ebic.gamma}{the gamma parameter value in the EBIC criteria. Default is
1.}

\item{refit}{whether to perform a MLE refit on the selected model. Default
is TRUE.}

\item{trace}{whether to trace the fitting process. Default is FALSE.}
}
\value{
An object with S3 class RAMP.
\item{a0}{intercept vector of length(\code{lambda}).}
\item{mainInd}{index for the selected main effects.}
\item{interInd}{index for the selected interaction effects}
\item{beta.m}{coefficients for the selected main effects.}
\item{beta.i}{coefficients for the selected interaction effects.}
}
\description{
Regularization Algorithm under Marginality Principle (RAMP) for high
dimensional generalized quadratic regression.
}
\examples{
set.seed(0)
n = 500
p = 10 #Can be changed to a much larger number say 50000
x = matrix(rnorm(n*p),n,p)
eta = 1 * x[,1] + 2 * x[,3]  + 3*x[,6]  + 4*x[,1]*x[,3] + 5*x[,1]*x[,6]
y =  eta + rnorm(n)
xtest = matrix(rnorm(n*p),n,p)
eta.test = 1 * xtest[,1] + 2 * xtest[,3]  + 3*xtest[,6] +
4*xtest[,1]*xtest[,3] + 5*xtest[,1]*xtest[,6]
ytest =  eta.test + rnorm(n)
fit1 = RAMP(x, y)
fit1    ###examine the results
ypred = predict(fit1, xtest)
mean((ypred-ytest)^2)

#fit1.scad = RAMP(x, y, penalty = 'SCAD')
#fit1.scad    ###examine the results

#fit1.mcp = RAMP(x, y, penalty = 'MCP')
#fit1.mcp    ###examine the results

##Now, try a binary response
#y = rbinom(n, 1, 1/(1+exp(-eta)))
#fit2 = RAMP(x, y, family='binomial')  ###for binary response

## Weak heredity
eta = 1 * x[,1] + 3*x[,6]  + 4*x[,1]*x[,3] + 5*x[,1]*x[,6]
y =  eta + rnorm(n)
eta.test = 1 * xtest[,1] +  3*xtest[,6] + 4*xtest[,1]*xtest[,3] +
5*xtest[,1]*xtest[,6]
ytest =  eta.test + rnorm(n)

fit3 = RAMP(x, y, hier = 'Strong')
fit3    ###examine the results
ypred3 = predict(fit3, xtest)
mean((ypred3-ytest)^2)
fit4 = RAMP(x, y, hier = 'Weak')
fit4
ypred4 = predict(fit4, xtest)
mean((ypred4-ytest)^2)
}
\seealso{
\code{\link{predict.RAMP}},\code{\link{print.RAMP}}
}

