% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fmx.R
\name{fmx}
\alias{fmx}
\alias{dfmx}
\alias{pfmx}
\alias{qfmx}
\alias{rfmx}
\title{The Finite Mixture Distribution}
\usage{
fmx(distname, w = 1, ...)

dfmx(
  x,
  dist,
  distname = dist@distname,
  K = dim(parM)[1L],
  parM = dist@parM,
  w = dist@w,
  ...,
  log = FALSE
)

pfmx(
  q,
  dist,
  distname = dist@distname,
  K = dim(parM)[1L],
  parM = dist@parM,
  w = dist@w,
  ...,
  lower.tail = TRUE,
  log.p = FALSE
)

qfmx(
  p,
  dist,
  distname = dist@distname,
  K = dim(parM)[1L],
  parM = dist@parM,
  w = dist@w,
  interval = qfmx_interval(dist = dist),
  ...,
  lower.tail = TRUE,
  log.p = FALSE
)

rfmx(
  n,
  dist,
  distname = dist@distname,
  K = dim(parM)[1L],
  parM = dist@parM,
  w = dist@w
)
}
\arguments{
\item{distname, K, parM, w}{auxiliary parameters, whose default values are determined by argument \code{dist}, 
see \code{\link{fmx-class}} for details.
The user-specified vector of \code{w} does not need to sum up to 1; \code{w/sum(w)} will be used internally.}

\item{...}{in \code{\link{fmx}}, these are the vectors of distribution parameters}

\item{x, q}{vector of quantiles.}

\item{dist}{\code{'fmx'} object, representing a finite mixture distribution}

\item{log, log.p}{logical; if \code{TRUE}, probabilities \eqn{p} are given as \eqn{log(p)}.}

\item{lower.tail}{logical; if \code{TRUE} (default), probabilities are \eqn{Pr(X\le x)} otherwise, \eqn{Pr(X>x)}.}

\item{p}{vector of probabilities.}

\item{interval}{interval for root finding (see \code{\link[rstpm2]{vuniroot}})}

\item{n}{number of observations.}
}
\value{
\code{\link{fmx}} returns an S4 \code{'fmx'} object which specifies the parameters of a finite mixture distribution.
See \code{\link{fmx-class}} for details.

\code{\link{dfmx}} returns a vector of probability density values of an S4 \code{'fmx'} object at specified quantiles \code{x}.

\code{\link{pfmx}} returns a vector of cumulative probability values of an S4 \code{'fmx'} object at specified quantiles \code{q}.

\code{\link{qfmx}} returns a vector of quantiles of an S4 \code{'fmx'} object, based on specified cumulative probabilities \code{p}.

\code{\link{rfmx}} generates random deviates of an S4 \code{'fmx'} object.
}
\description{
Density function, distribution function, quantile function and random generation for a finite mixture distribution 
with normal or Tukey's \eqn{g}-&-\eqn{h} components.
}
\details{
\code{\link{fmx}} creates an S4 \code{'fmx'} object which specifies the parameters of a finite mixture distribution.
See \code{\link{fmx-class}} for details.

\code{\link{dfmx}} gives the density, which is required in plotting and calculating
the log-likelihood. 

\code{\link{pfmx}} gives the distribution function.

\code{\link{qfmx}} gives the quantile function, by solving \code{\link{pfmx}} by \code{\link[rstpm2]{vuniroot}}.
One major challenge when dealing with the finite mixture of Tukey's \eqn{g}-&-\eqn{h} family distribution
is that Brent–Dekker's method needs to be performed in both \code{\link{pGH}} and \code{\link{qfmx}}, 
i.e. 'two layers' of root-finding algorithm.

\code{\link{rfmx}} generates random deviates.
}
\examples{

x = (-3):7

(e1 = fmx('norm', mean = c(0,3), sd = c(1,1.3), w = c(1, 1)))
# 'fmx' object `e1`, a two-component 50\%-50\% mixture of normal
isS4(e1) # 'fmx' is S4 object
slotNames(e1) # 'slot' is similar to S3 object component 
e1@parM # component parameters
plot(e1) # using vanilla R
autoplot(e1) # using ggplot2 package
hist(rfmx(n = 1e3L, dist = e1), main = '1000 obs from e1')
# generate a sample of size 1e3L from mixture distribution `e1`
round(dfmx(x, dist = e1), digits = 3L)
round(p1 <- pfmx(x, dist = e1), digits = 3L)
stopifnot(all.equal.numeric(qfmx(p1, dist = e1), x, tol = 1e-4))

(e2 = fmx('GH', A = c(0,3), g = c(.2, .3), h = c(.2, .1), w = c(2, 3)))
# 'fmx' object `e2`, a two-component 40\%-60\% mixture of Tukey
hist(rfmx(n = 1e3L, dist = e2), main = '1000 obs from e2') 
round(dfmx(x, dist = e2), digits = 3L)
round(p2 <- pfmx(x, dist = e2), digits = 3L)
stopifnot(all.equal.numeric(qfmx(p2, dist = e2), x, tol = 1e-4))

(e3 = fmx('GH', A = 0, g = .2, h = .2)) # one-component Tukey
hist(rfmx(1e3L, dist = e3))
hist(rGH(n = 1e3L, A = 0, g = .2, h = .2))
# identical (up to random seed); but ?rfmx has much cleaner code
round(dfmx(x, dist = e3), digits = 3L)
round(p3 <- pfmx(x, dist = e3), digits = 3L)
stopifnot(all.equal.numeric(qfmx(p3, dist = e3), x, tol = 1e-4))

}
