% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getPrevalence.R
\name{getPrevalence}
\alias{getPrevalence}
\title{Predicting Prevalence from a Mixed or Fixed Effect Logistic Regression with
Presence/Absence Tests on Pooled Samples}
\usage{
getPrevalence(
  model,
  newdata = NULL,
  re.form = NULL,
  robust = FALSE,
  level = 0.95
)
}
\arguments{
\item{model}{An object returned by [PoolReg()] or [PoolRegBayes()]}

\item{newdata}{The data for which prevalence needs to be estimated/predicted.
If not provided, defaults to using the data used to train the model (i.e.
returns the fitted values of the prevalence)}

\item{re.form}{A description of which random effects to include in the
prediction. If omitted, getPrevalence automatically tests to see if there
are any random effect terms. If not, it just returns the estimates based on
population effects. If there are random effects, it tests to see if the
random effect variables form a nested hierarchical structure. If so, in
addition to the estimates based on population effects only, it will
estimate at different levels of the nested hierarchical structure in order
of increasing granularity. For manual control you can set to NA for
population effects only, or a one-sided formula specifying the form of the
random effects to include in estimates, or a list of such objects.}

\item{robust}{Currently only relevant for brmsfit objects (returned by
PoolRegBayes). If \code{FALSE} (default) the point estimate of prevalence
is the mean over the posterior. If \code{TRUE}, the median over the
posterior is used instead.}

\item{level}{Defines the confidence level to be used for the confidence and
credible intervals. Defaults to 0.95 (i.e. 95\% intervals).}
}
\value{
A \code{list} with at least one field \code{PopulationEffects} and an
  additional field for every random/group effect variable. The field
  \code{PopulationEffects} contains a \code{data.frame} with the prevalence
  estimated based only the fixed/population effects. When the intercept is
  the only fixed/population effect, this is just the population mean
  (possibly adjusted for random/group effects). When there are group effects
  terms, \code{getPrevalence} attempts to order these with respect to
  'granularity' and extract the prevalence estimates for these random
  effects; e.g. if the random/group effects included are there to account for
  a hierarchical sampling frame with levels 'Village' and 'Site' with a
  formula like \code{Result ~ Cov1 + Cov2 + (1|Village/Site)}, then
  getPrevalence will be a list of three data frames: estimates for every
  combination of covariates, estimates for every combination of covariates
  and village, and estimates for every combination of covariates, village,
  and site.
}
\description{
This function works somewhat like a \code{predict} or \code{fitted} generic
function returning the model predicted prevalence for a given set of data;
however, as the quantity of interest (prevalence) is neither on the response
or link scale we do not use either of these generic functions. Further, when
the model accounts for the hierarchical structure of the sampling frame (e.g.
Region/Village/Site), it is common to want to know the predicted values at
each level of sampling (e.g. Prevalence at each region, village or site) so
these are calculated automatically.
}
\examples{
# Perform logistic-type regression modelling for a synthetic dataset consisting
# of pools (sizes 1, 5, or 10) taken from 4 different regions and 3 different
# years. Within each region specimens are collected at 4 different villages,
# and within each village specimens are collected at 8 different sites.


### Models in a frequentist framework
#ignoring hierarchical sampling frame within each region
Mod <- PoolReg(Result ~ Region + Year,
               data = SimpleExampleData,
               poolSize = NumInPool)
summary(Mod)

#accounting hierarchical sampling frame within each region
HierMod <- PoolReg(Result ~ Region + Year + (1|Village/Site),
                   data = SimpleExampleData,
                   poolSize = NumInPool)
summary(HierMod)
#Extract fitted prevalence for each combination of region and year and then at
#each level of the hierarchical sampling frame (i.e. for each village in each
#region and  each site in each village)
getPrevalence(HierMod)


### Models in a Bayesian framework with default (non-informative) priors
#ignoring hierarchical sampling frame within each region
\donttest{
  BayesMod <- PoolRegBayes(Result ~ Region + Year,
                           data = SimpleExampleData,
                           poolSize = NumInPool)
  summary(BayesMod)
  getPrevalence(BayesMod) #Extract fitted prevalence for each combination of region and year

  #accounting hierarchical sampling frame within each region
  BayesHierMod <- PoolRegBayes(Result ~ Region + Year + (1|Village/Site),
                               data = SimpleExampleData,
                               poolSize = NumInPool)
  summary(BayesHierMod)
  getPrevalence(BayesHierMod)
}

### Calculate adjusted estimates of prevalence
# We use the same function for all four models, but the outputs are slightly different

# Extract fitted prevalence for each combination of region and year
getPrevalence(Mod)
\donttest{
  getPrevalence(BayesMod)
}

#Extract fitted prevalence for each combination of region and year and then at
#each level of the hierarchical sampling frame (i.e. for each village in each
#region and  each site in each village)
getPrevalence(HierMod)
\donttest{
  getPrevalence(BayesHierMod)
}

# You can also use getPrevalence to predict at prevalence for other values of
# the covariates (e.g. predict prevalence in year 4)

#Making a data frame containing data make predict on
DataFuture <- unique(data.frame(Region = SimpleExampleData$Region,
                                Village = SimpleExampleData$Village,
                                Site = SimpleExampleData$Site,
                                Year = 4))

getPrevalence(Mod, newdata = DataFuture)
getPrevalence(HierMod, newdata = DataFuture)
\donttest{
  getPrevalence(BayesMod, newdata = DataFuture)
  getPrevalence(BayesHierMod, newdata = DataFuture)
}
}
\seealso{
\code{\link{PoolReg}},
   \code{\link{PoolRegBayes}}
}
