\name{plot_scatter}

\alias{plot_scatter}

\title{2D Visualization of Data Scatter and Box Vertices}

\description{
  S3 generic function for plotting the cross-validated box vertices of a \code{PRSP} object.
  Plot the data scatter and cross-validated box vertices in a plane 
  at a given peeling step of the peeling sequence (inner loop of our PRSP algorithm).
}

\usage{
  plot_scatter(object,
               main = NULL,
               proj = c(1,2), 
               splom = TRUE, 
               boxes = FALSE,
               steps = object$cvfit$cv.nsteps,
               pch = 16, 
               cex = 0.5, 
               col = 2:(length(steps)+1), 
               col.box = 2:(length(steps)+1), 
               lty.box = rep(2,length(steps)), 
               lwd.box = rep(1,length(steps)),
               add.legend = TRUE, 
               device = NULL, 
               file = "Scatter Plot", 
               path=getwd(), 
               horizontal = FALSE, 
               width = 5, 
               height = 5, ...)
}

\arguments{
  \item{object}{
    Object of class \code{PRSP} as generated by the main function \code{\link[PRIMsrc]{sbh}}.
  }
  \item{main}{
    \code{Character} \code{vector}. Main Title. Defaults to \code{NULL}.
  }
  \item{proj}{
    \code{Integer} \code{vector} of length two, specifying the two dimensions of the projection plane. Defaults to c(1,2).
  }
  \item{splom}{
    \code{Logical} scalar. Shall the scatter plot of points inside the box(es) be plotted? Default to \code{TRUE}.
  }
  \item{boxes}{
    \code{Logical} scalar. Shall the box vertices be plotted or just the scatter of points? Default to \code{FALSE}.
  }
  \item{steps}{
    \code{Integer} \code{vector}. Vector of peeling steps at which to plot the in-box samples and box vertices. 
    Defaults to the last peeling step of \code{PRSP} object \code{object}.
  }
  \item{pch}{
    \code{Integer} scalar of symbol number for the scatter plot. Defaults to 16.
  }
  \item{cex}{
    \code{Integer} scalar of symbol expansion. Defaults to 0.5.
  }
  \item{col}{
    \code{Integer} \code{vector} specifying the symbol color for each step. 
    Defaults to vector of colors of length the number of steps. 
    The vector is reused cyclically if it is shorter than the number of steps.
  }
  \item{col.box}{
    \code{Integer} \code{vector} of line color of box vertices for each step. 
    Defaults to vector of colors of length the number of steps. 
    The vector is reused cyclically if it is shorter than the number of steps.
  }
  \item{lty.box}{
    \code{Integer} \code{vector} of line type of box vertices for each step. 
    Defaults to vector of 2's of length the number of steps. 
    The vector is reused cyclically if it is shorter than the number of steps.
  }
  \item{lwd.box}{
    \code{Integer} \code{vector} of line width of box vertices for each step. 
    Defaults to vector of 1's of length the number of steps. 
    The vector is reused cyclically if it is shorter than the number of steps.
  }
  \item{add.legend}{
    \code{Logical} scalar. Shall the legend of steps numbers be plotted? Defaults to \code{TRUE}.
  }
  \item{device}{
    Graphic display device in \{NULL, "PS", "PDF"\}. Defaults to NULL (standard output screen).
    Currently implemented graphic display devices are "PS" (Postscript) or "PDF" (Portable Document Format).
  }
  \item{file}{
    File name for output graphic. Defaults to "Scatter Plot".
  }
  \item{path}{
    Absolute path (without final (back)slash separator). Defaults to working directory path.
  }
  \item{horizontal}{
    \code{Logical} scalar. Orientation of the printed image. Defaults to \code{FALSE}, that is potrait orientation.
  }
  \item{width}{
    \code{Numeric} scalar. Width of the graphics region in inches. Defaults to 5.
  }
  \item{height}{
    \code{Numeric} scalar. Height of the graphics region in inches. Defaults to 5.
  }
  \item{\dots}{
    Generic arguments passed to other plotting functions.
  }
}

\details{
  The scatterplot is drawn on a graphical device with geometrically equal scales on the \eqn{X} and \eqn{Y} axes.
}

\value{
  Invisible. None. Displays the plot(s) on the specified \code{device}.
}

\references{
    \itemize{
        \item Dazard J-E., Choe M., LeBlanc M. and Rao J.S. (2015).
              "\emph{Cross-validation and Peeling Strategies for Survival Bump Hunting using Recursive Peeling Methods.}"
              (Submitted).
        \item Dazard J-E., Choe M., LeBlanc M. and Rao J.S. (2014).
              "\emph{Cross-Validation of Survival Bump Hunting by Recursive Peeling Methods.}"
              In JSM Proceedings, Survival Methods for Risk Estimation/Prediction Section. Boston, MA, USA.
              American Statistical Association IMS - JSM, p. 3366-3380.
        \item Dazard J-E. and J. S. Rao (2010).
              "\emph{Local Sparse Bump Hunting.}"
              J. Comp Graph. Statistics, 19(4):900-92.
    }
}

\author{
    \itemize{
        \item "Jean-Eudes Dazard, Ph.D." \email{jxd101@case.edu}
        \item "Michael Choe, M.D." \email{mjc206@case.edu}
        \item "Michael LeBlanc, Ph.D." \email{mleblanc@fhcrc.org}
        \item "Alberto Santana, MBA." \email{ahs4@case.edu}
    }
    Maintainer: "Jean-Eudes Dazard, Ph.D." \email{jxd101@case.edu}

    Acknowledgments: This project was partially funded by the National Institutes of Health
    NIH - National Cancer Institute (R01-CA160593) to J-E. Dazard and J.S. Rao.
}

\note{
  End-user plotting function.
}

\keyword{Exploratory Survival/Risk Analysis}
\keyword{Survival/Risk Estimation & Prediction}
\keyword{Non-Parametric Method}
\keyword{Cross-Validation}
\keyword{Bump Hunting}
\keyword{Rule-Induction Method}

\examples{
#===================================================
# Loading the library and its dependencies
#===================================================
library("PRIMsrc")

#=================================================================================
# Simulated dataset #1 (n=250, p=3)
# Replicated Combined Cross-Validation (RCCV)
# Peeling criterion = LRT
# Optimization criterion = LRT
#=================================================================================
CVCOMBREP.synt1 <- sbh(dataset = Synthetic.1, 
                       cvtype = "combined", cvcriterion = "lrt",
                       B = 1, K = 5, 
                       vs = TRUE, cpv = FALSE, probval = 0.5, 
                       arg = "beta=0.05,
                              alpha=0.1,
                              minn=10,
                              L=NULL,
                              peelcriterion=\"lr\"",
                       parallel = FALSE, conf = NULL, seed = 123)

plot_scatter(object = CVCOMBREP.synt1,
             main = paste("Scatter plot for model #1", sep=""),
             proj = c(1,2), splom = TRUE, boxes = TRUE,
             steps = CVCOMBREP.synt1$cvfit$cv.nsteps,
             pch = 16, cex = 0.5, col = 2,
             col.box = 2, lty.box = 2, lwd.box = 1,
             add.legend = TRUE, 
             device = NULL, file = "Scatter Plot", path=getwd(),
             horizontal = FALSE, width = 5.0, height = 5.0)
}
