% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kruskalTest.R
\name{kruskalTest}
\alias{kruskalTest}
\alias{kruskalTest.default}
\alias{kruskalTest.formula}
\title{Kruskal-Wallis Rank Sum Test}
\usage{
kruskalTest(x, ...)

\method{kruskalTest}{default}(x, g, dist = c("Chisquare", "KruskalWallis", "FDist"), ...)

\method{kruskalTest}{formula}(
  formula,
  data,
  subset,
  na.action,
  dist = c("Chisquare", "KruskalWallis", "FDist"),
  ...
)
}
\arguments{
\item{x}{a numeric vector of data values, or a list of numeric data
vectors.}

\item{\dots}{further arguments to be passed to or from methods.}

\item{g}{a vector or factor object giving the group for the
corresponding elements of \code{"x"}.
Ignored with a warning if \code{"x"} is a list.}

\item{dist}{the test distribution. Defaults's to \code{"Chisquare"}.}

\item{formula}{a formula of the form \code{response ~ group} where
\code{response} gives the data values and \code{group} a vector or
factor of the corresponding groups.}

\item{data}{an optional matrix or data frame (or similar: see
\code{\link{model.frame}}) containing the variables in the
formula \code{formula}.  By default the variables are taken from
\code{environment(formula)}.}

\item{subset}{an optional vector specifying a 
subset of observations to be used.}

\item{na.action}{a function which indicates what should happen when
the data contain \code{NA}s.  Defaults to \code{getOption("na.action")}.}
}
\value{
A list with class \code{"htest"} containing the following components:
\describe{
 \item{method}{a character string indicating what type of test was performed.}
 \item{data.name}{a character string giving the name(s) of the data.}
 \item{statistic}{the estimated quantile of the test statistic.}
 \item{p.value}{the p-value for the test.}
 \item{parameter}{the parameters of the test statistic, if any.}
 \item{alternative}{a character string describing the alternative hypothesis.}
\item{estimates}{the estimates, if any.}
\item{null.value}{the estimate under the null hypothesis, if any.}
}
}
\description{
Performs a Kruskal-Wallis rank sum test.
}
\details{
For one-factorial designs with non-normally distributed
residuals the Kruskal-Wallis rank sum test can be performed to test
the H\eqn{_0: F_1(x) = F_2(x) = \ldots = F_k(x)} against
the H\eqn{_\mathrm{A}: F_i (x) \ne F_j(x)~ (i \ne j)} with at least
one strict inequality.

Let \eqn{R_{ij}} be the joint rank of \eqn{X_{ij}},
with \eqn{R_{(1)(1)} = 1, \ldots, R_{(n)(n)} = N, ~~ N = \sum_{i=1}^k n_i},
The test statistic is calculated as
\deqn{
 H = \sum_{i=1}^k n_i \left(\bar{R}_i - \bar{R}\right) / \sigma_R,
}{%
 SEE PDF
}

with the mean rank of the \eqn{i}-th group
\deqn{
\bar{R}_i = \sum_{j = 1}^{n_{i}} R_{ij} / n_i,
}{%
 SEE PDF
}

the expected value
\deqn{
 \bar{R} = \left(N +1\right) / 2
}{%
 SEE PDF
}

and the expected variance as
\deqn{
 \sigma_R^2 = N \left(N + 1\right) / 12.
}{%
 SEE PDF
}

In case of ties the statistic \eqn{H} is divided by
\eqn{\left(1 - \sum_{i=1}^r t_i^3 - t_i \right) / \left(N^3 - N\right)}

According to Conover and Imam (1981), the statistic \eqn{H} is related
to the \eqn{F}-quantile as
\deqn{
 F = \frac{H / \left(k - 1\right)}
     {\left(N - 1 - H\right) / \left(N - k\right)}
}{%
 SEE PDF
}
which is equivalent to a one-way ANOVA F-test using rank transformed data
(see examples).

The function provides three different \code{dist} for \eqn{p}-value estimation:
\describe{
 \item{Chisquare}{\eqn{p}-values are computed from the \code{\link{Chisquare}}
  distribution with \eqn{v = k - 1} degree of freedom.}
 \item{KruskalWallis}{\eqn{p}-values are computed from the
  \code{\link[SuppDists]{pKruskalWallis}} of the package \pkg{SuppDists}.}
 \item{FDist}{\eqn{p}-values are computed from the \code{\link{FDist}} distribution
  with \eqn{v_1 = k-1, ~ v_2 = N -k} degree of freedom.}
}
}
\examples{
## Hollander & Wolfe (1973), 116.
## Mucociliary efficiency from the rate of removal of dust in normal
## subjects, subjects with obstructive airway disease, and subjects
## with asbestosis.
x <- c(2.9, 3.0, 2.5, 2.6, 3.2) # normal subjects
y <- c(3.8, 2.7, 4.0, 2.4)      # with obstructive airway disease
z <- c(2.8, 3.4, 3.7, 2.2, 2.0) # with asbestosis
g <- factor(x = c(rep(1, length(x)),
                   rep(2, length(y)),
                   rep(3, length(z))),
             labels = c("ns", "oad", "a"))
dat <- data.frame(
   g = g,
   x = c(x, y, z))

## AD-Test
adKSampleTest(x ~ g, data = dat)

## BWS-Test
bwsKSampleTest(x ~ g, data = dat)

## Kruskal-Test
## Using incomplete beta approximation
kruskalTest(x ~ g, dat, dist="KruskalWallis")
## Using chisquare distribution
kruskalTest(x ~ g, dat, dist="Chisquare")

\dontrun{
## Check with kruskal.test from R stats
kruskal.test(x ~ g, dat)
}
## Using Conover's F
kruskalTest(x ~ g, dat, dist="FDist")

\dontrun{
## Check with aov on ranks
anova(aov(rank(x) ~ g, dat))
## Check with oneway.test
oneway.test(rank(x) ~ g, dat, var.equal = TRUE)
}
}
\references{
Conover, W.J., Iman, R.L. (1981) Rank Transformations as a Bridge
 Between Parametric and Nonparametric Statistics.
 \emph{Am Stat} \bold{35}, 124--129.

Kruskal, W.H., Wallis, W.A. (1952) Use of Ranks in One-Criterion Variance Analysis.
 \emph{J Am Stat Assoc} \bold{47}, 583--621.

Sachs, L. (1997) \emph{Angewandte Statistik}. Berlin: Springer.
}
\seealso{
\code{\link{kruskal.test}}, \code{\link[SuppDists]{pKruskalWallis}},
 \code{\link{Chisquare}}, \code{\link{FDist}}
}
\concept{kruskalranks}
\keyword{htest}
