#OUwie Master Controller

#written by Jeremy M. Beaulieu

#Fits the Ornstein-Uhlenbeck model of continuous characters evolving under discrete selective
#regimes. The input is a tree of class "phylo" that has the regimes as internal node labels
#and a trait file. The trait file must be in the following order: Species names, Regime, and
#continuous trait. Different models can be specified -- Brownian motion (BM), multiple rate BM (BMS)
#global OU (OU1), multiple regime OU (OUM), multiple sigmas (OUMV), multiple alphas (OUMA),
#and the multiple alphas and sigmas (OUMVA).

OUwie <- function(phy,data, model=c("BM1","BMS","OU1","OUM","OUMV","OUMA","OUMVA", "TrendyM", "TrendyMS"), simmap.tree=FALSE, root.age=NULL, scaleHeight=FALSE, root.station=FALSE, get.root.theta=FALSE, shift.point=0.5, clade=NULL, mserr="none", starting.vals=NULL, check.identify=TRUE, diagn=FALSE, quiet=FALSE, warn=TRUE, opts = list("algorithm"="NLOPT_LN_SBPLX", "maxeval"="1000", "ftol_rel"=.Machine$double.eps^0.5)){

    if(model=="BMS" & root.station==TRUE){
        warning("By setting root.station=TRUE, you have specified the group means model of Thomas et al. 2006", call.=FALSE, immediate.=TRUE)
        get.root.theta = FALSE
    }

    if(is.factor(data[,3])==TRUE){
        stop("Check the format of the data column. It's reading as a factor.", call. = FALSE)
    }

    #if(is.null(root.age)){
    #    if(any(branching.times(phy)<0)){
    #        stop("Looks like your tree is producing negative branching times. Must input known root age of tree.", call. = FALSE)
    #    }
    #}

    if(!is.null(starting.vals[1])){
        if(model == "OU1" | model == "OUM" | model == "OUMV" | model == "OUMA" | model == "OUMVA"){
            if(length(starting.vals)<2){
                stop("You only supplied one starting value. For OU models you need to supply two", call. = FALSE)
            }
        }
    }
    
    if(model == "OUMV" |  model == "OUMA" |  model == "OUMVA"){
        if(root.station == TRUE){
            stop("Assuming stationarity at the root is no longer allowed under these models. Try OU1 and OUM.", call. = FALSE)
        }
    }

    if(model == "BM1" |  model == "BMS" | model == "TrendyM" | model == "TrendyMS"){
        if(root.station == FALSE){
            get.root.theta = TRUE
        }
    }

    if(check.identify == TRUE){
            identifiable <- check.identify(phy=phy, data=data, simmap.tree=simmap.tree, quiet=TRUE)
            if(identifiable == 0){
                warning("The supplied regime painting may be unidentifiable for the regime painting. All regimes form connected subtrees.", call. = FALSE, immediate.=TRUE)
            }
    }

    #Makes sure the data is in the same order as the tip labels
    if(mserr == "none" | mserr == "est"){
		data<-data.frame(data[,2], data[,3], row.names=data[,1])
		data<-data[phy$tip.label,]
	}
	if(mserr=="known"){
		if(!dim(data)[2]==4){
			stop("You specified measurement error should be incorporated, but this information is missing", call. = FALSE)
		}
		else{
            if(is.factor(data[,4]) == TRUE){
                stop("Check the format of the measurement error column. It's reading as a factor.", call. = FALSE)
            }else{
                data <- data.frame(data[,2], data[,3], data[,4], row.names=data[,1])
                data <- data[phy$tip.label,]
            }
		}
	}

    #Values to be used throughout
	n <- max(phy$edge[,1])
	ntips <- length(phy$tip.label)
	#Will label the clade of interest if the user so chooses:
	if(!is.null(clade) & simmap.tree==FALSE){
		node<-mrca(phy)[clade[1],clade[2]]
		int<-c(node,Descendants(phy,node, "all"))
		tips<-int[int<ntips]
		data[,1]<-1
		data[tips,1]<-2
		int<-int[int>ntips]
		phy$node.label<-rep(1,phy$Nnode)
		pp<-int-length(phy$tip.label)
		phy$node.label[pp]<-2
	}
	if (is.character(model)) {
		if (model == "BM1"| model == "OU1"){
			simmap.tree=FALSE
		}
	}
	if(simmap.tree==TRUE){
		k <- length(colnames(phy$mapped.edge))
		tot.states <- factor(colnames(phy$mapped.edge))
		tip.states <- factor(data[,1])
		data[,1] <- as.numeric(tip.states)

		#Obtains the state at the root
		root.state=which(colnames(phy$mapped.edge)==names(phy$maps[[1]][1]))
		##Begins the construction of the edges matrix -- similar to the ouch format##
		edges=cbind(c(1:(n-1)),phy$edge, MakeAgeTable(phy, root.age=root.age))

		if(scaleHeight==TRUE){
			edges[,4:5]<-edges[,4:5]/max(MakeAgeTable(phy, root.age=root.age))
            root.age = 1
		}
		edges=edges[sort.list(edges[,3]),]

		#Resort the edge matrix so that it looks like the original matrix order
		edges=edges[sort.list(edges[,1]),]
	}
	if(simmap.tree==FALSE){

		#A boolean for whether the root theta should be estimated -- default is that it should be.
		if (is.character(model)) {
			if (model == "BM1"| model == "OU1"){
				##Begins the construction of the edges matrix -- similar to the ouch format##
				#Makes a vector of absolute times in proportion of the total length of the tree
                k <- 1
                phy$node.label <- rep(1, Nnode(phy))
                tip.states <- factor(rep(1, length(data[,1])))
                int.states <- factor(phy$node.label)
                tot.states <- factor(c(phy$node.label,tip.states))

                #Obtain root state -- for both models assume the root state to be 1 since no other state is used even if provided in the tree
				root.state <- 1
				#New tree matrix to be used for subsetting regimes
				edges=cbind(c(1:(n-1)),phy$edge,MakeAgeTable(phy, root.age=root.age))
				if(scaleHeight==TRUE){
					edges[,4:5] <- edges[,4:5]/max(MakeAgeTable(phy, root.age=root.age))
                    root.age <- 1
				}
				edges=edges[sort.list(edges[,3]),]

				regime <- matrix(0,nrow=length(edges[,1]),ncol=k)
				regime[,1] <- 1
				if (k>1) {
					regime[,2:k] <- 0
				}
				edges=cbind(edges,regime)
			}
			else{
                #Obtain a a list of all the regime states. This is a solution for instances when tip states and
                #the internal nodes are not of equal length:
                tot.states <- factor(c(phy$node.label,as.character(data[,1])))
                k <- length(levels(tot.states))
                int.states <- factor(phy$node.label)
                phy$node.label <- as.numeric(int.states)
                tip.states <- factor(data[,1])
                data[,1] <- as.numeric(tip.states)
				#Obtain root state and internal node labels
				root.state <- phy$node.label[1]
				int.state <- phy$node.label[-1]
				#New tree matrix to be used for subsetting regimes
                edges <- cbind(c(1:(n-1)),phy$edge,MakeAgeTable(phy, root.age=root.age))
				if(scaleHeight==TRUE){
					edges[,4:5] <- edges[,4:5]/max(MakeAgeTable(phy, root.age=root.age))
                    root.age <- 1
				}
				edges=edges[sort.list(edges[,3]),]

				mm <- c(data[,1],int.state)
				regime <- matrix(0,nrow=length(mm),ncol=k)
				#Generates an indicator matrix from the regime vector
				for (i in 1:length(mm)) {
					regime[i,mm[i]] <- 1
				}
				#Finishes the edges matrix
				edges=cbind(edges,regime)
			}
		}
		#Resort the edge matrix so that it looks like the original matrix order
		edges <- edges[sort.list(edges[,1]),]
	}

    #Initializes an integer specifying whether or not the trendy model is to be invoked. Assume 0, unless otherwise stated later:
	trendy <- 0
	#Data:
	x <- as.matrix(data[,2])

	#Matches the model with the appropriate parameter matrix structure
	if (is.character(model)) {
		index.mat <- matrix(0, 2,k)
		Rate.mat <- matrix(1, 2, k)
		if (model == "BM1"){
			np <- 1
			index <- matrix(TRUE,2,k)
			if(mserr=="est"){
				index.mat[1,1:k] <- np+2
			}
			else{
				index.mat[1,1:k] <- np+1
			}
			index.mat[2,1:k] <- 1
			param.count <- np+1
		}
		#The group mean model of Thomas et al, is trivial: set root.station to be TRUE:
		if (model == "BMS"){
			np <- k
			index <- matrix(TRUE,2,k)
			if(mserr=="est"){
				index.mat[1,1:k] <- np+2
			}
			else{
				index.mat[1,1:k] <- np+1
			}
			index.mat[2,1:k]<-1:np
			if(root.station==TRUE){
				param.count <- np+k
			}
            if(root.station==FALSE){
                param.count <- np+1
            }
		}
		if (model == "OU1"){
			np <- 2
			index <- matrix(TRUE,2,k)
			index.mat[1,1:k] <- 1
			index.mat[2,1:k] <- 2
            param.count <- np + 1
            if(get.root.theta == TRUE){
                param.count <- param.count + 1
            }
		}
		if (model == "OUM"){
			np <- 2
			index <- matrix(TRUE,3,k)
			index.mat[1,1:k] <- 1
			index.mat[2,1:k] <- 2
            param.count<-np+k
            if(get.root.theta == TRUE){
                param.count <- param.count + 1
            }
		}
		if (model == "OUMV") {
			np <- k+1
			index <- matrix(TRUE,2,k)
			index.mat[1,1:k] <- 1
			index.mat[2,1:k] <- 2:(k+1)
            param.count <- np+k
            if(get.root.theta == TRUE){
                param.count <- param.count + 1
            }
		}
		if (model == "OUMA") {
			np <- k+1
			index <- matrix(TRUE,2,k)
			index.mat[1,1:k] <- 1:k
			index.mat[2,1:k] <- k+1
            param.count <- np+k
            if(get.root.theta == TRUE){
                param.count <- param.count + 1
            }
		}
		if (model == "OUMVA") {
			np <- k*2
			index <- matrix(TRUE,2,k)
			index.mat[index] <- 1:(k*2)
            param.count <- np+k
            if(get.root.theta == TRUE){
                param.count <- param.count + 1
            }
		}
		if (model == "TrendyM"){
			index.mat <- matrix(0,3,k)
			Rate.mat <- matrix(1, 3, k)
			np=k+2 #We have k regimes, but 1 sigma, plus the root value
			index <- matrix(TRUE,3,k)
			if(mserr=="est"){
				index.mat[1,1:k] <- np+2
			}
			else{
				index.mat[1,1:k] <- np+1
			}
			index.mat[2,1:k] <- 1
			index.mat[3,1:k] <- 2:(np-1)
			param.count<-np
			root.station=TRUE
			trendy=1
		}
		if (model == "TrendyMS"){
			index.mat <- matrix(0,3,k)
			Rate.mat <- matrix(1, 3, k)
			np= (k*2)+1 #We have k regimes and assume k trends and k sigmas, plus the root value
			index <- matrix(TRUE,3,k)
			if(mserr=="est"){
				index.mat[1,1:k] <- np+2
			}
			else{
				index.mat[1,1:k] <- np+1
			}
			index.mat[2,1:k] <- 1:k
			index.mat[3,1:k] <- (k+1):(np-1)
			param.count <- np
			root.station=FALSE
			trendy=1
		}
	}

	if(warn==TRUE){
		if(param.count > (ntips/10)){
			warning("You might not have enough data to fit this model well", call.=FALSE, immediate.=TRUE)
		}
	}

	#Likelihood function for estimating model parameters
	dev <- function(p, index.mat, edges, mserr, trendy, get.root.theta){
        
		if(trendy == 0){
            p <- exp(p)
            Rate.mat[] <- c(p, 1e-10)[index.mat]

            N <- length(x[,1])
            root.par.index <- length(p)
            V <- varcov.ou(phy, edges, Rate.mat, root.state=root.state, simmap.tree=simmap.tree, root.age=root.age, scaleHeight=scaleHeight, assume.station=root.station, shift.point=shift.point)
            
            if (any(is.nan(diag(V))) || any(is.infinite(diag(V)))) return(1000000)
            if(mserr=="known"){
                diag(V) <- diag(V)+(data[,3]^2)
            }
            if(mserr=="est"){
                root.par.index = length(p) - 1
                diag(V) <- diag(V) + (p[length(p)])
            }
            
            if(get.root.theta == TRUE){
                W <- weight.mat(phy, edges, Rate.mat, root.state=root.state, simmap.tree=simmap.tree, root.age=root.age, scaleHeight=scaleHeight, assume.station=FALSE, shift.point=shift.point)
            }else{
                W <- weight.mat(phy, edges, Rate.mat, root.state=root.state, simmap.tree=simmap.tree, root.age=root.age, scaleHeight=scaleHeight, assume.station=TRUE, shift.point=shift.point)
            }

            theta <- Inf
			try(theta <- pseudoinverse(t(W)%*%pseudoinverse(V)%*%W)%*%t(W)%*%pseudoinverse(V)%*%x, silent=TRUE)
			if(any(theta==Inf)){
				return(10000000)
			}
			DET <- sum(log(abs(Re(diag(qr(V)$qr)))))
			#However, sometimes this fails (not sure yet why) so I just toggle between this and another approach:
			if(!is.finite(DET)){
				DET <- determinant(V, logarithm=TRUE)
				logl <- -.5*(t(x-W%*%theta)%*%pseudoinverse(V)%*%(x-W%*%theta))-.5*as.numeric(DET$modulus)-.5*(N*log(2*pi))
			}else{
				logl <- -.5*(t(x-W%*%theta)%*%pseudoinverse(V)%*%(x-W%*%theta))-.5*as.numeric(DET)-.5*(N*log(2*pi))
			}
		}else{
            p <- exp(p)
            Rate.mat[] <- c(p, 1e-10)[index.mat]
            N <- length(x[,1])
            root.par.index <- length(p)
            V <- varcov.ou(phy, edges, Rate.mat, root.state=root.state, simmap.tree=simmap.tree, root.age=root.age, scaleHeight=scaleHeight, assume.station=root.station, shift.point=shift.point)
            if (any(is.nan(diag(V))) || any(is.infinite(diag(V)))) return(1000000)
            if(mserr=="known"){
                diag(V)<-diag(V)+(data[,3]^2)
            }
            if(mserr=="est"){
                root.par.index <- length(p) - 1
                diag(V)<-diag(V)+(p[length(p)])
            }
			E_a <- Inf
			try(E_a <- expected.trendy(phy, edges, Rate.mat, root.state=root.state, simmap.tree=simmap.tree, root.age=root.age, scaleHeight=scaleHeight, root.value=p[root.par.index], shift.point=shift.point))
			if(any(E_a==Inf)){
				return(10000000)
			}
			DET <- sum(log(abs(Re(diag(qr(V)$qr)))))
			#However, sometimes this fails (not sure yet why) so I just toggle between this and another approach:
			if(!is.finite(DET)){
				DET <- determinant(V, logarithm=TRUE)
				logl <- -.5*(t(x-E_a)%*%pseudoinverse(V)%*%(x-E_a))-.5*as.numeric(DET$modulus)-.5*(N*log(2*pi))
			}else{
				logl <- -.5*(t(x-E_a)%*%pseudoinverse(V)%*%(x-E_a))-.5*as.numeric(DET)-.5*(N*log(2*pi))
			}
		}
        
		#When the model includes alpha, the values of V can get too small, the modulus does not seem correct and the loglik becomes unstable. This is one solution:
		if(!is.finite(logl)){
			return(10000000)
		}
		return(-logl)
	}

	if(quiet==FALSE){
		cat("Initializing...", "\n")
	}

	lb = -20
	ub = 20
	lower = rep(-20, np)
	upper = rep(20, np)

	#opts <- list("algorithm"="NLOPT_LN_SBPLX", "maxeval"="1000000", "ftol_rel"=.Machine$double.eps^0.5)

	if(model == "OU1" | model == "OUM" | model == "OUMV" | model == "OUMA" | model == "OUMVA"){
		if(scaleHeight==TRUE){
			d <- max(diag(vcv.phylo(phy)))
			phy$edge.length <- (phy$edge.length/d)
		}
		C.mat <- vcv.phylo(phy)
		a <- as.numeric(colSums(solve(C.mat))%*%x/sum(solve(C.mat)))
		sig <- as.numeric(t(x-a)%*%solve(C.mat)%*%(x-a)/n)
		init.np=2
		init.lower = rep(lb, init.np)
		init.upper = rep(ub, init.np)
		init.index.mat<-matrix(0,2,k)
		init.index.mat[1,1:k]<-1
		init.index.mat[2,1:k]<-2
		if(model == "OU1"){
            edges.tmp=edges
            edges.tmp[,6] <- rep(1, length(edges.tmp[,1]))
		}else{
			edges.tmp=edges
		}

		#Initial value for alpha is just the half life based on the entire length of the tree:
        if(is.null(starting.vals)){
            init <- nloptr(x0=log(c(log(2)/max(branching.times(phy)),sig)), eval_f=dev, lb=init.lower, ub=init.upper, opts=opts, index.mat=init.index.mat, edges=edges.tmp, mserr=mserr, trendy=trendy, get.root.theta=FALSE)
            init.ip <- c(exp(init$solution[1]), exp(init$solution[2]))
        }else{
            init.ip <- starting.vals
        }
		if(model=="OU1"){
			ip <- init.ip
		}
		if(model=="OUMV" | model=="OUMA" | model=="OUM"){
			ip <- c(rep(init.ip[1],length(unique(index.mat[1,]))),rep(init.ip[2],length(unique(index.mat[2,]))))
		}
		if(model=="OUMVA"){
			ip <- c(rep(init.ip,k))
		}
		if(mserr=="est"){
			ip <- c(ip,sig)
			lower = c(lower,0)
			upper = c(upper,ub)
		}
		if(quiet==FALSE){
			cat("Finished. Begin thorough search...", "\n")
		}
		out = nloptr(x0=log(ip), eval_f=dev, lb=lower, ub=upper, opts=opts, index.mat=index.mat, edges=edges, mserr=mserr, trendy=trendy, get.root.theta=get.root.theta)
	}
	else{
		if(scaleHeight==TRUE){
			d <- max(diag(vcv.phylo(phy)))
			phy$edge.length<-(phy$edge.length/d)
		}
		#Starting values follow from phytools:
		C.mat <- vcv.phylo(phy)
		a <- as.numeric(colSums(solve(C.mat))%*%x/sum(solve(C.mat)))
        if(is.null(starting.vals)){
            sig <- as.numeric(t(x-a)%*%solve(C.mat)%*%(x-a)/n)
        }else{
            sig <- starting.vals
        }
		#####################
		if(model=="BMS"){
			ip <- rep(sig,k)
		}
		else{
			if(model=="BM1"){
				ip <- sig
			}
			if(model=="TrendyM"){
				#We assume that the starting trend values are zero:
				ip <- c(sig, rep(exp(-20), param.count-2), a)
				lower <- c(lb,rep(-20, param.count-2), -20)
				upper <- c(ub,rep(20, param.count-2), 20)
			}
			if(model == "TrendyMS"){
				#We assume that the starting trend values are zero:
				ip <- c(rep(sig, k), rep(exp(-20), (np-1) - k), a)
				lower <- c(rep(lb,k), rep(-20, (np-1) - k), -20)
				upper <- c(rep(ub,k), rep(20, (np-1) - k), 20)
			}
		}
		if(mserr=="est"){
			ip <- c(ip,sig)
			lower <- c(lower,0)
			upper <- c(upper,10)
		}
		if(quiet==FALSE){
			cat("Finished. Begin thorough search...", "\n")
		}
		out = nloptr(x0=log(ip), eval_f=dev, lb=lower, ub=upper, opts=opts, index.mat=index.mat, edges=edges, mserr=mserr, trendy=trendy, get.root.theta=get.root.theta)
	}
	
    loglik <- -out$objective
	out$solution <- exp(out$solution)
	out$new.starting <- out$solution # note: back in regular param space, not log space used in the search
	
    #Takes estimated parameters from dev and calculates theta for each regime:
	dev.theta <- function(p, index.mat, edges=edges, mserr=mserr){
		tmp<-NULL
		Rate.mat[] <- c(p, 1e-10)[index.mat]
		N <- length(x[,1])
		V <- varcov.ou(phy, edges, Rate.mat, root.state=root.state, simmap.tree=simmap.tree, root.age=root.age, scaleHeight=scaleHeight, assume.station=root.station, shift.point=shift.point)
        
        if(get.root.theta == TRUE){
            W <- weight.mat(phy, edges, Rate.mat, root.state=root.state, simmap.tree=simmap.tree, root.age=root.age, scaleHeight=scaleHeight, assume.station=FALSE, shift.point=shift.point)
        }else{
            W <- weight.mat(phy, edges, Rate.mat, root.state=root.state, simmap.tree=simmap.tree, root.age=root.age, scaleHeight=scaleHeight, assume.station=TRUE, shift.point=shift.point)
        }
        
        if(mserr=="known"){
            diag(V) <- diag(V)+(data[,3]^2)
		}
		if(mserr=="est"){
			diag(V) <- diag(V)+(p[length(p)])
		}
		theta <- pseudoinverse(t(W)%*%pseudoinverse(V)%*%W)%*%t(W)%*%pseudoinverse(V)%*%x
        
        #Calculates the hat matrix:
		#H.mat<-W%*%pseudoinverse(t(W)%*%pseudoinverse(V)%*%W)%*%t(W)%*%pseudoinverse(V)
		#Standard error of theta -- uses pseudoinverse to overcome singularity issues
		se <- sqrt(diag(pseudoinverse(t(W)%*%pseudoinverse(V)%*%W)))
        tmp$res <- W%*%theta-x
		#Joins the vector of thetas with the vector of standard errors into a 2 column matrix for easy extraction at the summary stage
		tmp$theta.est <- cbind(theta,se)

        #Calculates R-sq according to Judge et al. 1985 eq. 2.3.16:
        residuals <- tmp$res
        fitted.vals <- W%*%theta
        Y <- residuals + fitted.vals
        one <- matrix(1, length(Y), 1)
        a <- as.numeric(pseudoinverse(t(one) %*% pseudoinverse(V) %*% one) %*% (t(one) %*% pseudoinverse(V) %*% Y))
        Rsq <- 1-(t(residuals) %*% pseudoinverse(V) %*% residuals) / (t(Y-a) %*% pseudoinverse(V) %*% (Y-a))
		tmp$Rsq <- Rsq
        tmp$W <- W
        return(tmp)
	}

	#Informs the user that the summarization has begun, output model-dependent and dependent on whether the root theta is to be estimated
	if(quiet==FALSE){
		cat("Finished. Summarizing results.", "\n")
	}

	if(model == "TrendyM" | model == "TrendyMS"){
		theta<-NULL
		root.est <- out$solution[length(out$solution)]
		theta$theta.est <- matrix(root.est, k+1, 2)
		theta$theta.est[,2] <- 0
		theta$res <- expected.trendy(phy, edges, Rate.mat, root.state=root.state, simmap.tree=simmap.tree, root.age=root.age, scaleHeight=scaleHeight, root.value=root.est, shift.point=shift.point) - x
	}else{
		theta <- dev.theta(out$solution, index.mat, edges, mserr)
	}
    
    
	#Calculates the Hessian for use in calculating standard errors and whether the maximum likelihood solution was found
	if(diagn==TRUE){
		h <- hessian(x=log(out$solution), func=dev, index.mat=index.mat, edges=edges, mserr=mserr, trendy=trendy, get.root.theta=get.root.theta)
		#Using the corpcor package here to overcome possible NAs with calculating the SE
		solution <- matrix(out$solution[index.mat], dim(index.mat))
		solution.se <- matrix(sqrt(diag(pseudoinverse(h)))[index.mat], dim(index.mat))
		rownames(solution) <- rownames(solution.se) <- rownames(index.mat) <- c("alpha","sigma.sq")
		if(simmap.tree==FALSE){
			colnames(solution) <- colnames(solution.se) <- levels(tot.states)
		}
		if(simmap.tree==TRUE){
			colnames(solution) <- colnames(solution.se) <- c(colnames(phy$mapped.edge))
		}
		#Eigendecomposition of the Hessian to assess reliability of likelihood estimates
		hess.eig<-eigen(h,symmetric=TRUE)
		#If eigenvect is TRUE then the eigenvector and index matrix will appear in the list of objects
		eigval<-signif(hess.eig$values,2)
		eigvect<-round(hess.eig$vectors, 2)
		if(mserr=="est"){
			mserr.est<-out$solution[length(out$solution)]
			param.count<-param.count+1
		}
		else{
			mserr.est<-NULL
		}
        
        if(get.root.theta == TRUE){
            rates <- cbind(NA, solution, NA)
            thetas <- cbind(t(theta$theta.est[,1]), NA)
            rates <- rbind(rates, thetas)
            weights <- cbind(theta$W, data[,1])
            regime.weights <- rbind(rates, weights)
            rownames(regime.weights) <- c("alpha", "sigma.sq", "theta", phy$tip.label)
            colnames(regime.weights) <- c("Root", levels(tot.states), "Tip_regime")
        }else{
            if(model=="TrendyM" | model=="TrendyMS"){
                regime.weights = NULL
            }else{
                rates <- cbind(solution, NA)
                thetas <- cbind(t(theta$theta.est[,1]), NA)
                rates <- rbind(rates, thetas)
                weights <- cbind(theta$W, data[,1])
                regime.weights <- rbind(rates, weights)
                rownames(regime.weights) <- c("alpha", "sigma.sq", "theta", phy$tip.label)
                colnames(regime.weights) <- c(levels(tot.states), "Tip_regime")
            }
        }

		obj = list(loglik = loglik, AIC = -2*loglik+2*param.count, AICc=-2*loglik+(2*param.count*(ntips/(ntips-param.count-1))), BIC=-2*loglik + log(ntips) * param.count, model=model, param.count=param.count, solution=solution, mserr.est=mserr.est, theta=theta$theta.est, solution.se=solution.se, tot.states=tot.states, index.mat=index.mat, simmap.tree=simmap.tree, root.age=root.age, shift.point=shift.point, opts=opts, data=data, phy=phy, root.station=root.station, scaleHeight=scaleHeight, starting.vals=starting.vals, lb=lower, ub=upper, iterations=out$iterations, res=theta$res, rsq=theta$Rsq, get.root.theta=get.root.theta, regime.weights=regime.weights, eigval=eigval, eigvect=eigvect, new.start=out$new.start)

	}
	if(diagn==FALSE){
		solution <- matrix(out$solution[index.mat], dim(index.mat))
		if(model=="TrendyM" | model=="TrendyMS"){
			rownames(solution) <- rownames(index.mat) <- c("alpha","sigma.sq","trend")
		}else{
			rownames(solution) <- rownames(index.mat) <- c("alpha","sigma.sq")
		}
		if(simmap.tree==FALSE){
			colnames(solution) <- levels(tot.states)
		}
		if(simmap.tree==TRUE){
			colnames(solution) <- c(colnames(phy$mapped.edge))
		}
		if(mserr=="est"){
			mserr.est<-out$solution[length(out$solution)]
			param.count<-param.count+1
		}
		else{
			mserr.est<-NULL
		}
        
        if(get.root.theta == TRUE){
            rates <- cbind(NA, solution, NA)
            thetas <- cbind(t(theta$theta.est[,1]), NA)
            rates <- rbind(rates, thetas)
            weights <- cbind(theta$W, data[,1])
            regime.weights <- rbind(rates, weights)
            rownames(regime.weights) <- c("alpha", "sigma.sq", "theta", phy$tip.label)
            colnames(regime.weights) <- c("Root", levels(tot.states), "Tip_regime")
        }else{
            if(model=="TrendyM" | model=="TrendyMS"){
                regime.weights = NULL
            }else{
                rates <- cbind(solution, NA)
                thetas <- cbind(t(theta$theta.est[,1]), NA)
                rates <- rbind(rates, thetas)
                weights <- cbind(theta$W, data[,1])
                regime.weights <- rbind(rates, weights)
                rownames(regime.weights) <- c("alpha", "sigma.sq", "theta", phy$tip.label)
                colnames(regime.weights) <- c(levels(tot.states), "Tip_regime")
            }
        }

        obj = list(loglik = loglik, AIC = -2*loglik+2*param.count,AICc=-2*loglik+(2*param.count*(ntips/(ntips-param.count-1))), BIC=-2*loglik + log(ntips) * param.count, model=model, param.count=param.count, solution=solution, mserr.est=mserr.est, theta=theta$theta.est, tot.states=tot.states, index.mat=index.mat, simmap.tree=simmap.tree, root.age=root.age, shift.point=shift.point, opts=opts, data=data, phy=phy, root.station=root.station, scaleHeight=scaleHeight, starting.vals=starting.vals, lb=lower, ub=upper, iterations=out$iterations, res=theta$res, rsq=theta$Rsq, get.root.theta=get.root.theta, regime.weights=regime.weights, new.start=out$new.start)
	}
	class(obj)<-"OUwie"
	return(obj)
}


print.OUwie<-function(x, ...){

	ntips=Ntip(x$phy)
	output<-data.frame(x$loglik,x$AIC,x$AICc,x$BIC,x$model,ntips, row.names="")
	names(output)<-c("lnL","AIC","AICc","BIC","model","ntax")
	cat("\nFit\n")
	print(output)
	cat("\n")

	if (is.character(x$model)) {
		if (x$model == "BM1" | x$model == "BMS" | x$model == "TrendyM" | x$model == "TrendyMS"){
			param.est <- x$solution
			if(x$root.station==FALSE){
			theta.mat <- matrix(t(x$theta[1,]), 2, length(levels(x$tot.states)))
			}
			else{
				theta.mat<-matrix(t(x$theta), 2, length(levels(x$tot.states)))
			}
			rownames(theta.mat)<-c("estimate", "se")
			if(x$simmap.tree==FALSE){
				colnames(theta.mat) <- levels(x$tot.states)
			}
			if(x$simmap.tree==TRUE){
				colnames(theta.mat) <- c(colnames(x$phy$mapped.edge))
			}
			cat("Rates\n")
			print(param.est)
			cat("\n")
			cat("Optima\n")
			print(theta.mat)
			cat("\n")
		}
        if (x$get.root.theta == FALSE){
            if (x$model == "OU1" | x$model == "OUM"| x$model == "OUMV"| x$model == "OUMA" | x$model == "OUMVA"){
                param.est<- x$solution
                theta.mat<-matrix(t(x$theta), 2, length(levels(x$tot.states)))
                rownames(theta.mat)<-c("estimate", "se")
                if(x$simmap.tree==FALSE){
                    colnames(theta.mat)<- levels(x$tot.states)
                }
                if(x$simmap.tree==TRUE){
                    colnames(theta.mat) <- c(colnames(x$phy$mapped.edge))
                }
                cat("\nRates\n")
                print(param.est)
                cat("\n")
                cat("Optima\n")
                print(theta.mat)
                cat("\n")
                cat("\nHalf life (another way of reporting alpha)\n")
                print(log(2)/param.est['alpha',])
                cat("\n")
            }
        }
        if (x$get.root.theta == TRUE){
            if (x$model == "OU1" | x$model == "OUM"| x$model == "OUMV"| x$model == "OUMA" | x$model == "OUMVA"){
                param.est<- x$solution
                theta.mat<-matrix(t(x$theta), 2, length(levels(x$tot.states))+1)
                rownames(theta.mat)<-c("estimate", "se")
                if(x$simmap.tree==FALSE){
                    colnames(theta.mat)<-c("root", levels(x$tot.states))
                }
                if(x$simmap.tree==TRUE){
                    colnames(theta.mat)<-c("root", colnames(x$phy$mapped.edge))
                }
                cat("\nRates\n")
                print(param.est)
                cat("\n")
                cat("Optima\n")
                print(theta.mat)
                cat("\n")
                cat("\nHalf life (another way of reporting alpha)\n")
                print(log(2)/param.est['alpha',])
                cat("\n")
            }
        }
    }
	if(any(x$eigval<0)){
		index.matrix <- x$index.mat
		if(x$simmap.tree==FALSE){
			colnames(index.matrix) <- levels(x$tot.states)
		}
		if(x$simmap.tree==TRUE){
			colnames(index.matrix) <- colnames(x$phy$mapped.edge)
		}
		#If any eigenvalue is less than 0 then the solution is not the maximum likelihood solution
		if (any(x$eigval<0)) {
			cat("The objective function may be at a saddle point -- check eigenvectors or try a simpler model", "\n")
		}
	}
	else{
		cat("Arrived at a reliable solution","\n")
	}
}



MakeAgeTable <- function(phy, root.age=NULL){
    if(is.null(root.age)){
        node.ages <- dateNodes(phy, rootAge=max(node.depth.edgelength(phy)))
    }else{
        node.ages <- dateNodes(phy, rootAge=root.age)
    }
    max.age <- max(node.ages)
    table.ages <- matrix(0, dim(phy$edge)[1], 2)
    for(row.index in 1:dim(phy$edge)[1]){
        table.ages[row.index,1] <- max.age - node.ages[phy$edge[row.index,1]]
        table.ages[row.index,2] <- max.age - node.ages[phy$edge[row.index,2]]
    }
    return(table.ages)
}
