#' @title Import event records from Oregon RFID antenna readers
#' @description Imports files from Oregon RFID (radio-frequency identification) ORMR (Oregon RFID Multi-Reader) and ORSR (Oregon RFID Single Reader) antenna readers. Only event records are retained during compilation (detections are removed). The function will not work with older generations of Oregon RFID antenna readers.
#' 
#' This function will only work with raw data downloaded directly from Oregon RFID stationary readers. The files must be delimited (tab, comma, or semicolon) and unedited by the user.
#' @param file .txt file generated by an Oregon RFID reader.
#' @param delim field/column delimiter, which must be tab ('\\t'), comma (',') or semi-colon (';').
#' @param verbose If \emph{TRUE}, a description of the compiled data is printed to the console.
#' @details The field/column delimiter must be tab, comma or semi-colon for data compilation. The function cannot be used for space delimited data. 
#' @return Returns a tibble object.
#' @author Hugo Marques <biohmarques@@gmail.com>
#' @importFrom magrittr %>%
#' @export
#' @examples
#' 
#' # Importing event records from a single ORFID reader 
#' import_ORFID_events(file = system.file("extdata/orfid_data_file.txt", 
#' package = "ORFID"), delim = "\t")
#' 

import_ORFID_events <- function(file, delim, verbose = FALSE) {
    
    if (!(delim %in% c("\t", ",", ";"))) {
        stop("Column delimiter must be tab, comma, or semicolon")
    }
    
    raw_data <- suppressWarnings(
        readr::read_delim(file, 
                          delim = delim, 
                          skip = grep("* records ---------$", readLines(file, warn = FALSE)),
                          show_col_types = FALSE)
    )
    
    if (ncol(raw_data) == 1) {
        stop("Column delimiter must be tab, comma, or semicolon")
    }
    
    if (("SCD" %in% names(raw_data))) {
        
        site_code <- raw_data %>% 
            dplyr::distinct(SCD) %>% 
            dplyr::filter(!is.na(SCD))
        
        raw_data2 <- raw_data %>%
            dplyr::filter(DTY == "E") %>%
            dplyr::mutate(SCD = as.character(site_code)) %>% 
            dplyr::mutate(message = DUR) %>%
            dplyr::mutate(TRF = as.factor(TRF)) %>%
            dplyr::select(DTY, SCD, ARR, TRF, message)
        
    }
    
    if ((!"SCD" %in% names(raw_data))) {
        
        raw_data2 <- raw_data %>%
            dplyr::filter(DTY == "E") %>%
            dplyr::mutate(SCD = "NA") %>% 
            dplyr::mutate(message = DUR) %>%
            dplyr::mutate(TRF = as.factor(TRF)) %>%
            dplyr::select(DTY, SCD, ARR, TRF, message)
        
        warning("Site code (SCD) is required for further analysis.")
        
    }
    
    if (verbose == TRUE) {
        return(dplyr::glimpse(raw_data2))
    } else {
        return(raw_data2)
    }
    
}
