\name{model.build}
\alias{model.build}


\title{ Model Building}
\description{
  Create sophisticated models using Random Forest, Quantile Regression Forests, Conditional Forests, or Stochastic Gradient Boosting from training data
}
\usage{ 

model.build(model.type = NULL, qdata.trainfn = NULL, folder = NULL,
MODELfn = NULL, predList = NULL, predFactor = FALSE, response.name = NULL,
response.type = NULL, unique.rowname = NULL, seed = NULL, na.action = NULL,
keep.data = TRUE, ntree = switch(model.type,RF=500,QRF=1000,CF=500,500),
mtry = switch(model.type,RF=NULL,QRF=ceiling(length(predList)/3),
CF = min(5,length(predList)-1),NULL), replace = TRUE, strata = NULL,
sampsize = NULL, proximity = FALSE, importance=FALSE, 
quantiles=c(0.1,0.5,0.9), subset = NULL, weights = NULL,
controls = NULL, xtrafo = NULL, ytrafo = NULL, scores = NULL) 
}

\arguments{

  \item{model.type}{ String.  Model type.  \code{"RF"} (random forest), \code{"QRF"} (quantile random forest), or \code{"CF"} (conditional forest). The \code{ModelMap} package does not currently support \code{SGB} models. }

  \item{qdata.trainfn}{String.  The name (full path or base name with path specified by \code{folder}) of the training data file used for building the model (file should include columns for both response and predictor variables).  The file must be a comma-delimited file \code{*.csv} with column headings. \code{qdata.trainfn} can also be an \code{R} dataframe. If predictions will be made (\code{predict = TRUE} or \code{map=TRUE}) the predictor column headers must match the names of the raster layer files, or a \code{rastLUT} must be provided to match predictor columns to the appropriate raster and band.  If \code{qdata.trainfn = NULL} (the default), a GUI interface prompts user to browse to the training data file.  }

  \item{folder}{ String.  The folder used for all output from predictions and/or maps.  Do not add ending slash to path string.  If \code{folder = NULL} (default), a GUI interface prompts user to browse to a folder.  To use the working directory, specify \code{folder = getwd()}.}

  \item{MODELfn}{ String.  The file name to use to save files related to the model object.  If \code{MODELfn = NULL} (the default), a default name is generated by pasting \code{model.type}, \code{response.type}, and \code{response.name}, separated by underscores. If the other output filenames are left unspecified, \code{MODELfn} will be used as the basic name to generate other output filenames. The filename can be the full path, or it can be the simple basename, in which case the output will be to the folder specified by \code{folder}.}

  \item{predList}{ String.  A character vector of the predictor short names used to build the model.  These names must match the column names in the training/test data files and the names in column two of the \code{rastLUT}.  If \code{predList = NULL} (the default), a GUI interface prompts user to select predictors from column 2 of \code{rastLUT}. 

If both \code{predList = NULL} and \code{rastLUT = NULL}, then a GUI interface prompts user to browse to rasters used as predictors, and select from a generated list, the individual layers (bands) of rasters used to build the model.  In this case (i.e., \code{rastLUT = NULL}), predictor column names of training data must be standard format, consisting of raster stack name followed by b1, b2, etc..., giving the band number within each stack (Example: \code{stacknameb1}, \code{stacknameb2}, \code{stacknameb3}, ...). 
 }

  \item{predFactor}{ String.  A character vector of predictor short names of the predictors from \code{predList} that are factors (i.e categorical predictors).  These must be a subset of the predictor names given in \code{predList} Categorical predictors may have multiple categories.}

  \item{response.name}{ String.  The name of the response variable used to build the model.  If \code{response.name = NULL}, a GUI interface prompts user to select a variable from the list of column names from training data file.  \code{response.name} must be column name from the training/test data files. }

  \item{response.type}{ String.  Response type: \code{"binary"}, \code{"categorical"} or \code{"continuous"}.  Binary response must be binary 0/1 variable with only 2 categories.  All zeros will be treated as one category, and everything else will be treated as the second category.}

  \item{unique.rowname}{ String.  The name of the unique identifier used to identify each row in the training data.  If \code{unique.rowname = NULL}, a GUI interface prompts user to select a variable from the list of column names from the training data file.  If \code{unique.rowname = FALSE}, a variable is generated of numbers from \code{1} to \code{nrow(qdata)} to index each row. }

  \item{seed}{ Integer.  The number used to initialize randomization to build RF or SGB models.  If you want to produce the same model later, use the same seed.  If \code{seed = NULL} (the default), a new seed is created each run. }

  \item{na.action}{String.  Model validation.  Specifies the action to take if there are \code{NA} values in the predictor data. There are 2 options: (1) \code{na.action = na.omit} where any data point with missing predictors is removed from the model building data; (2) \code{na.action = na.roughfix} where a missing categorical predictor is replaced with the most common category, and a missing continuous predictor or response is replaced with the median. Note: it is not recommended that \code{na.roughfix} will just be used for missing predictor. Data points with missing response will always be omitted.  }

  \item{keep.data}{Logical.  RF and SGB models. Should a copy of the predictor data be included in the model object. Useful for if \code{\link{model.interaction.plot}} will be used later. }
 
 \item{ntree}{ Integer.  RF QRF and CF models.  The number of random forest trees for a RF model.  The default is 500 trees. }
 
  \item{mtry}{ Integer.  RF QRF and CF models.  Number of variables to try at each node of Random Forest trees.  By default, RF models will use the \code{"tuneRF()"} function to optimize \code{mtry}. }

  \item{replace}{ Logical.  RF models.  Should sampling of cases be done with or without replacement? }

  \item{strata}{ Factor or String.  RF models.  A (factor) variable that is used for stratified sampling. Can be in the form of either the name of the column in \code{qdata} or a factor or vector with one element for each row of \code{qdata}. }

  \item{sampsize}{ Vector.  RF models.  Size(s) of sample to draw.  For classification, if \code{sampsize} is a vector of the length the number of factor levels \code{strata}, then sampling is stratified by \code{strata}, and the elements of \code{sampsize} indicate the numbers to be drawn from each strata. If argument \code{strata} is not provided, and \code{repsonse.type = "binary"} then sampling is stratified by presence/absence. If argument \code{sampsize} is not provided \code{model.build()} will use the default value from the \code{randomForest} package: \code{if (replace) nrow(data) else ceiling(.632*nrow(data))}. }

  \item{proximity}{ Logical.  RF models.  Should proximity measure among the rows be calculated for unsupervised models?}

  \item{importance}{ Logical. QRF models. For QRF models only, importance must be specified at the time of model building. If TRUE importance of predictors is assessed at the given \code{quantiles}. Warning, on large datasets calculating QRF importances is very memory intensive and may require increasing memory limits with \code{memory.limit()}. NOTE: Importance currently unavailable for QRF models.}

   \item{quantiles}{Numeric. Used for QRF models if \code{importance=TRUE}. Specify which quantiles of response variable to use. Later importance plots can only be made for \code{quantiles} specified at the time of model building.}

  \item{subset}{CF models. An optional vector specifying a subset of observations to be used in the fitting process. Note: \code{subset} is not supported for cross validation diagnostics.}

  \item{weights}{CF models. An optional vector of weights to be used in the fitting process. Non-negative integer valued weights are allowed as well as non-negative real weights. Observations are sampled (with or without replacement) according to probabilities \code{weights/sum(weights)}. The fraction of observations to be sampled (without replacement) is computed based on the sum of the weights if all weights are integer-valued and based on the number of weights greater zero else. Alternatively, \code{weights} can be a double matrix defining case weights for all \code{ncol(weights)} trees in the forest directly. This requires more storage but gives the user more control. Note: \code{weights} is not supported for cross validation diagnostics.}

  \item{controls}{CF models. An object of class \code{\link[party]{ForestControl-class}}, which can be obtained using cforest_control (and its convenience interfaces cforest_unbiased and cforest_classical). If \code{controls} is specified, then stand alone arguments \code{mtry} and \code{ntree} ignored and these parameters must be specified as part of the \code{controls} argument. If \code{controls} not specified, \code{model.build} defaults to \code{cforest_unbiased(mtry=mtry, ntree=ntree)} with the values of \code{mtry} and \code{ntree} specified by the stand alone arguments.}

  \item{xtrafo}{CF models. A function to be applied to all input variables. By default, the \code{ptrafo} function from the \code{party} package is applied. Defaults to \code{xtrafo=ptrafo}.}

  \item{ytrafo}{CF models. A function to be applied to all response variables. By default, the \code{ptrafo} function from the \code{party} package is applied. Defaults to \code{ytrafo=ptrafo}.}

  \item{scores}{CF models. An optional named list of scores to be attached to ordered factors. Note: \code{weights} is not supported for cross validation diagnostics.}
}

\details{
 
This package provides a push button approach to complex model building and production mapping.  It contains three main functions: \code{\link{model.build}},\code{\link{model.diagnostics}}, and \code{\link{model.mapmake}}. 

In addition it contains a simple function \code{\link{get.test}} that can be used to randomly divide a training dataset into training and test/validation sets; \code{\link{build.rastLUT}} that uses GUI prompts to walk a user through the process of setting up a Raster look up table to link predictors from the training data with the rasters used for map contruction; \code{\link{model.explore}}, for preliminary data exploration; and, \code{\link{model.importance.plot}} and \code{\link{model.interaction.plot}} for interpreting the effects of individual model predictors. 

These functions can be run in a traditional R command mode, where all arguments are specified in the function call.  However they can also be used in a full push button mode, where you type in, for example, the simple command \code{\link{model.build}}, and GUI pop up windows will ask questions about the type of model, the file locations of the data, etc...

When running the \code{ModelMap} package on non-Windows platforms, file names and folders need to be specified in the argument list, but other pushbutton selections are handled by the \code{select.list()} function, which is platform independent. 

Binary, categorical, and continuous response models are supported for Random Forest and Conditional Forest. Quantile Random Forest is appropriate for only continuous response models.

Random Forest is implemented through the \code{randomForest} package within \code{R}.  Random Forest is more user friendly than Stochastic Gradient Boosting, as it has fewer parameters to be set by the user, and is less sensitive to tuning of these parameters.  A Random Forest model consists of multiple trees that vote on predictions.  For each tree a random subset of the training data is used to construct the tree, with the remaining data points used to construct out-of-bag (OOB) error estimates.  At each node of the tree a random selection of predictors is chosen to determine the split.  The number of predictors used to select the splits (argument \code{mtry}) is the primary user specified parameter that can affect model performance.  

By default \code{mtry} will be automatically optimized using the \code{randomForest} package \code{tuneRF()} function. Note that this is a stochastic process. If there is a chance that models may be combined later with the \code{randomForest} package \code{combine} function then for consistency it is important to provide the \code{mtry} argument rather that using the default optimization process.

Random Forest will not over fit data, therefore the only penalty of increasing the number of trees is computation time.  Random Forest can compute variable importance, an advantage over some "black box" modeling techniques if it is important to understand the ecological relationships underlying a model (Brieman, 2001).

Quantile Regression Forests is implemented through the \code{quantregForest} package.

Conditional Forests is implemented with the \code{cforest()} function in the \code{party} package. As stated in the \code{party} package, ensembles of conditional inference trees have not yet been extensively tested, so this routine is meant for the expert user only and its current state is rather experimental.

For CF models, \code{ModelMap} currently only supports binary, categorical and continuous response models. Also, for some CF model parameters (\code{subset}, \code{weights}, and \code{scores}) \code{ModelMap} only provides OOB and independent test set diagnostics, and does not support cross validation diagnostics.

Stochastic gradient boosting is not currently supported by \code{ModelMap}. 
}

\value{

The function will return the model object. Additionally, it will write a text file to disk, in the folder specified by \code{folder}. This file lists the values of each argument as chosen from GUI prompts used for the function call.

}

\references{ 
Breiman, L. (2001) Random Forests. Machine Learning, 45:5-32.

Elith, J., Leathwick, J. R. and Hastie, T. (2008). A working guide to boosted regression trees. Journal of Animal Ecology. 77:802-813.

Liaw, A. and  Wiener, M. (2002). Classification and Regression by randomForest. R News 2(3), 18--22.

N. Meinshausen (2006) "Quantile Regression Forests", Journal of Machine Learning Research 7, 983-999 http://jmlr.csail.mit.edu/papers/v7/

Ridgeway, G., (1999). The state of boosting. Comp. Sci. Stat. 31:172-181

Carolin Strobl, Anne-Laure Boulesteix, Achim Zeileis and Torsten Hothorn (2007). Bias in Random Forest variable Importance Measures: Illustrations, Sources and a Solution. BMC Bioinformatics, 8, 25. http://www.biomedcentral.co,/1471-2105/8/25

Carolin Strobl, James Malley and Gerhard Tutz (2009). An Introduction to Recursive Partitioning: Rationale, Application, and Characteristics of Classification and Regression Trees, Bagging, and Random forests. Phsycological Methods, 14(4), 323-348. 

Torsten Hothorn, Berthold Lausen, Axel Benner and Martin Radespiel-Troeger (2004). Bagging Survival Trees. Statistics in Medicine, 23(1), 77-91.

Torsten Hothorn, Peter Buhlmann, Sandrine Dudoit, Annette Molinaro and Mark J. ven der Laan (2006a). Survival Ensembles. Biostatistics, 7(3), 355-373. 

Torston Hothorn, Kurt Hornik and Achim Zeileis (2006b). Unbiased Recursive Partitioning: A Conditional Inference Framework. Journal of Computational and Graphical Statistics, 15(3), 651-674. Preprint available from http://statmath.wu-wein.ac.at/~zeileis/papers/Hothorn+Hornik+Zeileis-2006.pdf
 }

\author{ Elizabeth Freeman and Tracey Frescino }

\seealso{ \code{\link{get.test}}, \code{\link{model.diagnostics}}, \code{\link{model.mapmake}}}
\examples{

###########################################################################
############################# Run this set up code: #######################
###########################################################################

# set seed:
seed=38

# Define training and test files:

qdata.trainfn = system.file("extdata", "helpexamples","DATATRAIN.csv", package = "ModelMap")

# Define folder for all output:
folder=getwd()	

#identifier for individual training and test data points

unique.rowname="ID"


###########################################################################
############## Pick one of the following sets of definitions: #############
###########################################################################


########## Continuous Response, Continuous Predictors ############

#file name:
MODELfn="RF_Bio_TC"				

#predictors:
predList=c("TCB","TCG","TCW")	

#define which predictors are categorical:
predFactor=FALSE	

# Response name and type:
response.name="BIO"
response.type="continuous"


########## binary Response, Continuous Predictors ############

#file name to store model:
MODELfn="RF_CONIFTYP_TC"				

#predictors:
predList=c("TCB","TCG","TCW")		

#define which predictors are categorical:
predFactor=FALSE

# Response name and type:
response.name="CONIFTYP"

# This variable is 1 if a conifer or mixed conifer type is present, 
# otherwise 0.

response.type="binary"


########## Continuous Response, Categorical Predictors ############

# In this example, NLCD is a categorical predictor.
#
# You must decide what you want to happen if there are categories
# present in the data to be predicted (either the validation/test set
# or in the image file) that were not present in the original training data.
# Choices:
#       na.action = "na.omit"
#                    Any validation datapoint or image pixel with a value for any
#                    categorical predictor not found in the training data will be
#                    returned as NA.
#       na.action = "na.roughfix"
#                    Any validation datapoint or image pixel with a value for any
#                    categorical predictor not found in the training data will have
#                    the most common category for that predictor substituted,
#                    and the a prediction will be made.

# You must also let R know which of the predictors are categorical, in other
# words, which ones R needs to treat as factors.
# This vector must be a subset of the predictors given in predList

#file name to store model:
MODELfn="RF_BIO_TCandNLCD"			

#predictors:
predList=c("TCB","TCG","TCW","NLCD")

#define which predictors are categorical:
predFactor=c("NLCD")

# Response name and type:
response.name="BIO"
response.type="continuous"



###########################################################################
########################### build model: ##################################
###########################################################################


### create model before batching (only run this code once ever!) ###

model.obj = model.build( model.type="RF",
                       qdata.trainfn=qdata.trainfn,
                       folder=folder,		
                       unique.rowname=unique.rowname,	
                       MODELfn=MODELfn,
                       predList=predList,
                       predFactor=predFactor,
                       response.name=response.name,
                       response.type=response.type,
                       seed=seed,
                       na.action="na.roughfix"
)



}

\keyword{ models }

