% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getTile.R
\name{getTile}
\alias{getTile}
\title{Get MODIS Tile ID(s)}
\usage{
getTile(extent = NULL, tileH = NULL, tileV = NULL, buffer = NULL,
  system = "MODIS", zoom = TRUE)
}
\arguments{
\item{extent}{Extent information, see Details.}

\item{tileH, tileV}{\code{numeric} or \code{character}. Horizontal and 
vertical tile number(s) (e.g., \code{tileH = 1:5}), see 
\url{https://nsidc.org/data/docs/daac/mod10_modis_snow/landgrid.html}. 
Ignored if 'extent' is specified.}

\item{buffer}{\code{numeric} (in map units). Buffers the specified 'extent', 
negative values are allowed. If 'extent' is a vector object (\code{Spatial*} 
or \code{character} name of a map object), only one value is allowed (e.g., 
\code{buffer = 0.5}) and \code{\link{gBuffer}} is used. In all other cases, 
also \code{buffer = c(x, y)} can be specified.}

\item{system}{\code{character}, defaults to \code{"MODIS"}. Available 
alternatives are \code{"MERIS"} and \code{"SRTM"} (see Note).}

\item{zoom}{\code{logical}, defaults to \code{TRUE}. The interactive mode is 
only activated if no other spatial extent (i.e., 'extent', 'tileH', 'tileV') 
is specified. If \code{zoom = TRUE}, the first two clicks on the map are 
defining the zoom-in area, and the next two clicks are the download area. For 
large areas you can set \code{zoom = FALSE}.}
}
\value{
A \code{MODISextent} object.
}
\description{
Get MODIS, MERIS, or SRTM tile ID(s) for a specific geographic area.
}
\details{
\describe{
\tabular{ll}{
  \code{extent}:\cr
  \cr 
  
  If \code{list}:\cr 
  \tab Then LatLon coordinates in the following form:\cr 
  \tab \code{list(xmin = numeric, xmax = numeric, ymax = numeric, ymin = numeric)}.\cr
  \cr
  If \code{character}:\cr
  \tab The country name of a \code{map} object (see \code{\link{map}}), you 
  can use \code{\link{search4map}} to find a map by regular expression.\cr
  \tab Or the file name (plus path) of a raster image or ESRI shapefile (shp).\cr
  \cr
  Other:\cr
  \tab If \code{Raster*}.\cr
  \tab Using a \code{Raster*} object as \code{extent}, the function 
  automatically determines the extent, resolution, and projection. This will 
  be used by \code{\link{runGdal}} creating perfectly matching files. If the 
  raster file has not a valid CRS string, LatLon is assumed.\cr
\tab Using an \code{extent} object, it must be in LatLon, as the extent 
object has no projection information attached.\cr
\tab If ESRI shapefile (shp) or \code{map} object, a call to 
\code{\link{over}} is performed to determine the MODIS tile containg the 
extent. This often considerably reduces the number of required tiles, but can 
lead to NO-DATA areas if not all tiles had to be downloaded in the bounding 
box of 'extent'! 
}
}

PS:\cr
If an extent is specified through \code{tileV, tileH} arguments, no cropping 
is performed and the full un-cutted tile(s) (if more than one then also 
mosaicked) is/are processed!
}
\note{
\strong{MODIS} does no longer support the automated download of MERIS and 
SRTM data. At least as far as the latter is concerned, easy data access is 
granted through \code{\link{getData}}. Despite this limitation, 
\code{\link{getTile}} may still be used to identify MERIS and SRTM tile IDs 
for manual download based on a user-defined spatial extent.
}
\examples{
\dontrun{
# ex 1 ############
# drawing the extent. NOTE: It is not possible to draw a date-line crossing area!
# draw extent with zoom, for smaller extents
getTile()

# ex 2 ############
# 'extent' specified with a 'Spatial*' object (taken from ?rgdal::readOGR)
dsn <- system.file("vectors/Up.tab", package = "rgdal")[1]
Up <- rgdal::readOGR(dsn, "Up")
getTile(extent = Up)

# ex 3 ############
# with 'tileH' and 'tileV'
getTile(tileH = 18:19, tileV = 4)

# ex 4 ############
# with 'extent' of class 'list'
Austria <- list(ymin = 46.12, ymax = 49.3, xmin = 9.2, xmax = 17.47)
getTile(extent = Austria)
getTile(extent = Austria, buffer = 10)
getTile(extent = Austria, buffer = c(0, 10))  # x, y

# ex 5 ############
# with 'extent' or 'Raster*' object from "raster" package
rasterObject <- raster(xmn = 9.2, xmx = 17.47, ymn = 46.12, ymx = 49.3, 
                       crs = "+init=epsg:4326")
getTile(extent = rasterObject)
getTile(extent = extent(rasterObject))

# also works for projected data
rasterObject2 <- projectExtent(rasterObject, crs = "+init=epsg:32633")
getTile(extent = rasterObject2)

# ex 6 #################
# Character name of a map contained in map("worldHires", plot = FALSE)$names
getTile(extent = "Austria")
getTile(extent = c("Austria", "Germany"))

# Search for specific map name patterns (use with caution):
m1 <- search4map("Per")
getTile(extent = m2)

# Or use 'map' objects directly (remember to use map(..., fill = TRUE)): 
m2 <- map("state", region = c("new york", "new jersey", "penn"), fill = TRUE)
getTile(extent = m2)

# SRTM and MERIS data
getTile(extent = c("austria", "germany", "switzerland"), system = "SRTM")
getTile(extent = c("austria", "germany", "switzerland"), system = "MERIS")
}

}
\author{
Matteo Mattiuzzi
}
\seealso{
\code{\link{extent}}, \code{\link{map}}, \code{\link{search4map}}.
}

