% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mb.predict-class.R
\name{plot.mb.predict}
\alias{plot.mb.predict}
\title{Plots predicted responses from a time-course MBNMA model}
\usage{
\method{plot}{mb.predict}(
  x,
  disp.obs = FALSE,
  overlay.ref = TRUE,
  overlay.nma = NULL,
  method = "random",
  col = "blue",
  max.col.scale = NULL,
  treat.labs = NULL,
  ...
)
}
\arguments{
\item{x}{An object of class \code{"mb.predict"} generated by
\code{predict("mbnma")}}

\item{disp.obs}{A boolean object to indicate whether to show shaded sections
of the plot for where there is observed data (\code{TRUE}) or not (\code{FALSE})}

\item{overlay.ref}{A boolean object indicating whether to overlay a line
showing the median network reference treatment response over time on the
plot (\code{TRUE}) or not (\code{FALSE}). The network reference treatment (treatment
\enumerate{
\item must be included in \code{predict} for this to display the network reference
treatment properly.
}}

\item{overlay.nma}{Can be used to overlay the predicted results from a standard NMA model that
"lumps" time-points together within the range specified in \code{overlay.nma}. Must be a numeric vector of length 2, or
left as \code{NULL} (the default) to indicate no NMA should be performed. \code{overlay.nma} can only be specified if
\code{overlay.ref==TRUE}. See Details for further information.}

\item{method}{Can take \code{"common"} or \code{"random"} to indicate the type of NMA model used to synthesise data points
given in \code{overlay.nma}. The default is \code{"random"} since this assumes different
time-points in \code{overlay.nma} have been lumped together to estimate the NMA.}

\item{col}{A character indicating the colour to use for shading if \code{disp.obs}
is set to \code{TRUE}. Can be either \code{"blue"}, \code{"green"}, or \code{"red"}}

\item{max.col.scale}{Rarely requires adjustment. The maximum count of
observations (therefore the darkest shaded color) only used if \code{disp.obs} is
used. This allows consistency of shading between multiple plotted graphs.
It should always be at least as high as the maximum count of observations
plotted}

\item{treat.labs}{A vector of treatment labels in the same order as treatment codes.
Easiest to use treatment labels stored by \code{mb.network()}}

\item{...}{Arguments for \code{ggplot()}}
}
\description{
Plots predicted responses from a time-course MBNMA model
}
\details{
For the S3 method \code{plot()}, if \code{disp.obs} is set to \code{TRUE} it is
advisable to ensure predictions in \code{predict} are estimated using an even
sequence of time points to avoid misrepresentation of shaded densities.
Shaded counts of observations will be relative to the treatment plotted in
each panel rather than to the network reference treatment if \code{disp.obs} is
set to \code{TRUE}.

\code{overlay.nma} can be useful to assess if the MBNMA predictions are in agreement with predictions from an NMA model
for a specific range of time-points. This can be a general indicator of the fit of the time-course model. However, it
is important to note that the wider the range specified in \code{overlay.nma}, the more likely it is that different time-points
are included, and therefore that there is greater heterogeneity/inconsistency in the NMA model. If \code{overlay.nma} includes
several follow-up times for any study then only a single time-point will be taken (the one closest to \code{mean(overlay.nma)}).
The NMA predictions are plotted over the range specified in \code{overlay.nma} as a horizontal line, with the 95\%CrI shown by a grey
rectangle. The NMA predictions represent those for \emph{any time-points within this range} since they lump together data at
all these time-points. Predictions for treatments that are disconnected from
the network reference treatment at data points specified within \code{overlay.nma} cannot be estimated so are not included.

It is important to note that the NMA model is not necessarily the "correct" model, since it "lump" different time-points
together and ignores potential differences in treatment effects that may arise from this. The wider the range specified in
\code{overlay.nma}, the greater the effect of "lumping" and the stronger the assumption of similarity between studies.
}
\examples{
\donttest{
# Create an mb.network object from a dataset
copdnet <- mb.network(copd)

# Run an MBNMA model with a log-linear time-course
loglin <- mb.run(copdnet,
  fun=tloglin(pool.rate="rel", method.rate="common"),
  rho="dunif(0,1)", covar="varadj")

# Predict responses using the original dataset to estimate the network reference
#treatment response
df.ref <- copd[copd$treatment=="Placebo",]
predict <- predict(loglin, times=c(0:20), E0=0, ref.resp=df.ref)

# Plot the predicted responses with observations displayed on plot as green shading
plot(predict, disp.obs=TRUE, overlay.ref=FALSE, col="green")

# Plot the predicted responses with the median network reference treatment response overlayed
#on the plot
plot(predict, disp.obs=FALSE, overlay.ref=TRUE)

# Plot predictions from an NMA calculated between different time-points
plot(predict, overlay.nma=c(5,10), n.iter=20000)
plot(predict, overlay.nma=c(15,20), n.iter=20000)
# Time-course fit may be less good at 15-20 weeks follow-up
}

}
