\name{LatticeKrig}
\alias{LatticeKrig}
\alias{LKrig}
\alias{MLE.LKrig}
\alias{predict.LKrig}
\alias{print.LKrig}
\alias{surface.LKrig}
\title{Spatial prediction and inference using a compactly supported
multi-resolution basis and a lattice model for the basis coefficients.
}
\description{
A variation of Kriging with fixed basis functions that uses the Wendland covariance kernel to 
create a regular set of basis functions on a grid. The coefficients of these
basis functions are modeled as a Markov random field (MRF).  
}
\usage{
LKrig(x,y=NULL, weights = rep(1, nrow(x)),Z=NULL,NC,lambda, LKinfo=NULL,
                        grid.info=NULL, alpha=1.0,a.wght= 5, beta=NULL, nlevel=1,
                        iseed=123,NtrA=20,
                        use.cholesky=NULL, return.cholesky=FALSE,
                        overlap=2.5,normalize=TRUE, edge=FALSE,
                        verbose=FALSE)
MLE.LKrig(x,y,...,par.grid=NULL,verbose=FALSE)
\method{predict}{LKrig}( object, xnew=NULL,Z=NULL, drop.Z=FALSE, ...)
\method{surface}{LKrig}( object, ...)
\method{print}{LKrig}( x, digits=4, ...)
}
\arguments{
  
  \item{alpha}{Weights of precision matrices for each level of resolution.}

  \item{a.wght}{The "a" spatial autoregessive parameter for a first order Markov Random field. This controls the spatial dependence and must be greater than or equal to 4. For \code{a.wght} = 4 normalize should be FALSE.
  If there are multiple levels this can be a vector.}

  \item{beta}{Another parameterization of the MRF dependence  beta = -1/a.wght. This parameter must be 
less than zero and larger or equal to -.25. (see a.wght>= 4 above.)}

  \item{digits}{Number of digits in printed output.}
   
  \item{drop.Z}{If true the fixed part will only be evaluated at the
polynomial part of the fixed model. The contribution from the other
covariates will be omitted.}

  \item{edge}{If TRUE an adjustment is made in precision matrices for edge effects.}

  \item{grid.info}{A list with information to construct the regular grid for the basis function centers.
  The components are: \code{xmin, xmax, ymin,ymax, delta}. If not passed this list is created by
  \code{LKrig.setup} from \code{x} and \code{NC}.}

 \item{iseed}{Random seed used in the Monte Carlo technique
  for approximating the effective degrees of freedom (trace of the
  smoothing matrix) for the GCV criterion. If NA, no seed is set. }

 \item{lambda}{The ratio of the nugget variance (called sigma squared in \code{fields} and \code{LatticeKrig}) to the parameter 
  controlling the marginal variance of the process (called rho in
  \code{fields} and \code{LatticeKrig}).} 

 \item{LKinfo}{A list that is an alternate form of specifying the multiresolution basis. This is usually 
        created by the function \code{LKrig.setup}. If NULL, this list is created and returned as 
a component of the LKrig object. }

   \item{NC}{Maximum number of grid points in one dimension. If the
domain is rectangular, the smaller dimension will have less than NC
points.}

  \item{nlevel}{Number of levels for the multiresolution basis.}

  \item{normalize}{If \code{TRUE} basis functions are normalized so that the 
    marginal variance of the process covariance is constant and equal to rho. This
    normalization avoids some edge and wavy artifacts from using a discrete
    set of basis functions.}

  \item{NtrA}{Number of random samples used in Monte Carlo method for
determining effective degrees of freedom. }
  
  \item{object}{The \code{LKrig} object.}

  \item{overlap}{The overlap between basis functions. This scaling is based on centers being spacing 
1 unit distance apart and the Wendland function decreasing to zero at 1 unit distance. A scaling/overlap of 2.5 (the default)
implies that the support of the basis functions will extend to a disc of radius 2.5. We recommend that this 
parameter not be changed. }

  \item{par.grid}{Parameter grid for maximum likelihood search. A matrix with columns
   \code{alpha}, \code{a.wght} and \code{lambda}. Currently, multiresolution covariances are not 
   supported.}

  \item{return.cholesky}{If TRUE the cholesky decomposition is included in the output list
    (with the name \code{Mc}). This is option is  used with a subsequent call to
     \code{LKrig} with \code{use.cholesky}. }

  \item{verbose}{If \code{TRUE} print out results as they are computed in loop.}
  
  \item{use.cholesky}{Use the symbolic part of the Cholesky decomposition passed in  this argument.}

  \item{weights}{A vector that is proportional to the reciprocal
  variances of the errors. I.e. errors are assumed to be uncorrelated
  with variances sigma^2/weights. }

  \item{xnew}{Matrix of locations for prediction.}
 
  \item{x}{Spatial locations of observations.}

  \item{y}{Spatial observations.}

  \item{Z}{Linear covariates to be included in fixed part of the model
  that are distinct from the default first order polynomial in \code{x}.}
 
  \item{\dots}{Additional optional arguments or for
  \code{MLE.LKrig} the arguments to LKrig (see details below).}
}
\details{
This is an experimental function that combines compactly supported basis
functions and a Markov random field covariance model to provide
spatial analysis for very large data sets.  The model for the spatial
field is
 
 f(x) = P(x) + Z d + sum Phi.j(x) c.j

and the data model is

 Y.k = f(x.k) + e.k

with e.k uncorrelated normal errors with variance sigma^2.

Where P is a first order linear polynomial, Z a matrix of covariates, d the fixed coefficients, Phi.j are
 fixed basis functions and c.j are random coefficients. The basis
 functions are two dimensional radial basis functions (RBF) proportional to 
 the Wendland covariance. There is one important difference in the basis construction
 that makes this different from a simple radial basis function specification and this is 
 described below. 
 For a single level (\code{nlevel ==1}) the
 RBFs centered at the grid points and with radial support
 \code{delta*overlap} where \code{delta} is the spacing between grid
 points and overlap has the default value of 2.5. For multi-resolution
 each subsequent level is based on a grid with delta divided by
 2. (See example below.)

The prior on c is a multivariate normal, with a mean of zero and the
inverse of the covariance matrix, Q, also known as the precision
matrix. Q is assumed to be block diagonal corresponding to the
organization of the basis functions into levels of resolution.  If
\code{nlevels} are specified, the ith block is a precision matrix
based on a first order spatial autoregression with \code{beta[i]}
being the autoregressive parameter. The specific precision matrix for
each block (level), Q.i, is implemented by
\code{LK.MRF.precision}. Briefly, this matrix has the form Q.i=
t(B)\%*\%B and is multiplied by the relative weight \code{alpha[i]}.
Each row of B, corresponding to a point in the lattice in the
interior, is "a" (\code{a.wght[i]}) on the diagonal and  -1 at each of the
four nearest neighbors of the lattice point. Points on the edges just
have less neighbors if \code{edge} is set to FALSE.
Following the notation in Lundgren and Rue  a= 4 + k2 with k2 greater than or equal to  0.
Some schematics for filling in the B matrix are:
\preformatted{

       -1                -1                    4+k2  -1
                   
  -1  4+k2  -1           4+k2  -1               -1

       -1                -1

Interior point        Left edge when           Upper left corner when 
                      edge is FALSE            edge is FALSE
}
 

If \code{edge} is TRUE the edges are weighted to reflect other boundary conditions. 
\preformatted{

       -1                -.5                    1+k2/4  -.5
                   
  -1  4+k2  -1           2+k2/2  -1               -.5

       -1                -.5

Interior point        Left edge when           Upper left corner when 
                      edge is TRUE            edge is TRUE
}
 
 The assumption is that for the coefficients
associated with the ith block of basis functions (c.i), t(B)\%*\%c.i
will be uncorrelated.  This description is a spatial autoregressive model 
(SAR). The matrix Q will of course have more nonzero values than B and the entries in Q can be identified as the
weights for a conditional autoregressive model (CAR). The CAR specification 
defines the neighborhood such that the Markov property holds. 
Values for \code{a.wght[i]} that are greater than 4 give reasonable covariance models.
Moerover setting setting a.wght to 4 and normalize to FALSE in the call to LKrig will
give a thin-plate spline type  model that does not have a range parameter. 

The overall design of this function does not require that the centers actually define a grid,
but then the construction of the matrix B may need to modified in the
internal function \code{LK.MRF.precision}. The reader is referred to
the function \code{LK.cov} for an explicit code that computes the
implied covariance function for the process f. 

Note that consistent with other fields functions, the two main
parameters in the model, sigma^2 and rho are parameterized as lambda
= sigma^2/rho and rho.

Given this model, it is possible to compute the conditional expectation of f
given Y and also the profile likelihood for lambda, alpha and
a.wght. Because both the basis functions and Q are sparse, it is possible
to compute the estimates and likelihood for large numbers of spatial
locations. This model has an advantage over covariance tapering or
compactly supported covariance functions (e.g. fastTps), because the
implied covariance functions can have longer range correlations.

The unnormalized basis functions result in a covariance that has some non-stationary artifacts
(see example below) and so the basis functions by default are scaled so that the implied 
covariance has a marginal variance that is equal to 1.0. This makes the basis functions
dependent on the choice of Q and results in some extra overhead for computation. But we believe it
is useful to avoid obvious artifacts resulting from the discretization used in the basis function model.
To examine what these artifacts are like, here is a simple figure of the marginal varainces for a 6 by 6 basis:
\preformatted{
  ginfo<- list( xmin=-1, xmax=1, ymin=-1, ymax=1, delta= 2/(6-1))
# specify covariance without normalization
  LKinfo<- LKrig.setup(grid.info=ginfo, nlevel=1, a.wght=4.5,alpha=1,
                       normalize= FALSE,edge=FALSE)
  xg<- make.surface.grid( list(x=seq(-1,1,,50), y= seq(-1,1,,50)))
  look<- LKrig.cov( xg, LKinfo=LKinfo,marginal =TRUE)
# surface plot of the marginal variances of the process.
  image.plot( as.surface(xg, look))
# basis function centers:
  xgrid<- seq(ginfo$xmin, ginfo$xmax, ginfo$delta)
  ygrid<- seq(ginfo$ymin, ginfo$ymax, ginfo$delta)
  points( expand.grid( xgrid, ygrid))
}

The function \code{LK.cov} is in the form to be used with \code{mKrig}
or \code{Krig} and is largely for checking and examining the implied
covariance function based on the grid and the values of \code{nlevel},
\code{alpha}, \code{beta}. 

\code{LKrig} Find spatial process estimate for fixed covariance specificed by 
 nlevel, alpha, a.wght (or beta), NC (or delta) and lambda.

\code{MLE.LKrig} Simple grid search over lambda, alpha and a.wght and
computes profile likelihood and GCV function.
}
\value{
\code{LKrig:} 
An LKrig class object with components for evaluating the estimate at arbitrary locations, 
describing the fit and as an option ( \code{Mc.return=TRUE}) the Cholesky decomposition to allow for fast updating with new values of lambda, alpha and a.wght. 
The "symbolic" first step in the sparse Cholesky decomposition can also be used to compute the sparse Cholesky decomposition for a different positive definite 
matrix that has the same pattern of zeroes. This option is useful in computing the likelihood under different covariance parameters. 

\code{MLE.LKrig:}
A matrix where each row is the result of evaluating the fit at specific values for a.wght and lambda.
Columns are: a.wght, lambda,  trA (Estimated Effective degrees of freedom in fit), SEtrA (Standard of trA estimate), sigmaMLE (MLE for sigma for fixed lambda and a.wght), 
rhoMLE ( MLE for rho for fixed lambda and a.wght), lnProfileLike 
(ln likelihood having maximized over rho  and fixed part with lambda and a.wght fixed.), GCV (GCV function at lambda and beta).


The component \code{LKinfo} is a list with the information that describes the layout of the multiresolution basis functions, which can be reconstructed with this list.
(see \code{LK.basis} as an example.)

\code{grid.info} A list with components xmin, xmax, ymin, ymax and
delta that specify ranges and spacing for first level of grid points
that are the centers of the basis functions. If NULL the ranges of the
coordinate locations are used and delta is determined from \code{NC}.
}
\author{
Doug Nychka
}

\seealso{
mKrig, Krig, fastTps, Wendland, LKrig.coef, Lkrig.lnPlike, LK.MRF.precision, LK.precision
}
\examples{
# Load ozone data set
  data(ozone2)  
  x<-ozone2$lon.lat
  y<- ozone2$y[16,]
# Find location that are not 'NA'.
# LKrig is not set up to handle missing observations.   
  good <-  !is.na( y)
  x<- x[good,]
  y<- y[good]
# Visualize the ozone data set
  quilt.plot(x,y)
  US(add=TRUE)

# Predict ozone concentrations over the domain using a single level lattice 
# with a maximum of 20 grid points in x and y direction.
  obj<- LKrig(x,y,NC=30, lambda=.01, a.wght=5)
# Plot fitted surface.
  surface(obj) # see also predict.surface to just evaluate on a grid
  US(add=TRUE)
# Check effective degrees of freedom in surface.
  obj$trA.est

# Search over lambda and a.wght to find MLE.
# (these ranges have been selected to give a nice result)
  NC<- 30
  a.wght<- 4+ 1/(seq(1,5,,8))**2
  lambda<- 10**seq( -6,-2,,8)*NC**2
  alpha<- 1
  out<- MLE.LKrig( x=x,y=y,NC=NC,alpha=alpha,a.wght=a.wght, lambda=lambda)
# Interpolate these values with Tps and find MLE
  temp<-Tps(cbind( log10(out[,3]), 1/sqrt(out[,2]-4)) , out$lnProfileLike,lambda=0)
  out.p<- predict.surface( temp)
# Plot ln profile likelihood  
  image.plot( out.p, ylab="equivalent lattice range: 1/sqrt(a-4)",
                           xlab="log10 lambda")
  points( temp$x, pch="o", col="grey50")
  title("ln Profike Likelihood")
# 'max.image' is a handy little function to find maximum in an image object
  max.image<- function( obj){
  ind<- which.max( obj$z)
  ix<- row(obj$z)[ind]
  iy<- col(obj$z)[ind]
  list(x= obj$x[ix], y=obj$y[iy], z= obj$z[ind], ind= cbind( ix,iy))}
# Apply 'max.image' to find MLE over lambda and a.wght grid
  max.out<- max.image(out.p)
  points( max.out$x, max.out$y, pch=1, col="magenta")
  contour( out.p, level= max.out$z - 2, add=TRUE, col="magenta", lwd=2)
# Refit surface using MLE of lambda and a.wght
  objMLE<- LKrig( x,y,NC=NC, lambda=10**max.out$x, alpha=1, a.wght= 4 + 1/max.out$y^2 )
  set.panel(2,1)
  surface(obj,zlim=c(0,220)) 
  title("Original Fit")
  US( add=TRUE)
  surface(objMLE,zlim=c(0,220)) 
  title("MLE Fit")
  US( add=TRUE)
# Note how the MLE fit is much smoother due to the larger value of lambda 

## Another example, this time with a covariate:
  data(COmonthlyMet)
  y<- CO.tmin.MAM.climate
  good<- !is.na( y)
  y<-y[good]
  x<- as.matrix(CO.loc[good,])
  Z<- CO.elev[good]
  out<- LKrig( x,y,Z=Z,NC=30, lambda=.1, a.wght=5)
  set.panel(2,1)
# Plot data
  quilt.plot(x,y)
  US(add=TRUE)
  title("Minimum spring temperature in Colorado")
# Plot elevation
  quilt.plot(x,Z)
  US(add=TRUE)
  title("Elevation at observation locations")
# Plot predictions with linear elevation term included
  quilt.plot( x, predict(out),zlim=c(-12,12))
  title("Predicted temperature with elevation term")
  US(add=TRUE)
# Plot predictions without linear elevation term included
  quilt.plot( x, predict(out, drop.Z=TRUE),zlim=c(-12,12))
  title("Predicted temperature without elevation term")
  US(add=TRUE)
# Note the larger gradients when elevation is included  
  set.panel()
  
# A bigger problem: fitting takes about 30 seconds on fast laptop
\dontrun{
  data(CO2)
  obj1<- LKrig( CO2$lon.lat,CO2$y,NC=100, lambda=5, a.wght=5)
# 4600 basis functions 100X46 lattice.
  obj1$trA.est # about 1040 effective degrees of freedom 
#
  glist<- list( x= seq( -180,180,,200),y=seq( -80,80,,100) )
  xg<-  make.surface.grid(glist)
  fhat<- predict( obj1, xg)
  fhat <- matrix( fhat,200,100) # convert to image
#Plot data and gap-filled estimate
  set.panel(2,1)
  quilt.plot(CO2$lon.lat,CO2$y,zlim=c(373,381))
  world(add=TRUE,col="magenta")
  title("Simulated CO2 satellite observations")
  image.plot( glist$x, glist$y, fhat,zlim=c(373,381))
  world( add=TRUE, col="magenta")
  title("Gap-filled global predictions")
}  

set.panel()
 
# Here is an illustration of using the fields function mKrig with this covariance
# to reproduce the computations of LKrig. The difference is that mKrig can
# not take advantage of any sparsity in the precision matrix 
# as the inverse covariance matrix may not be sparse.
# But this example reinforces the concept that LKrig finds the
# the standard geostatistical estiamte but just uses a
# particular covariance function
#
  a.wght<- 5
  lambda <-  1.5
  obj1<- LKrig( x,y,NC=16, lambda=lambda, a.wght=5, NtrA=20,iseed=122)
 
# in both calls iseed is set the same so we can compare 
# Monte Carlo estimates of effective degrees of freedom
  obj2<- mKrig( x,y, lambda=lambda, m=2, cov.function="LKrig.cov",
                      cov.args=list( LKinfo=obj1$LKinfo), NtrA=20, iseed=122)
# The covariance "parameters" are all in the list LKinfo
# to create this outside of a call to LKrig use
  LKinfo.test <- LKrig.setup( x, NC=16,  a.wght=5)

# compare the two results this is also an
# example of how tests are automated in fields
# set flag to have tests print results
  test.for.zero.flag<- TRUE
  test.for.zero( obj1$fitted.values, obj2$fitted.values,
                  tag="comparing predicted values LKrig and mKrig")
  test.for.zero( unlist(LKinfo.test), unlist(obj1$LKinfo),
                  tag="comparing two ways of creating covariance list")
}
% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{spatial}

