% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lmn_suff.R
\name{lmn_suff}
\alias{lmn_suff}
\title{Calculate the sufficient statistics of an LMN model.}
\usage{
lmn_suff(Y, X, V, Vtype, npred = 0)
}
\arguments{
\item{Y}{An \verb{n x q} matrix of responses.}

\item{X}{An \verb{N x p} matrix of covariates, where \code{N = n + npred} (see \strong{Details}). May also be passed as:
\itemize{
\item A scalar, in which case the one-column covariate matrix is \code{X = X * matrix(1, N, 1)}.
-\code{X = 0}, in which case the mean of \code{Y} is known to be zero, i.e., no regression coefficients are estimated.
}}

\item{V, Vtype}{The between-observation variance specification.  Currently the following options are supported:
\itemize{
\item \code{Vtype = "full"}: \code{V} is an \verb{N x N} symmetric positive-definite matrix.
\item \code{Vtype = "diag"}: \code{V} is a vector of length \code{N} such that \code{V = diag(V)}.
\item \code{Vtype = "scalar"}: \code{V} is a scalar such that \code{V = V * diag(N)}.
\item \code{Vtype = "acf"}: \code{V} is either a vector of length \code{N} or an object of class \code{\link[SuperGauss:Toeplitz]{SuperGauss::Toeplitz}}, such that \code{V = toeplitz(V)}.
}

For \code{V} specified as a matrix or scalar, \code{Vtype} is deduced automatically and need not be specified.}

\item{npred}{A nonnegative integer.  If positive, calculates sufficient statistics to make predictions for new responses. See \strong{Details}.}
}
\value{
An S3 object of type \code{lmn_suff}, consisting of a list with elements:
\describe{
\item{\code{Bhat}}{The \eqn{p \times q}{p x q} matrix \eqn{\hat{\boldsymbol{B}} = (\boldsymbol{X}'\boldsymbol{V}^{-1}\boldsymbol{X})^{-1}\boldsymbol{X}'\boldsymbol{V}^{-1}\boldsymbol{Y}}{B_hat = (X'V^{-1}X)^{-1}X'V^{-1}Y}.}
\item{\code{T}}{The \eqn{p \times p}{p x p} matrix \eqn{\boldsymbol{T} = \boldsymbol{X}'\boldsymbol{V}^{-1}\boldsymbol{X}}{T = X'V^{-1}X}.}
\item{\code{S}}{The \eqn{q \times q}{q x q} matrix \eqn{\boldsymbol{S} = (\boldsymbol{Y} - \boldsymbol{X} \hat{\boldsymbol{B}})'\boldsymbol{V}^{-1}(\boldsymbol{Y} - \boldsymbol{X} \hat{\boldsymbol{B}})}{S = (Y-X B_hat)'V^{-1}(Y-X B_hat)}.}
\item{\code{ldV}}{The scalar log-determinant of \code{V}.}
\item{\code{n}, \code{p}, \code{q}}{The problem dimensions, namely \code{n = nrow(Y)}, \code{p = nrow(Beta)} (or \code{p = 0} if \code{X = 0}), and \code{q = ncol(Y)}.}
}
In addition, when \code{npred > 0} and with \eqn{\boldsymbol{x}}{x}, \eqn{\boldsymbol{w}}{w}, and \eqn{v} defined in \strong{Details}:
\describe{
\item{\code{Ap}}{The \verb{npred x q} matrix \eqn{\boldsymbol{A}_p = \boldsymbol{w}'\boldsymbol{V}^{-1}\boldsymbol{Y}}{A_p = w'V^{-1}Y}.}
\item{\code{Xp}}{The \verb{npred x p} matrix \eqn{\boldsymbol{X}_p = \boldsymbol{x} - \boldsymbol{w}\boldsymbol{V}^{-1}\boldsymbol{X}}{X_p = x - w'V^{-1}X}.}
\item{\code{Vp}}{The scalar \eqn{V_p = v - \boldsymbol{w}\boldsymbol{V}^{-1}\boldsymbol{w}}{V_p = v - w'V^{-1}w}.}
}
}
\description{
Calculate the sufficient statistics of an LMN model.
}
\details{
The multi-response normal linear regression model is defined as
\deqn{
\boldsymbol{Y} \sim \textrm{Matrix-Normal}(\boldsymbol{X}\boldsymbol{B}, \boldsymbol{V}, \boldsymbol{\Sigma}),
}{
Y ~ Matrix-Normal(X B, V, \Sigma),
}
where \eqn{\boldsymbol{Y}_{n \times q}}{Y_(n x q)} is the response matrix, \eqn{\boldsymbol{X}_{n \times p}}{X_(n x p)} is the covariate matrix, \eqn{\boldsymbol{B}_{p \times q}}{B_(p x q)} is the coefficient matrix, \eqn{\boldsymbol{V}_{n \times n}}{V_(n x n)} and \eqn{\boldsymbol{\Sigma}_{q \times q}}{\Sigma_(q x q)} are the between-row and between-column variance matrices, and the Matrix-Normal distribution is defined by the multivariate normal distribution
\eqn{
\textrm{vec}(\boldsymbol{Y}) \sim \mathcal{N}(\textrm{vec}(\boldsymbol{X}\boldsymbol{B}), \boldsymbol{\Sigma} \otimes \boldsymbol{V}),
}{
vec(Y) ~ N( vec(X B), \Sigma \%x\% V ),
}
where \eqn{\textrm{vec}(\boldsymbol{Y})}{vec(Y)} is a vector of length \eqn{nq} stacking the columns of of \eqn{\boldsymbol{Y}}{Y}, and \eqn{\boldsymbol{\Sigma} \otimes \boldsymbol{V}}{\Sigma \%x\% V} is the Kronecker product.

The function \code{lmn_suff()} returns everything needed to efficiently calculate the likelihood function
\deqn{\mathcal{L}(\boldsymbol{B}, \boldsymbol{\Sigma} \mid \boldsymbol{Y}, \boldsymbol{X}, \boldsymbol{V}) = p(\boldsymbol{Y} \mid \boldsymbol{X}, \boldsymbol{V}, \boldsymbol{B}, \boldsymbol{\Sigma}).
}{
L(B, \Sigma | Y, X, V) = p(Y | X, V, B, \Sigma).
}

When \code{npred > 0}, define the variables \code{Y_star = rbind(Y, y)}, \code{X_star = rbind(X, x)}, and \code{V_star = rbind(cbind(V, w), cbind(t(w), v))}.  Then \code{lmn_suff()} calculates summary statistics required to estimate the conditional distribution
\deqn{
p(\boldsymbol{y} \mid \boldsymbol{Y}, \boldsymbol{X}_\star, \boldsymbol{V}_\star, \boldsymbol{B}, \boldsymbol{\Sigma}).
}{
p(y | Y, X_star, V_star, B, \Sigma).
}
The inputs to \code{lmn_suff()} in this case are \code{Y = Y}, \code{X = X_star}, and \code{V = V_star}.
}
\examples{
# Data
n <- 50
q <- 3
Y <- matrix(rnorm(n*q),n,q)

# No intercept, diagonal V input
X <- 0
V <- exp(-(1:n)/n)
lmn_suff(Y, X = X, V = V, Vtype = "diag")

# X = (scaled) Intercept, scalar V input (no need to specify Vtype)
X <- 2
V <- .5
lmn_suff(Y, X = X, V = V)

# X = dense matrix, Toeplitz variance matrix
p <- 2
X <- matrix(rnorm(n*p), n, p)
Tz <- SuperGauss::Toeplitz$new(acf = 0.5*exp(-seq(1:n)/n))
lmn_suff(Y, X = X, V = Tz, Vtype = "acf")

}
