#' Linear Permuted Williams Transformation
#'
#' \code{LPWT} returns a maximin distance LHD matrix generated by linear permuted williams transformation, along with its maximum minimum L_1 distance
#'
#' @param n A positive integer, which stands for the number of rows (or run size).
#' @param k A positive integer, which stands for the number of columns (or factor size). \code{k} must be smaller than \code{n}. In GLP designs, \code{k} <= the total number of positive integers that are smaller than and coprime to \code{n}.
#' @param h A vector whose length is \code{k}, with its elements that are smaller than and coprime to \code{n}. The default is set to be a random sample of \code{k} elements between 1 and \code{n-1}.
#' @param method A distance measure method. The default setting is "manhattan", and it could be one of the following: "euclidean", "maximum", "manhattan", "canberra", "binary" or "minkowski". Any unambiguous substring can be given.
#'
#' @return If all inputs are logical, then the output will be a list, which contains a \code{n} by \code{k} LHD and its maximum minimum L_1 distance. This is the construction method in section 2.1 of the reference, and it is particularly useful when \code{n} is prime and \code{k} equals \code{n-1}.
#'
#' @references Wang, L., Xiao, Q., and Xu, H. (2018)  Optimal maximin $L_{1}$-distance Latin hypercube designs based on good lattice point designs. \emph{The Annals of Statistics}, \strong{46}(6B), 3741-3766.
#'
#' @examples
#' #generate a 11 by 10 maximin distance LHD with default setting
#' try=LPWT(n=11,k=10)
#' try
#' phi_p(try[[1]])   #calculate the phi_p of "try".
#'
#' #Another example
#' #generate a 13 by 12 maximin distance LHD with given h vector
#' try2=LPWT(n=13,k=12,h=1:12)
#' try2
#' phi_p(try2[[1]])   #calculate the phi_p of "try2".
#'
#' @export

LPWT=function(n,k,h=sample(seq(from=1,to=(n-1)),k),method="manhattan"){
  ### search over 0 to n-1 mod n of the basedesign given the run size: n;
  #This is the step 3: search for the best b.
  #n and k are the run and factor sizes.
  #k <= the number of positive integers that are smaller than and coprime to n
  #h must be a vector whose length is k, with elements that are smaller than and coprime to n
  #method: the distance method, with default setting is "manhattan".

  if(k>=n){
    stop("k must be smaller than n")
  }

  basedesign=GLP(n=n,k=k,h=h)    #starting design: D_b with b=0

  resultdesign=WT(basedesign,baseline=0)   #E_b with b=0

  minvalue=min(stats::dist(resultdesign, method = method))  #current minimum distance

  for(c in 1:(n-1))   #for b=1, ... , n-1
  {
    newdesign = (basedesign + c) %% n      #new D_b
    newresult = WT(newdesign,baseline=0)   #new E_b
    newminvalue = min(stats::dist(newresult, method = method))
    if (newminvalue > minvalue)          #updating minimum distance
    {
      resultdesign = newresult
      minvalue = newminvalue
    }
  }

  list(resultdesign, minvalue)
}
