#' Extract results, conduct posterior inference and compute performance metrics for MCMC samples of models from the IMIFA family
#'
#' This function post-processes simulations generated by \code{\link{mcmc_IMIFA}} for any of the IMIFA family of models. It can be re-ran at little computational cost in order to extract different models explored by the sampler used for \code{sims}, without having to re-run the model itself. New results objects using different numbers of clusters and different numbers of factors (if visited by the model in question), or using different model selection criteria (if necessary) can be generated with ease. The function also performs post-hoc corrections for label switching, as well as post-hoc Procrustes rotation of loadings matrices and scores, to ensure sensible posterior parameter estimates, and constructs credible intervals.
#' @param sims An object of class "\code{IMIFA}" generated by \code{\link{mcmc_IMIFA}}.
#' @param burnin Optional additional number of iterations to discard. Defaults to 0, corresponding to no burnin.
#' @param thinning Optional interval for extra thinning to be applied. Defaults to 1, corresponding to no thinning.
#' @param G If this argument is not specified, results will be returned with the optimal number of clusters. If different numbers of clusters were explored in \code{sims} for the "\code{MFA}" or "\code{MIFA}" methods, supplying an integer value allows pulling out a specific solution with \code{G} clusters, even if the solution is sub-optimal. Similarly, this allows retrieval of samples corresponding to a solution, if visited, with \code{G} clusters for the "\code{OMFA}", "\code{OMIFA}", "\code{IMFA}" and "\code{IMIFA}" methods.
#' @param Q If this argument is non specified, results will be returned with the optimal number of factors. If different numbers of factors were explored in \code{sims} for the "\code{FA}", "\code{MFA}", "\code{OMFA}" or "\code{IMFA}" methods, this allows pulling out a specific solution with \code{Q} factors, even if the solution is sub-optimal. Similarly, this allows retrieval of samples corresponding to a solution, if visited, with \code{Q} factors for the "\code{IFA}", "\code{MIFA}", "\code{OMIFA}" and "\code{IMIFA}" methods.
#' @param criterion The criterion to use for model selection, where model selection is only required if more than one model was run under the "\code{FA}", "\code{MFA}", "\code{MIFA}", "\code{OMFA}" or "\code{IMFA}" methods when \code{sims} was created via \code{\link{mcmc_IMIFA}}. Note that these are \emph{all} calculated, this argument merely indicates which one will form the basis of the construction of the output. Note that the first three options here might exhibit bias in favour of zero-factor models for the finite factor "\code{FA}", "\code{MFA}", "\code{OMFA}" and "\code{IMFA}" methods and might exhibit bias in favour of one-cluster models for the "\code{MFA}" and "\code{MIFA}" methods.
#' @param G.meth If the object in \code{sims} arises from the "\code{OMFA}", "\code{OMIFA}", "\code{IMFA}" or "\code{IMIFA}" methods, this argument determines whether the optimal number of clusters is given by the mode or median of the posterior distribution of \code{G}. Defaults to "\code{Mode}".
#' @param Q.meth If the object in \code{sims} arises from the "\code{IFA}", "\code{MIFA}", "\code{OMIFA}" or "\code{IMIFA}" methods, this argument determines whether the optimal number of latent factors is given by the mode or median of the posterior distribution of \code{Q}. Defaults to "\code{Mode}".
#' @param dat The actual data set on which \code{\link{mcmc_IMIFA}} was originally run. This is necessary for computing error metrics between the estimated and empirical covariance matrix/matrices. If this is not supplied, the function will attempt to find the data set if it is still available in the global environment.
#' @param conf.level The confidence level to be used throughout for credible intervals for all parameters of inferential interest. Defaults to 0.95.
#' @param z.avgsim Logical indicating whether the clustering should also be summarised with a call to \code{\link{Zsimilarity}} by the clustering with minimum squared distance to the similarity matrix obtained by averaging the stored adjacency matrices, in addition to the MAP estimate. Note that the MAP clustering is computed \emph{conditional} on the estimate of the number of clusters (whether that be the modal estimate or the estimate according to \code{criterion}) and other parameters are extracted conditional on this estimate of \code{G}: however, in constrast, the number of distinct clusters in the summarised labels obtained by \code{z.avgsim=TRUE} may not necessarily coincide with the estimate of \code{G}, but may provide a useful alternative summary of the partitions explored during the chain. Please be warned that this can take considerable time to compute, and may not even be possible if the number of observations &/or number of stored iterations is large and the resulting matrix isn't sufficiently sparse, so the default is \code{FALSE}, otherwise both the summarised clustering and the similarity matrix are stored: the latter can be passed to \code{\link{plot.Results_IMIFA}}.
#' @param zlabels For any method that performs clustering, the true labels can be supplied if they are known in order to compute clustering performance metrics. This also has the effect of ordering the MAP labels (and thus the ordering of cluster-specific parameters) to most closely correspond to the true labels if supplied.
#'
#' @return An object of class "\code{Results_IMIFA}" to be passed to \code{\link{plot.Results_IMIFA}} for visualising results. Dedicated \code{print} and \code{summary} functions exist for objects of this class. The object, say \code{x}, is a list of lists, the most important components of which are:
#' \describe{
#' \item{Clust}{Everything pertaining to clustering performance can be found here for all but the "\code{FA}" and "\code{IFA}" methods, in particular \code{x$Clust$map}, the MAP summary of the posterior clustering. More detail is given if known \code{zlabels} are supplied: performance is always evaluated against the MAP clustering, with additional evaluation against the alternative clustering computed if \code{z.avgsim=TRUE}.}
#' \item{Error}{Error metrics (e.g. MSE) between the empirical and estimated covariance matrix/matrices.}
#' \item{GQ.results}{Everything pertaining to model choice can be found here, incl. posterior summaries for the estimated number of clusters and estimated number of factors, if applicable to the method employed. Information criterion values are also accessible here.}
#' \item{Means}{Posterior summaries for the means.}
#' \item{Loadings}{Posterior summaries for the factor loadings matrix/matrices. Posterior mean loadings given by x$Loadings$post.load are given the \code{\link[stats]{loadings}} class for printing purposes and thus the manner in which they are displayed can be modified.}
#' \item{Scores}{Posterior summaries for the latent factor scores.}
#' \item{Uniquenesses}{Posterior summaries for the uniquenesses.}
#' }
#' @export
#' @importFrom Rfast "med" "rowMaxs" "standardise" "colMaxs" "rowVars" "rowmeans" "Order" "cova" "Var"
#' @importFrom abind "adrop"
#' @importFrom e1071 "matchClasses" "classAgreement"
#' @importFrom mclust "classError"
#' @importFrom matrixStats "rowMedians" "rowQuantiles"
#'
#' @seealso \code{\link{mcmc_IMIFA}}, \code{\link{plot.Results_IMIFA}}, \code{\link{Procrustes}}, \code{\link{Zsimilarity}}
#' @references Murphy, K., Gormley, I. C. and Viroli, C. (2017) Infinite Mixtures of Infinite Factor Analysers: Nonparametric Model-Based Clustering via Latent Gaussian Models, \href{https://arxiv.org/abs/1701.07010}{arXiv:1701.07010}.
#'
#' @author Keefe Murphy
#'
#' @examples
#' # data(coffee)
#' # data(olive)
#'
#' # Run a MFA model on the coffee data over a range of clusters and factors.
#' # simMFAcoffee  <- mcmc_IMIFA(coffee, method="MFA", range.G=2:3, range.Q=0:3, n.iters=1000)
#'
#' # Accept all defaults to extract the optimal model.
#' # resMFAcoffee  <- get_IMIFA_results(simMFAcoffee)
#'
#'
#' # Instead let's get results for a 3-cluster model, allowing Q be chosen by aic.mcmc.
#' # resMFAcoffee2 <- get_IMIFA_results(simMFAcoffee, G=3, criterion="aic.mcmc")
#'
#' # Run an IMIFA model on the olive data, accepting all defaults.
#' # simIMIFAolive <- mcmc_IMIFA(olive, method="IMIFA", n.iters=10000)
#'
#' # Extract optimum results
#' # Estimate G & Q by the median of their posterior distributions
#' # Construct 90% credible intervals and try to return the similarity matrix.
#' # resIMIFAolive <- get_IMIFA_results(simIMIFAolive, G.meth="median", Q.meth="median",
#' #                                    conf.level=0.9, z.avgsim=TRUE)
#' # summary(resIMIFAolive)
get_IMIFA_results              <- function(sims = NULL, burnin = 0L, thinning = 1L, G = NULL, Q = NULL, criterion = c("bicm", "aicm", "log.iLLH", "dic", "bic.mcmc", "aic.mcmc"),
                                           G.meth = c("mode", "median"), Q.meth = c("mode", "median"), dat = NULL, conf.level = 0.95, z.avgsim = FALSE, zlabels = NULL) {
  UseMethod("get_IMIFA_results")
}

#' @export
get_IMIFA_results.IMIFA        <- function(sims = NULL, burnin = 0L, thinning = 1L, G = NULL, Q = NULL, criterion = c("bicm", "aicm", "log.iLLH", "dic", "bic.mcmc", "aic.mcmc"),
                                           G.meth = c("mode", "median"), Q.meth = c("mode", "median"), dat = NULL, conf.level = 0.95, z.avgsim = FALSE, zlabels = NULL) {

  call           <- match.call()
  defopt         <- options()
  options(warn=1)
  on.exit(suppressWarnings(options(defopt)), add=TRUE)
  if(missing(sims))               stop("Simulations must be supplied")
  if(class(sims) != "IMIFA")      stop("Object of class 'IMIFA' must be supplied")
  if(!exists(deparse(substitute(sims)),
             envir=.GlobalEnv))   stop(paste0("Object ", match.call()$sims, " not found\n"))
  burnin         <- as.integer(burnin)
  thinning       <- as.integer(thinning)
  if(any(c(length(thinning),
           length(burnin)) > 1))  stop("'burnin' and 'thinning' must be of length 1")
  if(any(burnin   < 0,
         thinning < 1))           stop("Invalid 'burnin' and/or 'thinning' supplied")
  store          <- seq(from=burnin + 1, to=attr(sims, "Store"), by=thinning)
  if(length(store) < 10)          stop("Not enough stored samples to proceed")
  n.store        <- length(store)
  tmp.store      <- store
  label.switch   <- attr(sims, "Label.Switch")
  method         <- attr(sims, "Method")
  learn.alpha    <- attr(sims, "Alph.step")
  learn.d        <- attr(sims, "Disc.step")
  inf.G          <- is.element(method, c("IMIFA", "OMIFA", "IMFA", "OMFA"))
  inf.Q          <- is.element(method, c("IMIFA", "OMIFA", "MIFA",  "IFA"))
  n.fac          <- attr(sims, "Factors")
  n.grp          <- attr(sims, "Groups")
  n.obs          <- attr(sims, "Obs")
  n.var          <- attr(sims, "Vars")
  sw             <- attr(sims, "Switch")
  cent           <- attr(sims, "Center")
  scaling        <- attr(sims, "Scaling")
  scal.meth      <- attr(scaling, "Method")
  uni.meth       <- attr(sims, "Uni.Meth")
  uni.type       <- unname(uni.meth["Uni.Type"])
  conf.level     <- as.numeric(conf.level)
  varnames       <- NULL
  if(any(length(conf.level) != 1,
     !is.numeric(conf.level),
     (conf.level <= 0   ||
      conf.level >= 1)))          stop("'conf.level' must be a single number between 0 and 1")
  conf.levels    <- c((1 - conf.level)/2, (1 + conf.level)/2)
  choice         <- length(n.grp) * length(n.fac) > 1
  criterion      <- match.arg(criterion)
  if(all(inf.Q, is.element(criterion,
     c("aic.mcmc", "bic.mcmc")))) stop(paste0(ifelse(isTRUE(choice), "Model choice is", "Though model choice isn't"), " actually required -\n 'criterion' cannot be 'aic.mcmc' or 'bic.mcmc' for the ", method, " method"))
  recomp         <- any(burnin  > 0,
                    thinning    > 1)
  if(any(!is.logical(z.avgsim),
         length(z.avgsim) != 1))  stop("'z.avgsim' must be TRUE or FALSE")

  G.T            <- !missing(G)
  Q.T            <- !missing(Q)
  G.ind          <- Q.ind      <- 1L
  if(inf.G)  {
    GQs          <- length(sims[[G.ind]])
    GQ1          <- GQs > 1
    G.store2     <- lapply(seq_len(GQs), function(gq) sims[[G.ind]][[gq]]$G.store[store])
    G.store      <- matrix(unlist(G.store2), nrow=GQs, ncol=n.store, byrow=TRUE)
    if(is.element(method, c("IMFA", "IMIFA"))) {
      act.store  <- lapply(seq_len(GQs), function(gq) sims[[G.ind]][[gq]]$act.store[store])
    }
    G.meth       <- ifelse(missing(G.meth), "mode", match.arg(G.meth))
    G.tab        <- if(GQ1) lapply(apply(G.store, 1, function(x) list(table(x, dnn=NULL))), "[[", 1) else table(G.store, dnn=NULL)
    G.prob       <- if(GQ1) lapply(G.tab, prop.table) else prop.table(G.tab)
    G.mode       <- if(GQ1) unlist(lapply(G.tab, function(gt) as.numeric(names(gt[gt == max(gt)])[1]))) else as.numeric(names(G.tab[G.tab == max(G.tab)])[1])
    G.med        <- if(GQ1) ceiling(matrixStats::rowMedians(G.store) * 2)/2 else ceiling(med(G.store) * 2)/2
    if(!G.T) {
      G          <- switch(G.meth, mode=G.mode, floor(G.med))
    }
    G.CI         <- if(GQ1) round(rowQuantiles(G.store, probs=conf.levels)) else round(quantile(G.store, conf.levels))
  }
  if(G.T)    {
    G            <- as.integer(G)
    if(any(length(G) != 1,
           !is.integer(G)))       stop("'G' must be an integer of length 1")
    if(!inf.G) {
      if(!is.element(method, c("FA", "IFA"))) {
        if(!is.element(G, n.grp)) stop("This 'G' value was not used during simulation")
        G.ind    <- which(n.grp == G)
      } else if(G > 1)            message(paste0("Forced G=1 for the ", method, " method"))
    } else   {
      if(all(!inf.Q, GQ1)) {
        if(!Q.T)                  stop(paste0("'G' cannot be supplied without 'Q' for the ", method, " method if a range of Q values were explored"))
        tmpQ     <- which(n.fac == unique(Q))
      } else {
        tmpQ     <- Q.ind
      }
      if(length(tmpQ > 0)  && !is.element(G,
         unique(G.store[tmpQ,]))) stop("This 'G' value was not visited during simulation")
    }
  }
  G              <- ifelse(any(inf.G, all(G.T, !is.element(method, c("FA", "IFA")))), G, 1L)
  if(Q.T)    {
    Q            <- as.integer(Q)
    if(!is.integer(Q))            stop("'Q' must of integer type")
    if(G.T)  {
      if(length(Q) == 1)     Q <- rep(Q, G)
      if(length(Q) != G)          stop(paste0("'Q' must be supplied for each group, as a scalar or vector of length G=", G))
    } else if(length(n.grp)    != 1 && all(!is.element(length(Q),
              c(1,  n.grp))))     stop("'Q' must be a scalar if G=1, 'G' is not suppplied, or a range of G values were explored")
    if(all(is.element(method, c("FA", "MFA", "OMFA", "IMFA")))) {
      if(length(unique(Q)) != 1)  stop(paste0("'Q' cannot vary across groups for the ", method, " method"))
      Q          <- unique(Q)
      if(!is.element(Q,   n.fac)) stop("This 'Q' value was not used during simulation")
      Q.ind      <- which(n.fac == Q)
    }
    if(inf.Q)  {
      if(any((Q  != 0) + (Q *
        (n.var - Q)   <= 0) > 1)) stop(paste0("'Q' must be less than the number of variables ", n.var))
      Qtmp       <- if(inf.G) Rfast::rowMaxs(sims[[1]][[1]]$Q.store[seq_len(G),, drop=FALSE], value=TRUE) else switch(method, MIFA=Rfast::rowMaxs(sims[[ifelse(G.T, which(G == n.grp), G.ind)]][[1]]$Q.store, value=TRUE), max(sims[[1]][[1]]$Q.store))
      if(any(Q * (Qtmp - Q) < 0)) stop(paste0("'Q' can't be greater than the maximum number of factors stored in ", ifelse(method == "IFA", "", "any group of "), match.call()$sims))
    }
  }
  if(inf.G)    {
    tmp.store    <- if(GQ1) lapply(seq_len(GQs), function(gq) store[which(G.store[gq,] == G[ifelse(G.T, 1, gq)])]) else store[which(G.store == G)]
    GQ.temp1     <- list(G = G, G.Mode = G.mode, G.Median = G.med, G.CI = G.CI, G.Probs = G.prob, G.Counts = G.tab)
    GQ.temp1     <- c(GQ.temp1, list(Stored.G = switch(method, OMIFA=provideDimnames(G.store, base=list("Non-Empty", ""),    unique=FALSE),
                      IMIFA=provideDimnames(do.call(rbind, c(G.store2, act.store)), base=list(c("Non-Empty", "Active"), ""), unique=FALSE),
                      OMFA=lapply(seq_len(GQs), function(g) provideDimnames(t(G.store[g,]),   base=list("Non-Empty", ""),    unique=FALSE)),
                      IMFA=lapply(seq_len(GQs), function(g) provideDimnames(rbind(G.store2[[g]], act.store[[g]]), base=list(c("Non-Empty", "Active"), ""), unique=FALSE)))))
  }
  G.range        <- ifelse(G.T, 1, length(n.grp))
  Q.range        <- ifelse(any(Q.T, all(!is.element(method, c("OMFA", "IMFA")), inf.Q)), 1, length(n.fac))
  crit.mat       <- matrix(NA, nrow=G.range, ncol=Q.range)

  # Retrieve log-likelihoods and/or tune G &/or Q according to criterion
    if(all(G.T, Q.T)) {
      dimnames(crit.mat) <- list(paste0("G", G),     if(inf.Q) "IFA" else paste0("Q", Q))
    } else if(G.T)    {
      dimnames(crit.mat) <- list(paste0("G", G),     if(inf.Q) "IFA" else paste0("Q", n.fac))
    } else if(Q.T)    {
      dimnames(crit.mat) <- list(paste0("G", n.grp), if(inf.Q) "IFA" else paste0("Q", Q))
    } else {
      dimnames(crit.mat) <- list(paste0("G", n.grp), if(inf.Q) "IFA" else paste0("Q", n.fac))
    }
    rownames(crit.mat)   <- switch(method, IMFA=, IMIFA="IM", OMFA=, OMIFA="OM", rownames(crit.mat))
    aicm         <- bicm       <- log.iLLH <-
    dic          <- aic.mcmc   <- bic.mcmc <- crit.mat
    log.N        <- log(n.obs)
    for(g in seq_len(G.range))   {
      gi                 <- ifelse(G.T, G.ind, g)
      for(q in seq_len(Q.range)) {
        qi               <- ifelse(Q.T, Q.ind, q)
        log.likes        <- if(is.element(method, c("OMFA", "IMFA")) && GQ1) sims[[gi]][[qi]]$ll.store[tmp.store[[qi]]] else sims[[gi]][[qi]]$ll.store[tmp.store]
        log.likes        <- log.likes[complete.cases(log.likes)]
        ll.max           <- 2 * max(log.likes)
        ll.var           <- ifelse(length(log.likes) != 1, 2 * Var(log.likes), 0)
        ll.mean          <- mean(log.likes)
        aicm[g,q]        <- ll.max  - ll.var   * 2
        bicm[g,q]        <- ll.max  - ll.var   * log.N
        log.iLLH[g,q]    <- ll.mean - ll.var   * (log.N - 1)
        dic[g,q]         <- (ll.max - ll.mean) * 3 - ll.mean
        if(!inf.Q) {
          K              <- switch(method, OMFA=, IMFA=PGMM_dfree(Q=n.fac[qi], P=n.var, G=G[ifelse(G.T, 1, qi)],
                            method=switch(uni.type, unconstrained="UUU", isotropic="UUC", constrained="UCU", single="UCC")), attr(sims[[gi]][[qi]], "K"))
          aic.mcmc[g,q]  <- ll.max  - K * 2
          bic.mcmc[g,q]  <- ll.max  - K * log.N
        }
      }
    }
    crit         <- get(criterion)
    crit.max     <- which(crit == max(crit), arr.ind=TRUE)

  # Control for supplied values of G &/or Q
    if(!any(Q.T, G.T)) {
      G.ind      <- crit.max[1]
      Q.ind      <- crit.max[2]
      if(!inf.G) {
        G        <- n.grp[G.ind]
      }
      if(!inf.Q) {
        Q        <- n.fac[Q.ind]
      }
    } else if(all(G.T, !Q.T)) {
      Q.ind      <- which.max(crit)
      if(!inf.Q) {
        Q        <- n.fac[Q.ind]
      }
    } else if(all(Q.T, !G.T)) {
      G.ind      <- which.max(crit)
      if(!inf.G) {
        G        <- n.grp[G.ind]
      }
    }
    G            <- ifelse(inf.G, ifelse(G.T, G, G[Q.ind]), ifelse(length(n.grp) == 1, n.grp, G))
    Gseq         <- seq_len(G)
    gnames       <- paste0("Group", Gseq)
    G.ind        <- ifelse(all(length(n.grp) == 1, !inf.G), which(n.grp == G), G.ind)
    GQ.temp2     <- list(AICMs = aicm, BICMs = bicm, LogIntegratedLikelihoods = log.iLLH, DICs = dic)
    if(is.element(method, c("OMFA", "IMFA")) &&
       GQ1)      {
      tmp.store  <- tmp.store[[Q.ind]]
    }
    if(!inf.Q)   {
      Q          <- if(length(n.fac)   > 1)  Q             else  n.fac
      Q.ind      <- if(all(!Q.T, length(n.fac) > 1)) Q.ind else which(n.fac == Q)
      Q          <- setNames(if(length(Q) != G) rep(Q, G)  else Q, gnames)
      if(all(inf.G, Q.T))  GQ.temp1$G <- rep(G, GQs)
      if(is.element(method, c("OMFA", "IMFA")) && GQ1) {
        GQ.temp1$G.CI     <- lapply(seq_len(GQs), function(gq) GQ.temp1$G.CI[gq,])
        GQ.temp1 <- lapply(GQ.temp1, "[[", Q.ind)
      } else if(inf.G) {
        GQ.temp1$Stored.G <- GQ.temp1$Stored.G[[1]]
      }
      GQ.temp3   <- c(GQ.temp2, list(AIC.mcmcs = aic.mcmc, BIC.mcmcs = bic.mcmc))
      GQ.res     <- switch(method, OMFA=, IMFA=c(GQ.temp1, list(Q = Q), list(Criteria = GQ.temp3)), c(list(G = G, Q = Q), list(Criteria = GQ.temp3)))
    }
    clust.ind    <- !any(is.element(method,   c("FA", "IFA")),
                     all(is.element(method, c("MFA", "MIFA")), G == 1))
    sw.mx        <- ifelse(clust.ind, sw["mu.sw"],  TRUE)
    sw.px        <- ifelse(clust.ind, sw["psi.sw"], TRUE)
    if(inf.Q) {
      Q.store    <- sims[[G.ind]][[Q.ind]]$Q.store[,tmp.store, drop=FALSE]
      Q.meth     <- ifelse(missing(Q.meth), "mode", match.arg(Q.meth))
    }
    if(length(tmp.store) <= 1)    stop(paste0("Not enough samples stored to proceed", ifelse(any(G.T, Q.T), paste0(": try supplying different Q or G values"), "")))

  # Retrieve dataset
    dat.nam      <- gsub("[[:space:]]", "", ifelse(missing(dat), attr(sims, "Name"), deparse(substitute(dat))))
    nam.dat      <- gsub("\\[.*", "", dat.nam)
    data.x       <- exists(nam.dat, envir=.GlobalEnv)
    pattern      <- c("(", ")")
    if(!any(is.element(method, c("FA", "IFA")),
       data.x)) {                 warning(paste0("Object ", nam.dat, " not found in .GlobalEnv: can't compute empirical covariance and error metrics"), call.=FALSE)
    } else if(!is.element(method, c("FA", "IFA"))) {
      dat        <- as.data.frame(get(nam.dat))
      nam.x      <- gsub(".*\\[(.*)\\].*", "(\\1)",  dat.nam)
      if(any(unlist(vapply(seq_along(pattern), function(p) grepl(pattern[p], nam.dat, fixed=TRUE), logical(1L))),
         !identical(dat.nam, nam.dat) && (any(grepl("[[:alpha:]]", gsub('c', '', nam.x))) || grepl(":",
         nam.x, fixed=TRUE)))) {  warning("Extremely inadvisable to supply 'dat' subsetted by any means other than row/column numbers or c() indexing:\n can't compute empirical covariance and error metrics, best to create new data object", call.=FALSE)
      } else  {
        spl.ind          <- if(grepl("(,", nam.x, fixed=TRUE)) sapply(gregexpr("\\(,", nam.x), head, 1L) else sapply(gregexpr("\\)", nam.x), head, 1L)
        spl.tmp          <- c(substring(nam.x, 1, spl.ind), substring(nam.x, spl.ind + 2L, nchar(nam.x)))
        neg.r            <- grepl("-", spl.tmp[1], fixed=TRUE) || grepl("!", spl.tmp[1], fixed=TRUE)
        neg.c            <- grepl("-", spl.tmp[2], fixed=TRUE) || grepl("!", spl.tmp[2], fixed=TRUE)
        rowx             <- as.numeric(unlist(strsplit(gsub('\\(', '', gsub(',', '', unlist(regmatches(spl.tmp[1], gregexpr('\\(?[0-9,.]+', spl.tmp[1]))))), '')))
        rowx             <- if(any(spl.ind <= 0, sum(rowx) %in% 0)) seq_len(nrow(dat)) else rowx
        colseq           <- ifelse(neg.c, -1, 1) * suppressWarnings(as.numeric(unlist(strsplit(gsub('\\(', '', gsub(',', '', unlist(regmatches(spl.tmp[2], gregexpr('\\(?[0-9,.]+', spl.tmp[2]))))), ''))))
        rowseq           <- rep(neg.r, nrow(dat))
        rowseq[rowx]     <- !rowseq[rowx]
        dat              <- subset(dat, select=if(any(spl.ind <= 0, sum(colseq) %in% 0)) seq_len(ncol(dat)) else colseq, subset=rowseq, drop=!grepl("drop=F", dat.nam))
      }
      dat        <- dat[complete.cases(dat),]
      dat        <- dat[vapply(dat, is.numeric, logical(1L))]
      dat        <- if(is.logical(scaling)) standardise(as.matrix(dat), center=cent, scale=scaling) else scale(dat, center=cent, scale=scaling)
      obsnames   <- rownames(dat)
      varnames   <- colnames(dat)
      if(!identical(dim(dat),
         c(n.obs, n.var)))        warning("Dimensions of data don't match those in the dataset supplied to mcmc_IMIFA():\n be careful using subsetted data, best to create new object", call.=FALSE)
      n.obs      <- nrow(dat)
    }

# Manage Label Switching & retrieve cluster labels/mixing proportions
  if(clust.ind) {
    label.miss   <- missing(zlabels)
    if(!label.miss)   {
      z.nam      <- gsub("[[:space:]]", "", deparse(substitute(zlabels)))
      nam.z      <- gsub("\\[.*", "", z.nam)
      nam.zx     <- gsub(".*\\[(.*)\\].*", "\\1)", z.nam)
      if(grepl("$", z.nam, fixed=TRUE)) {
        x.nam    <- strsplit(nam.z, "$", fixed=TRUE)[[1]]
        nam.z    <- z.nam  <- x.nam[1]
        if(x.nam[2] %in% colnames(get(nam.z))) {
         zlabels <- get(nam.z)[x.nam[2]][,1]
        } else                    stop(paste0("'", x.nam[2], "' not found within '", nam.z, "'"))
      }
      if(!exists(nam.z,
         envir=.GlobalEnv))       stop(paste0("Object ", match.call()$zlabels, " not found\n"))
      if(any(unlist(vapply(seq_along(pattern), function(p) grepl(pattern[p], nam.z, fixed=TRUE), logical(1L))),
         !identical(z.nam, nam.z) && (any(grepl("[[:alpha:]]", gsub('c', '', nam.zx))) || grepl(":",
         nam.zx, fixed=TRUE))))   stop("Extremely inadvisable to supply 'zlabels' subsetted by any means other than row/column numbers or c() indexing: best to create new object")
     if(length(zlabels) != n.obs) stop(paste0("'zlabels' must be a factor of length N=",  n.obs))
    }
    if(sw["mu.sw"])   {
      mus        <- sims[[G.ind]][[Q.ind]]$mu[,,tmp.store, drop=FALSE]
    }
    if(sw["l.sw"])    {
      lmats      <- sims[[G.ind]][[Q.ind]]$load
      if(inf.Q) {
        lmats    <- as.array(lmats)
      }
      lmats      <- lmats[,,,tmp.store, drop=FALSE]
    }
    if(sw["psi.sw"])  {
      psis       <- sims[[G.ind]][[Q.ind]]$psi[,,tmp.store, drop=FALSE]
    }
    if(sw["pi.sw"])   {
      pies       <- sims[[G.ind]][[Q.ind]]$pi.prop[,tmp.store, drop=FALSE]
    }
    zadj         <- sims[[G.ind]][[Q.ind]]$z.store
    z            <- as.matrix(zadj[,tmp.store])
    zadj         <- zadj[,store]
    if(!label.switch) {
      z.temp     <- tryCatch(factor(z[,1], labels=Gseq), error=function(e) factor(z[,1], levels=Gseq))
      for(sl in seq_along(tmp.store)) {
        sw.lab   <- .lab_switch(z.new=z[,sl], z.old=z.temp)
        z.perm   <- sw.lab$z.perm
        left     <- as.integer(unname(z.perm))
        right    <- as.integer(names(z.perm))
        if(!identical(left, right))   {
          z[,sl] <- sw.lab$z
          if(sw["mu.sw"])  {
            mus[,left,sl]      <- mus[,right,sl]
          }
          if(sw["l.sw"])   {
            lmats[,,left,sl]   <- lmats[,,right,sl]
          }
          if(sw["psi.sw"]) {
            psis[,left,sl]     <- psis[,right,sl]
          }
          if(sw["pi.sw"])  {
            pies[left,sl]      <- pies[right,sl]
          }
          if(inf.Q)        {
            Q.store[left,sl]   <- Q.store[right,sl]
          }
        }
      }
    }
    if(sw["mu.sw"])        mus <- tryCatch(mus[,Gseq,,     drop=FALSE], error=function(e) mus)
    if(sw["l.sw"])       lmats <- tryCatch(lmats[,,Gseq,,  drop=FALSE], error=function(e) lmats)
    if(sw["psi.sw"])      psis <- tryCatch(psis[,Gseq,,    drop=FALSE], error=function(e) psis)
    map          <- apply(z, 1,   function(x) factor(which.max(tabulate(x)), levels=Gseq))
    if(isTRUE(z.avgsim)) {
      zlog       <- capture.output(znew <- try(Zsimilarity(zs=zadj), silent=TRUE))
      condit     <- all(!is.element(method, c("MIFA", "MFA")), inherits(znew, "try-error"))
      if(isTRUE(condit)) {
        zlog     <- capture.output(znew <- try(Zsimilarity(zs=z),    silent=TRUE))
                                  warning("Constructing the similarity matrix failed:\n trying again using iterations corresponding to the modal number of clusters", call.=FALSE)
      }
      if(!inherits(znew, "try-error")) {
        zadj     <- znew$z.avg
        zadj     <- factor(zadj, labels=seq_along(unique(zadj)))
        zadj     <- as.integer(levels(zadj))[zadj]
        zavg     <- znew$z.sim
        zG       <- max(zadj)
        if(!label.miss) {
         zlabels <- factor(zlabels, labels=seq_along(unique(zlabels)))
         levs    <- levels(zlabels)
         if(length(levs) == zG) {
           zadj <- .lab_switch(z.new=zadj, z.old=zlabels)$z
         }
         tab     <- table(zadj, zlabels, dnn=list("Predicted", "Observed"))
         tabstat <- c(classAgreement(tab), classError(map, zlabels))
         if(nrow(tab) != ncol(tab))     {
         tabstat <- tabstat[-seq_len(2)]
           names(tabstat)[4]   <- "error.rate"
         } else {
           names(tabstat)[6]   <- "error.rate"
         }
         if(tabstat$error.rate == 0) {
         tabstat$misclassified <- NULL
         }
         tabstat <- c(list(confusion.matrix = tab), tabstat)
         if(length(levs) == zG)  {
          names(tabstat)[1]    <- "confusion.matrix.matched"
         }
         class(tabstat)        <- "listof"
        }
        z_simavg <- list(z.avg = zadj, z.sim = zavg)
        z_simavg <- c(z_simavg, if(!label.miss) list(avgsim.perf = tabstat))
        attr(z_simavg, "Conditional")   <- condit
      } else {                    warning("Can't compute similarity matrix or 'average' clustering: forcing 'z.avgsim' to FALSE", call.=FALSE)
        z.avgsim <- FALSE
      }
    }
    uncertain    <- 1 - Rfast::colMaxs(matrix(apply(z, 1, tabulate, nbins=G)/length(tmp.store), nrow=G, ncol=n.obs), value=TRUE)
    if(sw["pi.sw"]) {
      pi.prop    <- provideDimnames(pies[Gseq,seq_along(tmp.store), drop=FALSE], base=list(gnames, ""), unique=FALSE)
      var.pi     <- setNames(Rfast::rowVars(pi.prop),  gnames)
      ci.pi      <- rowQuantiles(pi.prop, probs=conf.levels)
      post.pi    <- setNames(rowmeans(pi.prop),        gnames)
    } else {
      post.pi    <- setNames(prop.table(tabulate(map, nbins=G)), gnames)
    }
    if(inf.Q)       {
      Q.store    <- provideDimnames(Q.store[Gseq,, drop=FALSE],  base=list(gnames, ""), unique=FALSE)
    }
    if(!label.miss) {
      zlabels    <- factor(zlabels, labels=seq_along(unique(zlabels)))
      levs       <- levels(zlabels)
      if(length(levs) == G) {
        sw.lab   <- .lab_switch(z.new=as.numeric(levels(map))[map], z.old=zlabels)
        map      <- factor(sw.lab$z, levels=Gseq)
        l.perm   <- sw.lab$z.perm
        left     <- as.integer(unname(l.perm))
        right    <- as.integer(names(l.perm))
        z.tmp    <- lapply(seq_along(tmp.store), function(i) factor(factor(z[,i], labels=left[which(tabulate(z[,i], nbins=G) > 0)]), levels=right))
        z        <- do.call(cbind, lapply(z.tmp, function(x) as.integer(levels(as.factor(x)))[as.integer(x)]))
        if(sw["mu.sw"])    mus <- mus[,right,,     drop=FALSE]
        if(sw["l.sw"])   lmats <- lmats[,,right,,  drop=FALSE]
        if(sw["psi.sw"])  psis <- psis[,right,,    drop=FALSE]
        index    <- Order(left)
        post.pi  <- setNames(post.pi[index], gnames)
        if(sw["pi.sw"]) {
         pi.prop <- provideDimnames(unname(pi.prop[index,, drop=FALSE]), base=list(gnames, ""), unique=FALSE)
         var.pi  <- setNames(var.pi[index],  gnames)
         ci.pi   <- provideDimnames(unname(ci.pi[index,,   drop=FALSE]), base=list(gnames,  colnames(ci.pi)))
        }
        if(inf.Q)   {
         Q.store <- provideDimnames(unname(Q.store[index,, drop=FALSE]), base=list(gnames, ""), unique=FALSE)
        }
      }
      tab        <- table(map, zlabels, dnn=list("Predicted", "Observed"))
      tab.stat   <- c(classAgreement(tab), classError(map, zlabels))
      if(nrow(tab) != ncol(tab))     {
        tab.stat <- tab.stat[-seq_len(2)]
        names(tab.stat)[4]     <- "error.rate"
      } else {
        names(tab.stat)[6]     <- "error.rate"
      }
      if(tab.stat$error.rate   == 0) {
        tab.stat$misclassified <- NULL
      }
      tab.stat   <- c(list(confusion.matrix = tab), tab.stat)
      if(length(levs) == G)   {
        names(tab.stat)[1]     <- "confusion.matrix.matched"
      }
      class(tab.stat)          <- "listof"
    }
    uncert.obs   <- which(uncertain >= 1/G)
    sizes        <- setNames(tabulate(map, nbins=G), gnames)
    if(any(sizes == 0))           warning("Empty group exists in modal clustering:\n examine trace plots and try supplying a lower G value to tune.imifa() or re-running the model", call.=FALSE)
    if(learn.alpha) {
      alpha      <- sims[[G.ind]][[Q.ind]]$alpha[store]
      post.alpha <- mean(alpha)
      var.alpha  <- Var(alpha)
      ci.alpha   <- quantile(alpha,    conf.levels)
      rate       <- sims[[G.ind]][[Q.ind]]$a.rate
      DP.alpha   <- list(alpha = alpha, post.alpha = post.alpha, var.alpha = var.alpha, ci.alpha = ci.alpha, alpha.rate = rate)
      DP.alpha   <- c(DP.alpha, if(isTRUE(attr(sims, "TuneZeta"))) list(avg.zeta = sims[[G.ind]][[Q.ind]]$avg.zeta))
      class(DP.alpha)          <- "listof"
    }
    if(learn.d)     {
      discount   <- as.vector(sims[[G.ind]][[Q.ind]]$discount[store])
      post.disc  <- mean(discount)
      post.kappa <- sum(discount == 0)/n.store
      var.disc   <- Var(discount)
      ci.disc    <- quantile(discount, conf.levels)
      rate       <- sims[[G.ind]][[Q.ind]]$d.rate
      post.dzero <- post.disc/(1  - post.kappa)
      discount   <- if(sum(discount  == 0)/n.store > 0.5) as.simple_triplet_matrix(discount)  else discount
      PY.disc    <- list(discount = discount, post.disc = post.disc, post.kappa = post.kappa, var.disc = var.disc,
                         ci.disc  = ci.disc,  disc.rate = rate,  post.d_nonzero = post.dzero)
      class(PY.disc)           <- "listof"
    }
    map          <- as.integer(levels(map))[map]
    uncertain    <- if(sum(uncertain == 0)/n.obs   > 0.5)  as.simple_triplet_matrix(uncertain) else uncertain
    attr(uncertain, "Obs")     <- if(sum(uncert.obs) != 0) uncert.obs
    tab.stat$uncertain         <- if(!label.miss)          attr(uncertain, "Obs")
    cluster      <- list(map = map, z = z, uncertainty = uncertain)
    cluster      <- c(cluster, list(post.sizes  = sizes, post.pi = post.pi/sum(post.pi)),
                      if(sw["pi.sw"]) list(pi.prop = pi.prop, var.pi = var.pi, ci.pi = ci.pi),
                      if(!label.miss) list(perf = tab.stat),
                      if(learn.alpha) list(DP.alpha = DP.alpha),
                      if(learn.d)     list(PY.disc = PY.disc),
                      if(z.avgsim)    list(Z.avgsim = z_simavg),
                      if(is.element(method, c("IMFA", "IMIFA"))) list(lab.rate = sims[[G.ind]][[Q.ind]]$lab.rate))
    attr(cluster, "Z.init")    <- attr(sims[[G.ind]], "Z.init")
    attr(cluster, "Init.Meth") <- attr(sims, "Init.Z")
    attr(cluster, "Label.Sup") <- !label.miss
    z.ind        <- lapply(Gseq, function(g) map == g)
  }
  if(inf.Q)   {
    G1           <- G > 1
    Q.tab        <- if(G1) lapply(apply(Q.store, 1, function(x) list(table(x, dnn=NULL))), "[[", 1) else table(Q.store, dnn=NULL)
    Q.prob       <- if(G1) lapply(Q.tab, prop.table) else prop.table(Q.tab)
    Q.mode       <- if(G1) unlist(lapply(Q.tab, function(qt) as.numeric(names(qt[qt == max(qt)])[1]))) else as.numeric(names(Q.tab[Q.tab == max(Q.tab)])[1])
    Q.med        <- if(G1) setNames(ceiling(matrixStats::rowMedians(Q.store) * 2)/2, gnames) else ceiling(med(Q.store) * 2)/2
    if(!Q.T)  {
      Q          <- switch(Q.meth, mode=Q.mode, floor(Q.med))
    } else    {
      Q          <- if(G.T) Q else setNames(rep(Q, G), gnames)
    }
    leder.b      <- min(n.obs - 1, Ledermann(n.var))
    if(any(unlist(Q) > leder.b))  warning(paste0("Estimate of Q", ifelse(G > 1, " in one or more of the groups ", " "), "is greater than ", ifelse(any(unlist(Q) > n.var), paste0("the number of variables (", n.var, ")"), paste0("the suggested Ledermann upper bound (", leder.b, ")")), ":\nsolution may be invalid"), call.=FALSE)
    Q.CI         <- if(G1) round(rowQuantiles(Q.store, probs=conf.levels)) else round(quantile(Q.store, conf.levels))
    GQ.temp4     <- list(Q = Q, Q.Mode = Q.mode, Q.Median = Q.med,
                         Q.CI = Q.CI, Q.Probs = Q.prob, Q.Counts = Q.tab,
                         Stored.Q = if(clust.ind) Q.store else as.vector(Q.store))
    GQ.res       <- if(inf.G) c(GQ.temp1, GQ.temp4) else c(list(G = G), GQ.temp4)
    GQ.res       <- c(GQ.res, list(Criteria = GQ.temp2))
    attr(GQ.res, "Q.big") <- attr(sims[[G.ind]][[Q.ind]], "Q.big")
  }
  Q0             <- Q == 0
  if(all(isTRUE(choice), is.element(criterion, c("aicm", "bicm", "log.iLLH")))) {
    if(all(!G.T, !is.element(method,
       c("FA", "IFA")), G == 1))  warning(paste0("Chosen model has only one group:\n Note that the ", criterion, " criterion may exhibit bias toward one-group models"),   call.=FALSE)
    if(all(!Q.T, method   == "MIFA")) {
      if(any(Q0))                 warning(paste0("Chosen model has ", ifelse(sum(Q0) == G, "zero factors", "a group with zero factors"), ":\n Note that the ", criterion, " criterion may exhibit bias toward models ", ifelse(sum(Q0) == G, "with zero factors", "where some groups have zero factors")), call.=FALSE)
    } else if(all(Q0))            warning(paste0("Chosen model has zero factors:\n Note that the ",   criterion, " criterion may exhibit bias toward zero-factor models"), call.=FALSE)
  }


# Retrieve (unrotated) scores
  if((no.score   <- all(Q0))) {
    if(sw["s.sw"])                message("Scores & loadings not stored as model has zero factors")
    sw["s.sw"]   <- FALSE
  }
  if(inf.Q) {
    l.store      <- lapply(Gseq, function(g, ts=seq_along(tmp.store)) ts[which(Q.store[g,] >= Q[g])])
    eta.store    <- sort_unique(unlist(l.store))
  } else {
    eta.store    <- tmp.store
  }
  if(sw["s.sw"]) {
    eta          <- sims[[G.ind]][[Q.ind]]$eta
    if(inf.Q) {
      eta        <- as.array(eta)
    }
    eta          <- eta[,,eta.store, drop=FALSE]
  }

# Loop over g in G to extract other results
  result         <- list(list())
  mse   <- mae   <- medse  <-
  medae <- rmse  <- nrmse  <-
  emp.T <- est.T <- rep(NA, G)
  for(g in Gseq) {
    Qg           <- Q[g]
    Q0g          <- Q0[g]
    Qgs          <- seq_len(Qg)
    sw["l.sw"]   <- attr(sims, "Switch")["l.sw"]
    if(Q0g)      {
      if(all(sw["l.sw"],
             !no.score))          message(paste0("Loadings ", ifelse(G > 1, paste0("for group ", g, " not stored as it"), " not stored as model"), " has zero factors"))
      sw["l.sw"] <- FALSE
    }
    store        <- seq_along(tmp.store)
    n.store      <- length(store)

  # Retrieve (unrotated) loadings
    if(sw["l.sw"]) {
      tmpind     <- ifelse(inf.Q, which.max(Q.store[g,] == Qg), 1)
      if(clust.ind)  {
        lmat     <- adrop(lmats[,,g,store, drop=FALSE], drop=3)
        l.temp   <- adrop(lmat[,,tmpind,   drop=FALSE], drop=3)
      } else {
        lmat     <- sims[[G.ind]][[Q.ind]]$load
        if(inf.Q) {
          lmat   <- as.array(lmat)
        }
        lmat     <- lmat[,,store,          drop=FALSE]
        l.temp   <- adrop(lmat[,,tmpind,   drop=FALSE], drop=3)
      }
    }

  # Loadings matrix / identifiability / error metrics / etc.
    if(sw["l.sw"])    {
      for(p in store) {
        if(p    %in% eta.store) {
          proc   <- Procrustes(X=as.matrix(lmat[,,p]), Xstar=l.temp)
          lmat[,,p]        <- proc$X.new
          if(sw["s.sw"])  {
            rot  <- proc$R
            p2   <- ifelse(inf.Q, which(eta.store == p), p)
            if(clust.ind) {
              zp <- z[,p]  == g
              eta[zp,,p2]  <- eta[zp,,p2] %*% rot
            } else {
              eta[,,p2]    <- eta[,,p2]   %*% rot
            }
          }
        }
      }
    }

  # Retrieve means, uniquenesses & empirical covariance matrix
    if(clust.ind) {
      if(sw["mu.sw"])  {
        mu       <- as.matrix(mus[,g,store])
      }
      if(sw["psi.sw"]) {
        psi      <- as.matrix(psis[,g,store])
      }
      if(all(data.x, sizes[g] > 1)) {
        dat.gg   <- dat[z.ind[[g]],, drop=FALSE]
        cov.emp  <- if(n.var > 500) provideDimnames(cova(dat.gg), base=list(varnames)) else cov(dat.gg)
      } else cov.emp     <- NULL
    } else {
      post.mu    <- sims[[G.ind]][[Q.ind]]$post.mu
      post.psi   <- sims[[G.ind]][[Q.ind]]$post.psi
      if(sw["mu.sw"])  {
        mu       <- sims[[G.ind]][[Q.ind]]$mu[,store]
      }
      if(sw["psi.sw"]) {
        psi      <- sims[[G.ind]][[Q.ind]]$psi[,store]
      }
      cov.emp    <- sims[[G.ind]][[Q.ind]]$cov.emp
    }
    emp.T[g]     <- !is.null(cov.emp)
    if(data.x)         {
      if(is.null(rownames(mu)))   {
        rownames(mu)     <- varnames
      }
      if(is.null(rownames(psi)))  {
        rownames(psi)    <- varnames
      }
      if(sw["l.sw"]      &&
         is.null(rownames(lmat))) {
        rownames(lmat)   <- varnames
      }
    }

  # Compute posterior means and % variation explained
    if(sw["mu.sw"])  {
      post.mu    <- if(clust.ind) rowmeans(mu)  else post.mu
      var.mu     <- Rfast::rowVars(mu)
      ci.mu      <- rowQuantiles(mu,  probs=conf.levels)
    }
    if(sw["psi.sw"]) {
      post.psi   <- if(clust.ind) rowmeans(psi) else post.psi
      var.psi    <- Rfast::rowVars(psi)
      ci.psi     <- rowQuantiles(psi, probs=conf.levels)
    }
    if(sw["l.sw"])   {
      lmat       <- provideDimnames(lmat[,Qgs,if(inf.Q) l.store[[g]] else store, drop=FALSE], base=list("", paste0("Factor", Qgs), ""), unique=FALSE)
      post.load  <- rowMeans(lmat, dims=2)
      var.load   <- apply(lmat, c(1, 2), Var)
      ci.load    <- apply(lmat, c(1, 2), quantile, conf.levels)
      var.exp    <- sum(colSums(post.load * post.load))/n.var
      class(post.load)     <- "loadings"
    } else if(all(emp.T[g], sw["psi.sw"])) {
      var.exp    <- ifelse(exists("z.ind", envir=.GlobalEnv) && sizes[g] == 0, 0, max(0, (sum(diag(cov.emp)) - sum(post.psi))/n.var))
    } else {
      var.exp    <- NULL
    }

  # Calculate estimated covariance matrices & compute error metrics
    if(clust.ind) {
      if(all(sw["psi.sw"], any(sw["l.sw"], Q0g)))  {
        cov.est  <- if(!Q0g)      tcrossprod(post.load) + diag(post.psi) else diag(post.psi)
        if(data.x)      {
          dimnames(cov.est)    <- list(varnames, varnames)
        }
      } else   {
        cov.est  <- NULL
        if(g == 1) {
         if(all(!sw["l.sw"], !Q0g, !sw["psi.sw"])) {
                                  warning("Loadings & Uniquenesses not stored: can't estimate covariance matrix and compute error metrics", call.=FALSE)
         } else if(all(!Q0g,
                  !sw["l.sw"])) { warning("Loadings not stored: can't estimate covariance matrix and compute error metrics", call.=FALSE)
         } else if(!sw["psi.sw"]) warning("Uniquenesses not stored: can't estimate covariance matrix and compute error metrics", call.=FALSE)
        }
      }
    } else     {
      cov.est    <- sims[[G.ind]][[Q.ind]]$cov.est
      if(all(recomp, sw["psi.sw"], any(sw["l.sw"], Q0g))) {
        cov.est  <- replace(cov.est, is.numeric(cov.est), 0)
        for(r in seq_len(n.store))    {
         sigma   <- if(!Q0g)      tcrossprod(lmat[,,r]) + diag(psi[,r]) else diag(psi[,r])
         cov.est <- cov.est + sigma/n.store
        }
      } else if(all(recomp,  g == 1)) {
        if(all(!sw["l.sw"], !Q0g, !sw["psi.sw"]))         {
                                  warning("Loadings & Uniquenesses not stored: can't re-estimate covariance matrix", call.=FALSE)
        } else if(all(!Q0g,
                  !sw["l.sw"])) { warning("Loadings not stored: can't re-estimate covariance matrix", call.=FALSE)
        } else if(!sw["psi.sw"])  warning("Uniquenesses not stored: can't re-estimate covariance matrix", call.=FALSE)
      }
    }
    est.T[g]     <- !is.null(cov.est)

    if(all(emp.T[g], est.T[g])) {
      error      <- cov.emp - cov.est
      sq.error   <- error * error
      abs.error  <- abs(error)
      mse[g]     <- mean(sq.error)
      mae[g]     <- mean(abs.error)
      medse[g]   <- med(sq.error)
      medae[g]   <- med(abs.error)
      rmse[g]    <- sqrt(mse[g])
      nrmse[g]   <- rmse[g]/(max(cov.emp) - min(cov.emp))
      if(any(all(scal.meth != "none", cent) &&
                 sum(round(diag(cov.est))   !=
                 round(diag(cov.emp)))      != 0,
         sum(abs(post.psi  - (1 - post.psi)) < 0) != 0,
         var.exp  > 1))           warning(paste0(ifelse(G == 1, "C", paste0("Group ", g, "'s c")), "hain may not have fully converged"), call.=FALSE)
    }

    results      <- list(if(sw["mu.sw"])  list(means     = mu,
                                               var.mu    = var.mu,
                                               ci.mu     = ci.mu),
                         if(sw["l.sw"])   list(loadings  = lmat,
                                               post.load = post.load,
                                               var.load  = var.load,
                                               ci.load   = ci.load),
                         if(sw["psi.sw"]) list(psis      = psi,
                                               var.psi   = var.psi,
                                               ci.psi    = ci.psi),
                         if(sw.mx)        list(post.mu   = post.mu),
                         if(sw.px)        list(post.psi  = post.psi),
                         if(any(sw["l.sw"],
                                sw.px))   list(var.exp   = var.exp),
                         if(emp.T[g])     list(cov.emp   = cov.emp),
                         if(est.T[g])     list(cov.est   = cov.est))
    result[[g]]  <- unlist(results, recursive=FALSE)
    attr(result[[g]], "Store") <- n.store
  }
  if(sw["s.sw"])   {
    Qseq         <- seq_len(max(Q))
    eta          <- provideDimnames(eta[,Qseq,, drop=FALSE], base=list("", paste0("Factor", Qseq), ""), unique=FALSE)
    scores       <- list(eta = eta, post.eta = rowMeans(eta, dims=2), var.eta = apply(eta,
                         c(1, 2), Var), ci.eta = apply(eta, c(1, 2), quantile, conf.levels))
    attr(scores, "Eta.store")  <- length(eta.store)
  }
  names(result)  <- gnames
  class(GQ.res)                <- "listof"
  attr(GQ.res, "Criterion")    <- criterion
  attr(GQ.res, "Factors")      <- n.fac
  attr(GQ.res, "Groups")       <- n.grp
  attr(GQ.res, "Supplied")     <- c(Q=Q.T, G=G.T)
  err.T                        <- vapply(Gseq, function(g) all(emp.T[g], est.T[g]), logical(1L))
  var.exps       <- vapply(lapply(result, "[[", "var.exp"), function(x) ifelse(is.null(x), NA, x), numeric(1L))
  var.exps       <- if(sum(is.na(var.exps)) == G) NULL else var.exps
  if(any(err.T))   {
    Err          <- lapply(list(MSE = mse, MAE = mae, MEDSE = medse, MEDAE = medae,
                                RMSE = rmse, NRMSE = nrmse), setNames, gnames)
    if(G > 1)      {
      Err        <- c(Err, list(Averages = unlist(lapply(Err, mean, na.rm=TRUE))))
    } else {
      Err        <- setNames(unlist(Err), names(Err))
    }
    emp.covs     <- lapply(result, "[[", "cov.emp")
    est.covs     <- lapply(result, "[[", "cov.est")
    Err          <- c(Err, list(Var.Exps = var.exps, Empirical.Covs = emp.covs, Estimated.Covs = est.covs))
  } else Err     <- if(!is.null(var.exps)) list(Var.Exps = var.exps)
  if(!is.null(Err)) class(Err) <- "listof"
  if(sw["mu.sw"])  {
    mus          <- Filter(Negate(is.null), lapply(result, "[[", "means"))
    post.mu      <- provideDimnames(do.call(cbind, lapply(result, "[[", "post.mu")),  base=list(rownames(mus[[1]]),  gnames))
    var.mu       <- provideDimnames(do.call(cbind, lapply(result, "[[", "var.mu")),   base=list(rownames(mus[[1]]),  gnames))
    ci.mu        <- Filter(Negate(is.null), lapply(result, "[[", "ci.mu"))
    means        <- list(mus = mus, post.mu = post.mu, var.mu = var.mu, ci.mu = ci.mu)
  }
  sw["l.sw"]     <- attr(sims, "Switch")["l.sw"] && !all(Q == 0)
  if(sw["l.sw"])   {
    lmats        <- Filter(Negate(is.null), lapply(result, "[[", "loadings"))
    post.load    <- Filter(Negate(is.null), lapply(result, "[[", "post.load"))
    var.load     <- Filter(Negate(is.null), lapply(result, "[[", "var.load"))
    ci.load      <- Filter(Negate(is.null), lapply(result, "[[", "ci.load"))
    loads        <- list(lmats = lmats, post.load = post.load, var.load = var.load, ci.load = ci.load)
  }
  if(sw["psi.sw"]) {
    psis         <- Filter(Negate(is.null), lapply(result, "[[", "psis"))
    post.psi     <- provideDimnames(do.call(cbind, lapply(result, "[[", "post.psi")), base=list(rownames(psis[[1]]), gnames))
    var.psi      <- provideDimnames(do.call(cbind, lapply(result, "[[", "var.psi")),  base=list(rownames(psis[[1]]), gnames))
    ci.psi       <- Filter(Negate(is.null), lapply(result, "[[", "ci.psi"))
    uniquenesses <- list(psis = psis, post.psi = post.psi, var.psi = var.psi, ci.psi = ci.psi)
  }
  result         <- c(if(exists("cluster", envir=environment())) list(Clust = cluster),
                      list(Error     = Err),   list(GQ.results = GQ.res),
                      if(sw["mu.sw"])  list(Means        =        means),
                      if(sw["l.sw"])   list(Loadings     =        loads),
                      if(sw["s.sw"])   list(Scores       =       scores),
                      if(sw["psi.sw"]) list(Uniquenesses = uniquenesses))

  attr(result, "Alph.step")    <- if(is.element(method, c("IMFA", "IMIFA"))) learn.alpha
  attr(result, "Alpha")        <- if(!learn.alpha) attr(sims, "Alpha")
  attr(result, "Call")         <- call
  attr(result, "Conf.Level")   <- conf.level
  attr(result, "Disc.step")    <- if(is.element(method, c("IMFA", "IMIFA"))) learn.alpha
  attr(result, "Discount")     <- if(is.element(method, c("IMFA", "IMIFA")) && !learn.d) attr(sims, "Discount")
  attr(result, "Errors")       <- any(err.T)
  attr(result, "G.init")       <- if(inf.G) attr(sims, "G.init")
  attr(result, "Ind.Slice")    <- if(is.element(method, c("IMFA", "IMIFA"))) attr(sims, "Ind.Slice")
  attr(result, "Method")       <- method
  attr(result, "N.Loadstore")  <- if(inf.Q) vapply(l.store, length, numeric(1L)) else rep(length(tmp.store), G)
  attr(result, "Name")         <- attr(sims, "Name")
  attr(result, "Obs")          <- n.obs
  attr(result, "Obsnames")     <- if(all(!sw["s.sw"], exists("obsnames", envir=.GlobalEnv))) obsnames
  attr(result, "Pitman")       <- attr(sims, "Pitman")
  attr(result, "range.G")      <- attr(sims, "Groups")
  attr(result, "range.Q")      <- attr(sims, "Factors")
  attr(result, "Store")        <- tmp.store
  attr(result, "Switch")       <- sw
  attr(result, "TuneZeta")     <- attr(sims, "TuneZeta")
  attr(result, "Uni.Meth")     <- uni.meth
  attr(result, "Varnames")     <- if(all(!sw["l.sw"], !sw["mu.sw"], !sw["psi.sw"], exists("varnames", envir=.GlobalEnv))) varnames
  attr(result, "Vars")         <- n.var
  attr(result, "Z.sim")        <- z.avgsim
  class(result)                <- "Results_IMIFA"
  cat(print.Results_IMIFA(result))
  return(result)
}
