#' tide.info gets the annual tide calendar for one particular location.
#' @title Annual tide calendar for one particular location
#' @author Marc Girondot \email{marc.girondot@@gmail.com}
#' @return Return a data.frame with annual tide calendar.
#' @param location Textual information about location name
#' @param year Year to get the calendar
#' @param longitude Longitude to search for
#' @param latitude Latitude to search for
#' @param tz The time zone to format the local time
#' @family Periodic patterns of indices
#' @description Annual tide information.\cr
#' The columns are: Location, Longitude, Latitude, Phase, DateTime.local, DateTime.UTC, Tide.meter\cr
#' This function uses an API linking xtide software 
#' (https://flaterco.com/xtide/) with tide.info() function.\cr
#' You must have a working internet connection for this function.
#' @keywords Tide
#' @examples
#' \dontrun{
#' library("HelpersMG")
#' Location <- "Les Hattes"
#' Year <- 2010
#' tide <- tide.info(Location, Year, tz="America/Cayenne")
#' plot(tide[, "DateTime.local"], tide[, "Tide.meter"], 
#'      type="l", bty="n", las=1, 
#'      main=tide[1, "Location"], 
#'      xlab=as.character(Year), ylab="Tide level in meter")
#' 
#' Location <- "Hawaii"
#' Year <- 2010
#' tide <- tide.info(Location, Year)
#' 
#' Location <- "Hanamaulu Bay, Kauai Island, Hawaii"
#' Year <- 2010
#' tide <- tide.info(Location, Year)
#' plot(tide[, "DateTime.local"], tide[, "Tide.meter"], 
#'      type="l", bty="n", las=1, 
#'      main=tide[1, "Location"], 
#'      xlab=as.character(Year), ylab="Tide level in meter")
#'      
#' tide <- tide.info(year=2010, longitude=-32, latitude=-4)
#' library(maps)
#' map(database = "world", regions = "Brazil", asp=1, 
#'     xlim=c(-80, -30), ylim=c(-33, 5))
#' points(tide[1, "Longitude"], tide[1, "Latitude"], col="red", pch=19)
#' points(-32, -4, col="blue", pch=19)
#' axis(1)
#' axis(2, las=1)
#' 
#' # Show the locations with data    
#' library(maps)
#' map(xlim=c(-180, 180), ylim=c(-90, 90))
#' title("Locations with harmonics data")
#' axis(1, at=seq(from=-180, to=180, by=45))
#' axis(2, las=1, at=seq(from=-90, to=90, by=15))
#' points(getFromNamespace(x="tide_location", ns="HelpersMG")[, c("longitude")], 
#'        getFromNamespace(x="tide_location", ns="HelpersMG")[, c("latitude")], 
#'        pch=".", col="red", cex=2)
#' # Another example
#' tikei_lon  <- (-144.5465183)
#' tikei_lat <- -14.9505897
#' Year <- 2021
#' tikei_tide <- tide.info(year=Year, longitude=tikei_lon, latitude=tikei_lat)
#' plot(tikei_tide[, "DateTime.local"], tikei_tide[, "Tide.meter"], 
#'      type="l", bty="n", las=1, 
#'      main=tikei_tide[1, "Location"], 
#'      xlab=as.character(Year), ylab="Tide level in meter")
#' }
#' @export

tide.info <- function(location=NULL, year=2021, longitude=NULL, latitude=NULL, tz=NULL) {
  
  tide_location <- getFromNamespace(x="tide_location", ns="HelpersMG")
  
  if (is.null(location) & (is.null(longitude) | is.null(latitude))) {
    stop("Location or longitude/latitude must be provided.")
  }
  
  if (!is.null(location)) {
    
    pos <- which(grepl(location, tide_location$index))
    
    if (identical(pos, integer(0))) {
      stop("No location is found")
    }
    
    if (length(pos) != 1) {
      if (all(tide_location$name[pos[1]] == tide_location$name[pos[-1]])) {
        pos_time <- which.max(tide_location$date_imported[pos])
        pos <- pos[pos_time[1]]
      } else {
        print(tide_location$name[pos])
        stop("More that one location are found; be more precise.")
      }
    }
    
  } else {
    indLl <- sqrt((tide_location$latitude-latitude)^2+(tide_location$longitude-longitude)^2)
    pos <- which.min(abs(indLl))[1]
  }
  
  location <- tide_location$name[pos]
  message(paste0("The location ", location, " has been chosen."))
  print(tide_location[pos, c("name", "longitude", "latitude", "timezone", "country")])
  
  Begin <- paste0(as.character(year), "-01-01 00:00")
  End <- paste0(as.character(year), "-12-31 23:59")
  
  file <- tempfile(pattern = "file", tmpdir = tempdir(), fileext = ".txt")
  
  com <- paste0('curl --silent -o ', file, ' --data-urlencode "location=', location,'" --data-urlencode "begin=', Begin,'" --data-urlencode "end=', End,'" "http://134.158.74.46:20000/tide"')
  out <- system(command = com, intern = FALSE)
  out <- readChar(file, nchars=file.info(file)$size)

  out <- strsplit(out, split = "\",\"")[[1]]
  out <- gsub("\\[\"", "", out)
  out <- gsub("\"\\]", "", out)
  out <- strsplit(out, split = ",")
  
  out_df <- data.frame(Location=character(), 
                       Date=character(), Time=character(), 
                       Tide=character(), Phase=character())
  
  for (i in 1:length(out)) out_df <- rbind(out_df, as.data.frame(matrix(out[[i]], nrow=1)))
  out <- out_df
  rm(out_df)
  colnames(out) <- c("Location", "Date", "Time", "Tide", "Phase")
  
  out <- cbind(out, DateTime=paste(out$Date, out$Time))
  
  loc <- Sys.getlocale("LC_TIME")
  Sys.setlocale("LC_TIME", "C") 
  
  out <- cbind(out, DateTime.local=strptime(out$DateTime, format = "%Y-%m-%d %I:%M %p", tz=tide_location$timezone[pos]))
  Sys.setlocale("LC_TIME", loc) 
  
  if (!is.null(tz)) {
    out <- cbind(out, DateTime.UTC=convert.tz(out$DateTime.local, tz="UTC"))
  } else {
    out <- cbind(out, DateTime.UTC=convert.tz(out$DateTime.local, tz=tz))
  }
  
  
  out <- cbind(out, Longitude=tide_location$longitude[pos])
  out <- cbind(out, Latitude=tide_location$latitude[pos])
  
  
  out <- cbind(out, Tide.meter=as.numeric(gsub(" ft", "", out$Tide))*0.3084)[, c("Location", "Longitude", "Latitude", 
                                                                                 "Phase", "DateTime.local", "DateTime.UTC", "Tide.meter")]
  out <- na.omit(out)
  
  return(out)
}
