% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RandomFromHessianOrMCMC.R
\name{RandomFromHessianOrMCMC}
\alias{RandomFromHessianOrMCMC}
\title{Random numbers based on Hessian matrix or MCMC}
\usage{
RandomFromHessianOrMCMC(
  se = NULL,
  Hessian = NULL,
  mcmc = NULL,
  chain = 1,
  regularThin = TRUE,
  MinMax = NULL,
  fitted.parameters = NULL,
  fixed.parameters = NULL,
  method = NULL,
  probs = c(0.025, 0.5, 0.975),
  replicates = 10000,
  fn = NULL,
  silent = TRUE,
  ParTofn = "par",
  ...
)
}
\arguments{
\item{se}{A nammed vector with SE of parameters}

\item{Hessian}{An Hessian matrix}

\item{mcmc}{A result from MHalgogen()}

\item{chain}{MCMC chain to be used}

\item{regularThin}{If TRUE, use regular thin for MCMC}

\item{MinMax}{A data.frame with at least two columns: Min and Max and rownames being the variable names}

\item{fitted.parameters}{The fitted parameters}

\item{fixed.parameters}{The fixed parameters}

\item{method}{Can be NULL, "SE", "Hessian", "MCMC", or "PseudoHessianFromMCMC"}

\item{probs}{Probability for quantiles}

\item{replicates}{Number of replicates to generate the randoms}

\item{fn}{The function to apply to each replicate}

\item{silent}{Should the function display some information}

\item{ParTofn}{Name of the parameter to send random values to fn}

\item{...}{Parameters send to fn function}
}
\value{
Returns a list with three data.frames named random, fn, and quantiles
}
\description{
If it is very long, use silent parameter to check if something goes wrong.\cr
If replicates is null or is 0, or if method is NULL, parameters are just copied into data.frame.
}
\details{
RandomFromHessianOrMCMC returns random numbers based on Hessian matrix or MCMC
}
\examples{
\dontrun{
library(HelpersMG)
val <- rnorm(100, mean=20, sd=5)+(1:100)/10
# Return -ln L of values in val in Gaussian distribution with mean and sd in par
fitnorm <- function(par, data) {
  -sum(dnorm(data, par["mean"], abs(par["sd"]), log = TRUE))
}
# Initial values for search
p<-c(mean=20, sd=5)
# fit the model
result <- optim(par=p, fn=fitnorm, data=val, method="BFGS", hessian=TRUE)
# Using Hessian
df <- RandomFromHessianOrMCMC(Hessian=result$hessian, 
                              fitted.parameters=result$par, 
                              method="Hessian")$random
hist(df[, 1], main="mean")
hist(df[, 2], main="sd")
plot(df[, 1], df[, 2], xlab="mean", ylab="sd", las=1, bty="n")

# Using MCMC
parameters_mcmc <- data.frame(Density=c('dnorm', 'dlnorm'), 
Prior1=c(10, 0.5), Prior2=c(2, 0.5), SDProp=c(0.35, 0.2), 
Min=c(-3, 0), Max=c(100, 10), Init=c(10, 2), stringsAsFactors = FALSE, 
row.names=c('mean', 'sd'))
# Use of trace and traceML parameters
# trace=1 : Only one likelihood is printed
mcmc_run <- MHalgoGen(n.iter=50000, parameters=parameters_mcmc, data=val, 
parameters_name = "par", 
likelihood=fitnorm, n.chains=1, n.adapt=100, thin=1, trace=1)
df <- RandomFromHessianOrMCMC(mcmc=mcmc_run, fitted.parameters=NULL, 
                              method="MCMC")$random
hist(df[, 1], main="mean")
hist(df[, 2], main="sd")
plot(df[, 1], df[, 2], xlab="mean", ylab="sd", las=1, bty="n")

# Using a function fn
fitnorm <- function(par, data, x) { 
  y=par["a"]*(x)+par["b"]
  -sum(dnorm(data, y, abs(par["sd"]), log = TRUE))
}
p<-c(a=0.1, b=20, sd=5)
# fit the model
x <- 1:100
result <- optim(par=p, fn=fitnorm, data=val, x=x, method="BFGS", hessian=TRUE)
# Using Hessian
df <- RandomFromHessianOrMCMC(Hessian=result$hessian, fitted.parameters=result$par, 
                              method="Hessian", 
                              fn=function(par) (par["a"]*(x)+par["b"]))
plot(1:100, val)
lines(1:100, df$quantiles["50\%", ])
lines(1:100, df$quantiles["2.5\%", ], lty=2)
lines(1:100, df$quantiles["97.5\%", ], lty=2)
}
}
\author{
Marc Girondot
}
