% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/microHIMA.R
\name{microHIMA}
\alias{microHIMA}
\title{High-dimensional mediation analysis for compositional microbiome data}
\usage{
microHIMA(X, Y, OTU, COV = NULL, FDPcut = 0.05)
}
\arguments{
\item{X}{a vector of exposure.}

\item{Y}{a vector of outcome.}

\item{OTU}{a \code{data.frame} or \code{matrix} of high-dimensional compositional OTUs (mediators). Rows represent samples, 
columns represent variables.}

\item{COV}{a \code{data.frame} or \code{matrix} of adjusting covariates. Rows represent samples, columns represent microbiome variables. 
Can be \code{NULL}.}

\item{FDPcut}{FDP (false discovery proportions) cutoff applied to define and select significant mediators. Default = \code{0.05}.}
}
\value{
A data.frame containing mediation testing results of selected mediators (FDP < \code{FDPcut}). 
\itemize{
    \item{ID: }{index of selected significant mediator.}
    \item{alpha: }{coefficient estimates of exposure (X) --> mediators (M).}
    \item{alpha_se: }{standard error for alpha.}
    \item{beta: }{coefficient estimates of mediators (M) --> outcome (Y) (adjusted for exposure).}
    \item{beta_se: }{standard error for beta}
    \item{p_FDP: }{false discovery proportions of selected significant mediator.}
}
}
\description{
\code{microHIMA} is used to estimate and test high-dimensional mediation effects for compositional microbiome data.
}
\examples{
## Generate simulated survival data
n <- 200 # Sample size
p <- 25 # Number of microbiome
Treatment = rbinom(n, 1, 0.2) # binary outcome

## Generate two covariates, one binary, one continuous
covariates = cbind(sample(c(1,0), n, replace = TRUE), rnorm(n))

## parameters
beta0 = as.numeric(matrix(0, 1, p))
betaT = rep(0, p)
betaT[c(1, 2, 3)] = c(1, 1.2, 1.5) # let the first three are non-zero
betaX = matrix(0, p, 2)

alpha0 = 0
alphaT = 1
alphaZ = alphaC = rep(0, p)
alphaZ[c(1, 2, 3)] = c(1.3, -0.7, -0.6) # let the first three are non-zero for response
alphaX = c(0.5, 0.5)

## Generate microbiome data
X = cbind(rep(1, n), covariates, Treatment) # n * (1 + q + p)
b = cbind(beta0, betaX, betaT) # p * (1 + q + p)
gamma.simu = exp(X \%*\% t(b)) # n * p
otu.com = t(apply(gamma.simu, 1, HIMA:::rdirichlet, n = 1)) # Dirichlet distribution

## Generate outcome data
X = cbind(rep(1, n),Treatment, covariates, log(otu.com), log(otu.com) * Treatment)
b = c(alpha0, alphaT, alphaX, alphaZ, alphaC)
outcome = c(b \%*\% t(X) + rnorm(n, mean = 0, sd = 1))
exposure = t(t(Treatment))

\dontrun{
microHIMA.fit <- microHIMA(X = exposure, Y = outcome, OTU = otu.com, COV = covariates)
microHIMA.fit
}

}
\references{
Zhang H, Chen J, Feng Y, Wang C, Li H, Liu L. Mediation effect selection in high-dimensional and compositional microbiome data. 
Stat Med. 2021. DOI: 10.1002/sim.8808. PMID: 33205470; PMCID: PMC7855955.

Zhang H, Chen J, Li Z, Liu L. Testing for mediation effect with application to human microbiome data. 
Stat Biosci. 2021. DOI: 10.1007/s12561-019-09253-3. PMID: 34093887; PMCID: PMC8177450.
}
