% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GaSPUtils.R
\name{GaSP}
\alias{GaSP}
\title{Building, fitting, predicting for a GaSP model}
\usage{
GaSP(
  formula = ~1,
  output,
  input,
  param,
  smooth.est = FALSE,
  input.new = NULL,
  cov.model = list(family = "CH", form = "isotropic"),
  model.fit = "Cauchy_prior",
  prior = list(),
  proposal = list(range = 0.35, tail = 2, nugget = 0.8, nu = 0.8),
  nsample = 5000,
  burnin = 1000,
  opt = NULL,
  bound = NULL,
  dtype = "Euclidean",
  verbose = TRUE
)
}
\arguments{
\item{formula}{an object of \code{formula} class that specifies regressors; see \code{\link[stats]{formula}} for details.}

\item{output}{a numerical vector including observations or outputs in a GaSP}

\item{input}{a matrix including inputs in a GaSP}

\item{param}{a list including values for regression parameters, covariance parameters, 
and nugget variance parameter.
The specification of \strong{param} should depend on the covariance model. 
\itemize{
\item{The regression parameters are denoted by \strong{coeff}. Default value is \eqn{\mathbf{0}}.}
\item{The marginal variance or partial sill is denoted by \strong{sig2}. Default value is 1.}
\item{The nugget variance parameter is denoted by \strong{nugget} for all covariance models. 
Default value is 0.}
\item{For the Confluent Hypergeometric class, \strong{range} is used to denote the range parameter \eqn{\beta}. 
\strong{tail} is used to denote the tail decay parameter \eqn{\alpha}. \strong{nu} is used to denote the 
smoothness parameter \eqn{\nu}.}
\item{For the generalized Cauchy class, \strong{range} is used to denote the range parameter \eqn{\phi}. 
\strong{tail} is used to denote the tail decay parameter \eqn{\alpha}. \strong{nu} is used to denote the 
smoothness parameter \eqn{\nu}.}
\item{For the Matérn class, \strong{range} is used to denote the range parameter \eqn{\phi}. 
\strong{nu} is used to denote the smoothness parameter \eqn{\nu}. When \eqn{\nu=0.5}, the 
Matérn class corresponds to the exponential covariance.}  
\item{For the powered-exponential class, \strong{range} is used to denote the range parameter \eqn{\phi}.
\strong{nu} is used to denote the smoothness parameter. When \eqn{\nu=2}, the powered-exponential class
corresponds to the Gaussian covariance.}
}}

\item{smooth.est}{a logical value indicating whether smoothness parameter will be estimated.}

\item{input.new}{a matrix of new input locations}

\item{cov.model}{a list of two strings: \strong{family}, \strong{form}, where \strong{family} indicates the family of covariance functions 
including the Confluent Hypergeometric class, the Matérn class, the Cauchy class, the powered-exponential class. \strong{form} indicates the 
specific form of covariance structures including the isotropic form, tensor form, automatic relevance determination form. 
\describe{
\item{\strong{family}}{
\describe{
\item{CH}{The Confluent Hypergeometric correlation function is given by 
\deqn{C(h) = \frac{\Gamma(\nu+\alpha)}{\Gamma(\nu)} 
\mathcal{U}\left(\alpha, 1-\nu, \left(\frac{h}{\beta}\right)^2\right),}
where \eqn{\alpha} is the tail decay parameter. \eqn{\beta} is the range parameter.
\eqn{\nu} is the smoothness parameter. \eqn{\mathcal{U}(\cdot)} is the confluent hypergeometric
function of the second kind. For details about this covariance, 
see Ma and Bhadra (2019) at \url{https://arxiv.org/abs/1911.05865}.  
}
\item{cauchy}{The generalized Cauchy covariance is given by
\deqn{C(h) = \left\{ 1 + \left( \frac{h}{\phi} \right)^{\nu}  
            \right\}^{-\alpha/\nu},}
where \eqn{\phi} is the range parameter. \eqn{\alpha} is the tail decay parameter.
\eqn{\nu} is the smoothness parameter with default value at 2.
}

\item{matern}{The Matérn correlation function is given by
\deqn{C(h)=\frac{2^{1-\nu}}{\Gamma(\nu)} \left( \frac{h}{\phi} \right)^{\nu} 
\mathcal{K}_{\nu}\left( \frac{h}{\phi} \right),}
where \eqn{\phi} is the range parameter. \eqn{\nu} is the smoothness parameter. 
\eqn{\mathcal{K}_{\nu}(\cdot)} is the modified Bessel function of the second kind of order \eqn{\nu}.
}
\item{exp}{The exponential correlation function is given by 
\deqn{C(h)=\exp(-h/\phi),}
where \eqn{\phi} is the range parameter. This is the Matérn correlation with \eqn{\nu=0.5}.
}
\item{matern_3_2}{The Matérn correlation with \eqn{\nu=1.5}.}
\item{matern_5_2}{The Matérn correlation with \eqn{\nu=2.5}.}


\item{powexp}{The powered-exponential correlation function is given by
               \deqn{C(h)=\exp\left\{-\left(\frac{h}{\phi}\right)^{\nu}\right\},}
where \eqn{\phi} is the range parameter. \eqn{\nu} is the smoothness parameter.
}
\item{gauss}{The Gaussian correlation function is given by 
\deqn{C(h)=\exp\left(-\frac{h^2}{\phi^2}\right),}
where \eqn{\phi} is the range parameter.
 }
}
}

\item{\strong{form}}{
\describe{
 \item{isotropic}{This indicates the isotropic form of covariance functions. That is,
 \deqn{C(\mathbf{h}) = C^0(\|\mathbf{h}\|; \boldsymbol \theta),} where \eqn{\| \mathbf{h}\|} denotes the 
Euclidean distance or the great circle distance for data on sphere. \eqn{C^0(\cdot)} denotes 
any isotropic covariance family specified in \strong{family}.}
 \item{tensor}{This indicates the tensor product of correlation functions. That is, 
\deqn{ C(\mathbf{h}) = \prod_{i=1}^d C^0(|h_i|; \boldsymbol \theta_i),}
where \eqn{d} is the dimension of input space. \eqn{h_i} is the distance along the \eqn{i}th input dimension. This type of covariance structure has been often used in Gaussian process emulation for computer experiments.
}
 \item{ARD}{This indicates the automatic relevance determination form. That is, 
\deqn{C(\mathbf{h}) = C^0\left(\sqrt{\sum_{i=1}^d\frac{h_i^2}{\phi^2_i}}; \boldsymbol \theta \right),}
where \eqn{\phi_i} denotes the range parameter along the \eqn{i}th input dimension.}
 }
}

}}

\item{model.fit}{a string indicating the choice of priors on correlation parameters:
\describe{
\item{Cauchy_prior}{This indicates that a fully Bayesian approach with objective priors is used
for parameter estimation, where location-scale parameters are assigned with constant priors and 
correlation parameters are assigned with half-Cauchy priors (default). 
}
\item{Ref_prior}{This indicates that a fully Bayesian approach with objective priors is used
for parameter estimation, where location-scale parameters are assigned with constant priors and 
correlation parameters are assigned with reference priors. This is only supported for isotropic
covariance functions. For details, see \code{\link{gp.mcmc}}.
}
\item{Beta_prior}{This indicates that a fully Bayesian approach with subjective priors is used
for parameter estimation, where location-scale parameters are assigned with constant priors and 
correlation parameters are assigned with \link{beta} priors parameterized as \eqn{Beta(a, b, lb, ub)}.
In the beta distribution, \strong{lb} and \strong{ub} are the support for correlation parameters, and
they should be determined based on domain knowledge. \strong{a} and \strong{b} are two shape parameters with default values at 1,
corresponding to the uniform prior over the support \eqn{(lb, ub)}. 
}
\item{MPLE}{This indicates that the \emph{maximum profile likelihood estimation} 
(\strong{MPLE}) is used.}
\item{MMLE}{This indicates that the \emph{maximum marginal likelihood estimation} 
(\strong{MMLE}) is used.}
\item{MAP}{This indicates that the marginal/integrated posterior is maximized.}
}}

\item{prior}{a list containing tuning parameters in prior distribution. This is used only if a subjective Bayes estimation method with informative priors is used.}

\item{proposal}{a list containing tuning parameters in proposal distribution. This is used only if a Bayes estimation method is used.}

\item{nsample}{an integer indicating the number of MCMC samples.}

\item{burnin}{an integer indicating the burn-in period.}

\item{opt}{a list of arguments to setup the \code{\link[stats]{optim}} routine. Current implementation uses three arguments: 
\describe{
 \item{method}{The optimization method: \code{Nelder-Mead} or \code{L-BFGS-B}.}
\item{lower}{The lower bound for parameters.}
\item{upper}{The upper bound for parameters.}
}}

\item{bound}{Default value is \code{NULL}. Otherwise, it should be a list
containing the following elements depending on the covariance class:
\describe{
\item{\strong{nugget}}{a list of bounds for the nugget parameter.
It is a list containing lower bound \strong{lb} and 
upper bound \strong{ub} with default value 
\code{list(lb=0, ub=Inf)}.}
\item{\strong{range}}{a list of bounds for the range parameter. It has default value
\code{range=list(lb=0, ub=Inf)} for the Confluent Hypergeometric covariance, the Matérn covariance, exponential covariance, Gaussian 
covariance, powered-exponential covariance, and Cauchy covariance. The log of range parameterization
 is used: \eqn{\log(\phi)}.}
\item{\strong{tail}}{a list of bounds for the tail decay parameter. It has default value
\code{list(lb=0, ub=Inf)}} for the Confluent Hypergeometric covariance and the Cauchy covariance.
 \item{\strong{nu}}{a list of bounds for the smoothness parameter. It has default value 
\code{list(lb=0, ub=Inf)} for the Confluent Hypergeometric covariance and the Matérn covariance.
when the powered-exponential or Cauchy class 
is used, it has default value \strong{nu}=\code{list(lb=0, ub=2)}. 
This can be achieved by specifying the \strong{lower} bound in \code{opt}.}
}}

\item{dtype}{a string indicating the type of distance:
\describe{
\item{Euclidean}{Euclidean distance is used. This is the default choice.}
\item{GCD}{Great circle distance is used for data on sphere.}
}}

\item{verbose}{a logical value. If it is \code{TRUE}, the MCMC progress bar is shown.}
}
\value{
a list containing the \code{S4} object \linkS4class{gp} and prediction results
}
\description{
This function serves as a wrapper to build, fit, and make prediction 
for a Gaussian process model. It calls on functions \code{\link{gp}}, \code{\link{gp.mcmc}},
\code{\link{gp.optim}}, \code{\link{gp.predict}}.
}
\examples{

code = function(x){
y = (sin(pi*x/5) + 0.2*cos(4*pi*x/5))*(x<=9.6) + (x/10-1)*(x>9.6) 
return(y)
}
n=100
input = seq(0, 20, length=n)
XX = seq(0, 20, length=99)
Ztrue = code(input)
set.seed(1234)
output = Ztrue + rnorm(length(Ztrue), sd=0.1)

# fitting a GaSP model with the objective Bayes approach
fit = GaSP(formula=~1, output, input,  
          param=list(range=3, nugget=0.1, nu=2.5), 
          smooth.est=FALSE, input.new=XX,
          cov.model=list(family="matern", form="isotropic"),
          proposal=list(range=.35, nugget=.8, nu=0.8),
          dtype="Euclidean", model.fit="Cauchy_prior", nsample=50, 
          burnin=10, verbose=TRUE)

}
\seealso{
\code{\link{GPBayes-package}}, \code{\link{gp}}, \code{\link{gp.mcmc}}, \code{\link{gp.optim}}, \code{\link{gp.predict}}
}
\author{
Pulong Ma \email{mpulong@gmail.com}
}
