#' Compute the centrality metrics for the nodes composing the network generated by the NetVis function
#'
#' @param data A data frame containing the relationship between the two groups to be represented in the network 
#' @param g1 Name of the column containing the labels of the first group that will be used to create the network
#' @param g2 Name of the column containing the labels of the second group that will be used to create the network
#' @details This function returns the following centrality metrics for each node that composed the network: Degree (The number of edges incident to the node), Betweenness (The fraction of shortest paths between pairs of nodes that pass through the node), Closeness (The inverse of the sum of the shortest path distances from the node to all other nodes), and Eigenvector Centrality (The centrality measure based on the eigenvector of the adjacency matrix).     
#' @return A data frame with the centrality metrics for each node in the network. 
#' @importFrom igraph graph.empty
#' @importFrom igraph graph.data.frame
#' @importFrom igraph V
#' @importFrom igraph degree
#' @importFrom igraph betweenness
#' @importFrom igraph closeness
#' @importFrom igraph eigen_centrality
#' @name NetCen
#' @export

NetCen<-function(data,g1,g2){
  
  data<-data[!is.na(data[,g1]),]
  
  data<-data[!is.na(data[,g2]),]
  
  g <- igraph::graph.empty()
  
  g <- igraph::graph.data.frame(data[,c(g1,g2)], directed = FALSE)
  
  degree <- igraph::degree(g)
  
  betweenness <- igraph::betweenness(g)
  
  closeness <- igraph::closeness(g)
  
  eigenvector <- igraph::eigen_centrality(g)$vector
  
  out.g<-data.frame(node=names(V(g)),degree=degree,betweenness=betweenness,closeness=closeness,eigenvector=eigenvector)
  
  return(out.g)
  
}