### FuzzyR - Fuzzy Membership Functions

#' @title Fuzzy membership function generator
#' @description
#' To generate the corresponding membership function f(x), also called fuzzy set,
#' according to type and parameters
#' @param mf.type The membership function type
#' @param mf.params The parameters for a membership function
#' @return The desired type of membership function f(x),
#' where x is a generic element of U, which is the universe of discourse for a fuzzy set
#' @examples
#' mf <- genmf(gbellmf, c(1,2,3))
#' evalmf(1:10, mf)
#' @author Chao Chen
#' @export

genmf <- function(mf.type, mf.params) {
    FUN <- match.fun(mf.type)
    FUN(mf.params)
}


#' @title Evaluate fuzzy membership function with membership function type and parameters
#' @description
#' To obtain the corresponding membership grade(s) for crisp input(s) x
#' @param x A generic element of U, which is the universe of discourse for a fuzzy set
#' @param mf.type The member function type
#' @param mf.params The parameters for a member function
#' @return Membership grade(s)
#' @examples
#' evalmftype(5, mf.type=gbellmf, mf.params=c(1,2,3))
#' evalmftype(1:10, mf.type=gbellmf, mf.params=c(1,2,3))
#' @author Chao Chen
#' @export

evalmftype <- function(x, mf.type, mf.params) {
    MF <- genmf(mf.type, mf.params)
    sapply(c(MF), function(F) F(x))
}


#' @title Evaluate fuzzy membership function
#' @description
#' To obtain the corresponding membership grade(s) for the crsip input(s) x
#'
#' @param ... This function has accepted these arguments namely; x, mf.type, mf.params and mf. See the explanation on details section.
#' @details
#' This function involved such as these arguments:\cr
#'
#' x - A generic element of U, which is the universe of discourse for a fuzzy set \cr
#' mf.type - The type of fuzzy membership function\cr
#' mf.params - The parameters for the given type of membership function\cr
#' mf - the membership function generated by \code{\link{genmf}}
#'
#' This function can be used in two ways in order to obtain the membership grade(s) (see the examples section) : \cr\cr
#' 1. evalmf(x, mf.type, mf.params)\cr
#' 2. evalmf(x,mf)
#' @return Membership grade(s)
#' @examples
#' evalmf(5, mf.type=gbellmf, mf.params=c(1,2,3))
#' evalmf(1:10, mf.type=gbellmf, mf.params=c(1,2,3))
#'
#' mf <- genmf(gbellmf, c(1,2,3))
#' evalmf(5, mf)
#' evalmf(1:10, mf)
#' @author Chao Chen
#' @export
evalmf <- function(...) {
    params <- list(...)
    params.len <- length(params)

    x <- params[[1]]

    if(params.len == 3) {
        MF <- genmf(mf.type=params[[2]], mf.params=params[[3]])

    } else if(params.len == 2) {
        MF <- params[[2]]

    }

   sapply(c(MF), function(F) F(x))
}


#' @title Gaussian bell membership function
#' @description
#' To specify a gaussian bell membership function with a pair of particular parameters
#' @param mf.params The parameters c(a, b, c) for a gaussian bell membership function
#' @return The gaussian bell membership function of x for a given pair of parameters,
#' where x is a generic element of U, which is the universe of discourse of a fuzzy set X
#' @details
#' This is not an external function. It should be used through \code{\link{genmf}}.
#' @examples
#' mf <- gbellmf(c(1,2,3))
#' # This is the same as:
#' mf <- genmf(gbellmf, c(1,2,3))
#'
#' evalmf(5, mf)
#' @author Chao Chen
#' @export

gbellmf <- function(mf.params) {

    if(length(mf.params) != 3) {
        stop("improper parameters for gaussian bell membership function")
    }

    a <- mf.params[1]
    b <- mf.params[2]
    c <- mf.params[3]

    gbellmf <- function(x) {
        1 / ( 1 + (((x - c)/a)^2)^b)
    }
}


#' @title Singleton membership function
#' @description
#' To specify a singleton membership function at the particular point
#' @param mf.params the particular singleton point
#' @return The singleton membership function of x at the particular point,
#' where x is a generic element of U, which is the universe of discourse of a fuzzy set X
#' @details
#' This is not an external function. It should be used through \code{\link{genmf}}.
#' @examples
#' mf <- singletonmf(3)
#' # This is the same as:
#' mf <- genmf(singletonmf, 3)
#'
#' evalmf(1:10, mf)
#' @author Chao Chen
#' @export

singletonmf <- function(mf.params) {

    if(length(mf.params) != 1) {
        stop("improper parameters for singleton membership function")
    }

    singletonmf <- function(x) {
        ifelse(x == mf.params, 1, 0)
    }
}


#' @title Linear membership function
#' @description
#' To specify a 1st order linear membership function with given parameters
#' @param mf.params The linear parameters, which is a vector of the size of input numbers plus 1
#' @return A linear membership function
#' @author Chao Chen
#' @export

linearmf <- function(mf.params) {

    if(length(mf.params) < 2) {
        stop("improper parameters for linear membership function")
    }

    linearmf <- function(x) {
        x %*% mf.params
    }
}


gaussmf <- function(mf.params) {
    sig <- mf.params[1]
    c <- mf.params[2]

    gaussmf <- function(x) {
        exp(-(x - c)^2/(2 * sig^2))
    }
}


trapmf <- function(mf.params) {
    a <- mf.params[1]
    b <- mf.params[2]
    c <- mf.params[3]
    d <- mf.params[4]

    trapmf <- function(x) {
        y <- pmax(pmin( (x-a)/(b-a), 1, (d-x)/(d-c) ), 0)
        y[is.na(y)]= 1; y
    }
}


trimf <- function(mf.params) {
    a <- mf.params[1]
    b <- mf.params[2]
    c <- mf.params[3]

    trimf <- function(x) {
        y <- pmax(pmin( (x-a)/(b-a), (c-x)/(c-b) ), 0)
        y[is.na(y)]= 1; y
    }
}



#' @title Singleton Fuzzification
#' @description
#' To generate a fuzzy membership function based on singleton fuzzification for the given crisp input x
#' @param x the crisp input
#' @param mf.params not used, singleton fuzzification does not need additional parameters
#' @return The singleton MF at the crisp point x
#' @examples
#' mf <- singleton.fuzzification(3)
#' evalmf(1:10, mf)
#' @author Chao Chen
#' @export

singleton.fuzzification <- function(x, mf.params) {
    mf.params <- x
    singletonmf(mf.params)
}


#' @title Gaussian bell fuzzification
#' @description
#' To generate a fuzzy membership function based on Gaussian bell fuzzification for the given crisp input x
#' @param x the crisp input, which will be the parameter c for a gaussian bell membership function
#' @param mf.params the parameters c(a, b) for a gaussian bell membership function
#' @return The gbell MF centred at the crisp point x
#' @examples
#' mf <- gbell.fuzzification(3, c(1,2))
#' # This is the same as:
#' mf <- genmf(gbellmf, c(1,2,3))
#'
#' evalmf(1:10, mf)
#' @author Chao Chen
#' @export

gbell.fuzzification <- function(x, mf.params) {
    mf.params <- c(mf.params, x)
    genmf(gbellmf, mf.params)
}



