% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/maxlogL.R
\name{maxlogL}
\alias{maxlogL}
\title{Maximum Likelihood Estimation}
\usage{
maxlogL(x, dist = "dnorm", fixed = NULL, link = NULL, start = NULL,
  lower = NULL, upper = NULL, optimizer = "nlminb", control = NULL,
  ...)
}
\arguments{
\item{x}{a vector with data to be fitted. This argument must be a matrix with hierarchical distributions.}

\item{dist}{a length-one character vector with the name of density/mass function of interest. The default value
is "dnorm", to compute maximum likelihood estimators of normal distribution.}

\item{fixed}{a list with fixed/known parameters of distribution of interest. Fixed parameters
must be passed with its name.}

\item{link}{a list with names of parameters to be linked, and names of the link object. For names of parameters, please
visit documentation of density function. There are three link functions available: \code{\link{log_link}},
\code{\link{logit_link}} and \code{\link{NegInv_link}}.}

\item{start}{a numeric vector with initial values for the parameters to be estimated.}

\item{lower}{a numeric vector with lower bounds, with the same lenght of argument `start` (for box-constrained optimization).}

\item{upper}{a numeric vector with upper bounds, with the same lenght of argument `start` (for box-constrained optimization).}

\item{optimizer}{a lenght-one character vector with the name of optimization routine. \code{\link{nlminb}}, \code{\link{optim}}
and \code{\link[DEoptim]{DEoptim}} are available; \code{\link{nlminb}} is the default.
routine.}

\item{control}{control parameters of the optimization routine. Please, visit documentation of selected
optimizer for further information.}

\item{...}{Further arguments to be supplied to the optimizer.}
}
\value{
A list with class \code{"maxlogL"} containing the following
 lists:
\item{fit}{A list with output information about estimation and method used.}
\item{inputs}{A list with all input arguments.}
\item{outputs}{A list with number of parameters, sample size and standard error method.}
}
\description{
Function to compute maximum likelihood estimators (MLE)
of any distribution implemented in \code{R}.
}
\details{
\code{maxlogL} calculates computationally the likelihood function corresponding to
the distribution specified in argument \code{dist} and maximizes it through
\code{\link{optim}}, \code{\link{nlminb}} or \code{\link{DEoptim}}. \code{maxlogL}
generates an S3 object of class \code{maxlogL}.
}
\examples{
# Estimation with one fixed parameter
x <- rnorm(n = 10000, mean = 160, sd = 6)
theta_1 <- maxlogL(x = x, dist = 'dnorm', control = list(trace = 1),
                 link = list(over = "sd", fun = "log_link"),
                 fixed = list(mean = 160))
summary(theta_1)

# Both parameters of normal distribution mapped with logarithmic function
theta_2 <- maxlogL( x = x, dist = "dnorm",
                    link = list(over = c("mean","sd"),
                                fun = c("log_link","log_link")) )
summary(theta_2)

# Parameter estimation in ZIP distribution
library(gamlss.dist)
z <- rZIP(n=10000, mu=6, sigma=0.08)
theta_3  <- maxlogL( x = z, dist='dZIP', start = c(0, 0), lower = c(-Inf, -Inf),
                     upper = c(Inf, Inf), optimizer = 'optim',
                     link = list(over=c("mu", "sigma"),
                     fun = c("log_link", "logit_link")) )
summary(theta_3)

}
\references{
\insertRef{Nelder1965}{EstimationTools}

\insertRef{Fox1978}{EstimationTools}

\insertRef{Nash1979}{EstimationTools}

\insertRef{Dennis1981}{EstimationTools}
}
\seealso{
\code{\link{summary.maxlogL}}, \code{\link{optim}}, \code{\link{nlminb}},
         \code{\link{DEoptim}}, \code{\link{DEoptim.control}}
}
\author{
Jaime Mosquera Gutiérrez, \email{jmosquerag@unal.edu.co}
}
