% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EGM.R
\name{EGM}
\alias{EGM}
\title{Exploratory Graph Model}
\usage{
EGM(
  data,
  EGM.model = c("standard", "EGA"),
  communities = NULL,
  structure = NULL,
  search = FALSE,
  p.in = NULL,
  p.out = NULL,
  opt = c("AIC", "BIC", "CFI", "chisq", "logLik", "RMSEA", "SRMR", "TEFI", "TEFI.adj",
    "TLI"),
  constrained = TRUE,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{data}{Matrix or data frame.
Should consist only of variables to be used in the analysis.
Can be raw data or a correlation matrix}

\item{EGM.model}{Character vector (length = 1).
Sets the procedure to conduct \code{EGM}.
Available options:

\itemize{

\item \code{"EGA"} (default) --- Applies \code{\link[EGAnet]{EGA}} to obtain the
(sparse) regularized network structure, communities, and memberships

\item \code{"standard"} --- Applies the standard EGM model which
estimates communities based on the non-regularized empirical partial
correlation matrix and sparsity is set using \code{p.in} and \code{p.out}

}}

\item{communities}{Numeric vector (length = 1).
Number of communities to use for the \code{"standard"} type of EGM.
Defaults to \code{NULL}.
Providing no input will use the communities and memberships output
from the Walktrap algorithm (\code{\link[igraph]{cluster_walktrap}}) based
on the empirical non-regularized partial correlation matrix}

\item{structure}{Numeric or character vector (length = \code{ncol(data)}).
Can be theoretical factors or the structure detected by \code{\link[EGAnet]{EGA}}.
Defaults to \code{NULL}}

\item{search}{Boolean (length = 1).
Whether a search over parameters should be conducted.
Defaults to \code{FALSE}.
Set to \code{TRUE} to select a model over a variety of parameters that
optimizes the \code{opt} objective}

\item{p.in}{Numeric vector (length = 1).
Probability that a node is randomly linked to other nodes in the same community.
Within community edges are set to zero based on \code{quantile(x, prob = 1 - p.in)}
ensuring the lowest edge values are set to zero (i.e., most probable to \emph{not}
be randomly connected).
Only used for \code{EGM.type = "standard"}.
Defaults to \code{NULL} but must be set}

\item{p.out}{Numeric vector (length = 1).
Probability that a node is randomly linked to other nodes \emph{not} in the same community.
Between community edges are set to zero based on \code{quantile(x, prob = 1 - p.out)}
ensuring the lowest edge values are set to zero (i.e., most probable to \emph{not}
be randomly connected).
Only used for \code{EGM.type = "standard"} and \code{search = FALSE}.
Defaults to \code{NULL} but must be set}

\item{opt}{Character vector (length = 1).
Fit index used to select from when searching over models
(only applies to \code{search = TRUE}).
Available options include:

\itemize{

\item \code{"AIC"}

\item \code{"BIC"}

\item \code{"CFI"}

\item \code{"chisq"}

\item \code{"logLik"}

\item \code{"RMSEA"}

\item \code{"SRMR"}

\item \code{"TEFI"}

\item \code{"TEFI.adj"}

\item \code{"TLI"}

}

Defaults to \code{"SRMR"}}

\item{constrained}{Boolean (length = 1).
Whether memberships of the communities should
be added as a constraint when optimizing the network loadings.
Defaults to \code{TRUE} which ensures assigned loadings are
guaranteed to never be smaller than any cross-loadings.
Set to \code{FALSE} to freely estimate each loading similar to exploratory factor analysis}

\item{verbose}{Boolean (length = 1).
Should progress be displayed?
Defaults to \code{TRUE}.
Set to \code{FALSE} to not display progress}

\item{...}{Additional arguments to be passed on to
\code{\link[EGAnet]{auto.correlate}},
\code{\link[EGAnet]{network.estimation}},
\code{\link[EGAnet]{community.detection}},
\code{\link[EGAnet]{community.consensus}},
\code{\link[EGAnet]{community.unidimensional}},
\code{\link[EGAnet]{EGA}}, and
\code{\link[EGAnet]{net.loads}}}
}
\description{
Function to fit the Exploratory Graph Model
}
\examples{
# Get depression data
data <- depression[,24:44]

# Estimate EGM (using EGA)
egm_ega <- EGM(data)

# Estimate EGM (using EGA) specifying communities
egm_ega_communities <- EGM(data, communities = 3)

# Estimate EGM (using EGA) specifying structure
egm_ega_structure <- EGM(
  data, structure = c(
    1, 1, 1, 2, 1, 1, 1,
    1, 1, 1, 3, 2, 2, 2,
    2, 3, 3, 3, 3, 3, 2
  )
)

# Estimate EGM (using standard)
egm_standard <- EGM(
  data, EGM.model = "standard",
  communities = 3, # specify number of communities
  p.in = 0.95, # probability of edges *in* each community
  p.out = 0.80 # probability of edges *between* each community
)

\dontrun{
# Estimate EGM (using EGA search)
egm_ega_search <- EGM(
  data, EGM.model = "EGA", search = TRUE
)

# Estimate EGM (using EGA search and AIC criterion)
egm_ega_search_AIC <- EGM(
  data, EGM.model = "EGA", search = TRUE, opt = "AIC"
)

# Estimate EGM (using search)
egm_search <- EGM(
  data, EGM.model = "standard", search = TRUE,
  communities = 3, # need communities or structure
  p.in = 0.95 # only need 'p.in'
)}

}
\author{
Hudson F. Golino <hfg9s at virginia.edu> and Alexander P. Christensen <alexpaulchristensen@gmail.com>
}
