




#' @title Chapter 4
#' 
#' @description 
#' 
#' Functions for Chapter 4, \emph{Probability Distributions} of Wayne W. Daniel's 
#' \emph{Biostatistics: A Foundation for Analysis in the Health Sciences}, Tenth Edition.
#' 
#' @param size,prob see \code{\link[stats]{dbinom}}
#' 
#' @param lambda see \code{\link[stats]{dpois}}
#' 
#' @param xlim length-2 \link[base]{numeric} vector, horizontal limit of the figure.
#' 
#' @return 
#' 
#' \code{\link{binomBar}} generates a bar plot of a Binomial distribution.
#' 
#' \code{\link{poisBar}} generates a bar plot of a Poisson distribution.
#' 
#' @references
#' 
#' Wayne W. Daniel, \emph{Biostatistics: A Foundation for Analysis in the Health Sciences}, Tenth Edition.
#' Wiley, ISBN: 978-1-119-62550-6.
#' 
#' @examples 
#' # Page 93, Example 4.2.1 - Page 97, Example 4.2.7
#' (d421 = rep(1:8, times = c(62L, 47L, 39L, 39L, 58L, 37L, 4L, 11L)))
#' (tab_421 = table(d421))
#' (prob_421 = tab_421 / sum(tab_421))
#' (cumprob_421 = cumsum(tab_421) / sum(tab_421))
#' noquote(cbind(
#'  Freq = tab_421, 
#'  'Relative Freq' = sprintf(fmt = '%.1f%%', prob_421 * 100), 
#'  'Cumulative Relative Freq' = sprintf(fmt = '%.1f%%', cumprob_421 * 100)), right = TRUE)
#' # Page 94, Table 4.2.2 and Page 96, Table 4.2.3
#' barplot(prob_421, main = 'Relative Frequence Bar Plot; Page 95, Figure 4.2.1')
#' plot(stepfun(x = seq_along(tab_421), y = c(0, cumprob_421)), 
#'  main = 'Cumulative Relative Frequence; Page 96, Figure 4.2.2')
#'  
#' # Page 99, Example 4.3.1
#' dbinom(x = 3L, size = 5L, prob = .858)
#' # Page 103, Example 4.3.2
#' dbinom(x = 4L, size = 10L, prob = .14)
#' # Page 103, Example 4.3.3
#' (pL = pbinom(q = 5L, size = 25L, prob = .1, lower.tail = TRUE)) # (a) including!
#' (pU = pbinom(q = 5L, size = 25L, prob = .1, lower.tail = FALSE)) # (b) excluding!
#' pL + pU # R makes sure they add up to 1
#' binomBar(size = 25L, prob = .1)
#' # Page 103, Example 4.3.4
#' dbinom(x = 7L, size = 12L, prob = .55)
#' pbinom(q = 5L, size = 12L, prob = .55)
#' pbinom(q = 7L, size = 12L, prob = .55, lower.tail = FALSE)
#' 
#' # Page 110, Example 4.4.1
#' dpois(x = 3L, lambda = 12) 
#' # Page 110, Example 4.4.2
#' ppois(2L, lambda = 12, lower.tail = FALSE) 
#' # Page 110, Example 4.4.3
#' ppois(1L, lambda = 2) 
#' # Page 111, Example 4.4.4
#' dpois(3L, lambda = 2)
#' # Page 112, Example 4.4.5
#' ppois(5L, lambda = 2, lower.tail = FALSE)
#'
#' # Page 119. Example 4.6.1
#' pnorm(2)
#' # Page 120. Example 4.6.2
#' pnorm(2.55) - pnorm(-2.55)
#' 1 - 2 * pnorm(-2.55) # alternative solution
#' # Page 121. Example 4.6.3
#' pnorm(1.53) - pnorm(-2.74)
#' # Page 121. Example 4.6.4
#' pnorm(2.71, lower.tail = FALSE)
#' # Page 122. Example 4.6.5
#' pnorm(2.45) - pnorm(.84)
#' 
#' # Page 122. Example 4.7.1
#' pnorm(q = 3, mean = 5.4, sd = 1.3)
#' pnorm(q = (3-5.4)/1.3) # manual solution
#' # Page 125. Example 4.7.2
#' pnorm(649, mean = 491, sd = 119) - pnorm(292, mean = 491, sd = 119)
#' # Page 122. Example 4.7.3
#' 1e4L * pnorm(8.5, mean = 5.4, sd = 1.3, lower.tail = FALSE)
#' 
#' @name Chp4
#' @export
binomBar <- function(size, prob, xlim = size) {
  if (!is.integer(size) || length(size) != 1L) stop('size must be len-1 integer')
  if (!is.numeric(prob) || length(prob) != 1L || is.na(prob) || prob < 0 || prob > 1) stop('prob must be between 0,1')
  if (!is.integer(xlim)) stop('xlim must be integer')
  nl <- length(xlim)
  xl <- if (nl == 1L) 0:xlim else if (nl == 2L) xlim[1L]:xlim[2L] else stop('illegal xlim')
  pr <- dbinom(setNames(nm = xl), size = size, prob = prob)
  barplot(pr, main = sprintf(fmt = 'Binomial Distribution: n = %d; p = %.1f', size, prob), ylab = 'Density')
  return(invisible())
}

#' @rdname Chp4
#' @export
poisBar <- function(lambda, xlim) {
  if (!is.numeric(lambda) || length(lambda) != 1L || is.na(lambda) || lambda < 0) stop('lambda must be len-1 positive numeric')
  if (!is.integer(xlim)) stop('xlim must be integer')
  nl <- length(xlim)
  xl <- if (nl == 1L) 0:xlim else if (nl == 2L) xlim[1L]:xlim[2L] else stop('illegal xlim')
  pr <- dpois(setNames(nm = xl), lambda = lambda)
  barplot(pr, main = sprintf(fmt = 'Poisson Distribution: lambda = %.1f', lambda), ylab = 'Density')
  return(invisible())
}

