% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Mechanisms.R
\name{LaplaceMechanism}
\alias{LaplaceMechanism}
\title{Laplace Mechanism}
\usage{
LaplaceMechanism(true.values, eps, sensitivities, alloc.proportions = NULL)
}
\arguments{
\item{true.values}{Real number or numeric vector corresponding to the true
value(s) of the desired function.}

\item{eps}{Positive real number defining the epsilon privacy parameter.}

\item{sensitivities}{Real number or numeric vector corresponding to the
l1-global sensitivity(-ies) of the function(s) generating true.values. This
value must be of length 1 or of the same length as true.values. If it is of
length 1 and true.values is a vector, this indicates that the given
sensitivity applies simultaneously to all elements of true.values and that
the privacy budget need not be allocated (alloc.proportions is unused in
this case). If it is of the same length as true.values, this indicates that
each element of true.values comes from its own function with different
corresponding sensitivities. In this case, the l1-norm of the provided
sensitivities is used to generate the Laplace noise.}

\item{alloc.proportions}{Optional numeric vector giving the allocation
proportions of epsilon to the function values in the case of vector-valued
sensitivities. For example, if sensitivities is of length two and
alloc.proportions = c(.75, .25), then 75\% of the privacy budget eps is
allocated to the noise computation for the first element of true.values,
and the remaining 25\% is allocated to the noise computation for the second
element of true.values. This ensures eps-level privacy across all
computations. Input does not need to be normalized, meaning
alloc.proportions = c(3,1) produces the same result as the example above.}
}
\value{
Sanitized function values based on the bounded and/or unbounded
definitions of differential privacy, sanitized via the Laplace mechanism.
}
\description{
This function implements the Laplace mechanism for differential privacy by
adding noise to the true value(s) of a function according to specified values
of epsilon and l1-global sensitivity(-ies). Global sensitivity calculated
based either on bounded or unbounded differential privacy can be used
\insertCite{Kifer2011}{DPpack}. If true.values is a vector, the provided
epsilon is divided such that epsilon-differential privacy is satisfied across
all function values. In the case that each element of true.values comes from
its own function with different corresponding sensitivities, a vector of
sensitivities may be provided. In this case, if desired, the user can specify
how to divide epsilon among the function values using alloc.proportions.
}
\examples{
# Simulate dataset
n <- 100
c0 <- 5 # Lower bound
c1 <- 10 # Upper bound
D1 <- stats::runif(n, c0, c1)
epsilon <- 1 # Privacy budget
sensitivity <- (c1-c0)/n

private.mean <- LaplaceMechanism(mean(D1), epsilon, sensitivity)
private.mean

# Simulate second dataset
d0 <- 3 # Lower bound
d1 <- 6 # Upper bound
D2 <- stats::runif(n, d0, d1)
D <- matrix(c(D1,D2),ncol=2)
sensitivities <- c((c1-c0)/n, (d1-d0)/n)
epsilon <- 1 # Total privacy budget for all means

# Here, sensitivities are summed and the result is used to generate Laplace
# noise. This is essentially the same as allocating epsilon proportional to
# the corresponding sensitivity. The results satisfy 1-differential privacy.
private.means <- LaplaceMechanism(apply(D, 2, mean), epsilon, sensitivities)
private.means

# Here, privacy budget is explicitly split so that 75\% is given to the first
# vector element and 25\% is given to the second.
private.means <- LaplaceMechanism(apply(D, 2, mean), epsilon, sensitivities,
                                  alloc.proportions = c(0.75, 0.25))
private.means

}
\references{
\insertRef{Dwork2006a}{DPpack}

\insertRef{Kifer2011}{DPpack}
}
