\name{DMRViterbi}
\alias{DMRViterbi}

\title{
	Viterbi algorithm to estimate posterior probabilities of DMRs.
}
\description{
	This function takes M-values and estimated parameters 
	from 'DMRMark', then uses Viterbi algorithm for 
	estimating states' posterior probabilities for each locus.
}
\usage{
DMRViterbi(mv, pars, L = rep(1, nrow(mv)), starting = NULL, 
			pd = NULL, region = TRUE, 
			orderBy = c("max", "mean", "median", "min"), VitP = NULL)
}

\arguments{
  \item{mv}{The input M-values matrix, NA is not allowed.
}
  \item{pars}{The list of model parameters. Getting by calling 'DMRMark'.
}
  \item{L}{A vector to specify the distance between each probes in bp. 
  $L < 0$ represents
  change of chromosome. Default is $L = 1$ for all probes.
}
  \item{starting}{A vector to specify the position 
  to initial new chains. We suggest new chains
  should be initiated at least at starting of new chromosome.
  When it is null, new chains initiate at beginning and where $L > 100000
  or $L < 0$. 
}
  \item{pd}{A design matrix, which can be generated by 'stats::model.matrix'.
  If the M-values are totally paired or single paired, just leave it to be NULL.
}
  \item{region}{If set to TRUE, this function returns the regions formed by
  Viterbi posterior states. Otherwise, it returns posterior probabilities 
  and states for individual loci. Default is TRUE.
}
  \item{orderBy}{Only enabled when 'region = TRUE'.
  Order the regions by which statistics? Choice include 'max', 'mean', 'median' and 'min', which orders the regions by the maximum, geometric mean, 
  median or minimum of the posterior probabilities in each region respectively.
  Default is 'max'.
}
  \item{VitP}{Only enabled when 'region = FALSE'.
  The minimum posterior probabilities required to be the DMC states.
  A locus with DMC's posterior probability lower than 'VitP' will in the 
  non-DMC states with highest probabilities. When set to NULL, simply return 
  the MAP states. Default is NULL.
}
}

\value{
	If 'region = FALSE', the return value is a list contains:
	\item{states}{The MAP methylation status satisfies the 'VitP'.
}
	\item{deltas}{The matrix with each row corresponds to the posterior
	probabilities of each locus in which states.
}

	If 'region = TRUE', the return value is a dataframe with the following fields:
	\item{begin}{Beginning of each region. In probe index.
}
	\item{ending}{Ending of each region. In probe index.
}
	\item{MAP_state}{The MAP state of each region.
}
	\item{minVP}{The minimum Viterbi posterior probability
	of the MAP state in each region 
}
	\item{meanVP}{The geometric mean of Viterbi posterior probability
	of the MAP state in each region 
}
	\item{maxVP}{The maximum Viterbi posterior probability
	of the MAP state in each region 
}
	\item{midVP}{The median Viterbi posterior probability
	of the MAP state in each region 
}
}

\author{
	Linghao SHEN <sl013@ie.cuhk.edu.hk>
}

\seealso{
	See \code{\link{DMRMark}} about model parameter estimation
}
\examples{

	# DMRViterbi
	# DMR detection performed on chr18 of a small BLCA dataset from TCGA
	data(BLCA)
	
	# Use a small subset
	nprobe <- 500 
	# M-values
	mv <- BLCA$mv[1:nprobe,]
	
	# Distance between probes, L<0 indicates acorssing chromosomes 
	L = BLCA$distance[1:nprobe]
	
	# Initialize new chain when probe distance too long 
	# or across different chromosomes
	newChains <- which((L > 100000) | L < 0)
	# The starting positions of new chains
	starting <- c(1, newChains[-length(newChains)]+1)
	
	# Run DMRMark with default options
	pars <- DMRMark(mv, L, starting)
	# Get the posterior of being certain states
	results <- DMRViterbi(mv, pars, L, starting)
	head(results)
}
