% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/connectivity_estimation.distributions.R
\encoding{UTF-8}
\name{d.rel.conn.dists.func}
\alias{d.rel.conn.dists.func}
\alias{p.rel.conn.dists.func}
\alias{q.rel.conn.dists.func}
\title{Functions for estimating the probability distribution for relative 
connectivity given a pair of distributions for scores for marked and unmarked
individuals}
\usage{
d.rel.conn.dists.func(obs, d.unmarked, d.marked, p = 1, N = max(100,
  min(5000, 2 * length(obs))), prior.shape1 = 0.5,
  prior.shape2 = prior.shape1, prior.func = function(phi) dbeta(phi,
  prior.shape1, prior.shape2), ...)

p.rel.conn.dists.func(obs, d.unmarked, d.marked, p = 1, N = max(100,
  min(5000, 2 * length(obs))), prior.shape1 = 0.5,
  prior.shape2 = prior.shape1, prior.func = function(phi) dbeta(phi,
  prior.shape1, prior.shape2), ...)

q.rel.conn.dists.func(obs, d.unmarked, d.marked, p = 1, N = max(100,
  min(5000, 2 * length(obs))), prior.shape1 = 0.5,
  prior.shape2 = prior.shape1, prior.func = function(phi) dbeta(phi,
  prior.shape1, prior.shape2), ...)
}
\arguments{
\item{obs}{Vector of observed score values for potentially marked individuals}

\item{d.unmarked}{A function representing the PDF of unmarked individuals. 
Must be normalized so that it integrates to 1 for the function to work 
properly.}

\item{d.marked}{A function representing the PDF of marked individuals.  Must 
be normalized so that it integrates to 1 for the function to work properly.}

\item{p}{Fraction of individuals (i.e., eggs) marked in the source 
population. Defaults to 1.}

\item{N}{number of steps between 0 and 1 at which to approximate likelihood 
function as input to \code{\link{approxfun}}. Defaults to
\code{2*length(obs)} so long as that number is comprised between \code{100}
and \code{5000}.}

\item{prior.shape1}{First shape parameter for Beta distributed prior. 
Defaults to 0.5.}

\item{prior.shape2}{Second shape parameter for Beta distributed prior. 
Defaults to being the same as \code{prior.shape1}.}

\item{prior.func}{Function for prior distribution.  Should take one 
parameter, \code{phi}, and return a probability.  Defaults to 
\code{function(phi) dbeta(phi,prior.shape1,prior.shape2)}.  If this is 
specified, then inputs \code{prior.shape1} and \code{prior.shape2} are 
ignored.}

\item{\dots}{Additional arguments for the \code{\link{integrate}} function.}
}
\value{
A function that takes one argument (the relative connectivity for 
  \code{d.rel.conn.dists.func} and \code{p.rel.conn.dists.func}; the quantile
  for \code{q.rel.conn.dists.func}) and returns the probability density, 
  cumulative probability or score value, respectively. The returned function 
  accepts both vector and scalar input values.
}
\description{
These functions return functions that calculate the probability density 
function (\code{d.rel.conn.dists.func}), the probability distribution 
function (aka the cumulative distribution function; 
\code{p.rel.conn.dists.func}) and the quantile function 
(\code{q.rel.conn.dists.func}) for relative connectivity given a set of 
observed score values, distributions for unmarked and marked individuals, and
an estimate of the fraction of all eggs marked at the source site, \code{p}.
}
\details{
The normalization of the probability distribution is carried out using a 
simple, fixed-step trapezoidal integration scheme.  By default, the number of
steps between relative connectivity values of 0 and 1 defaults to 
\code{2*length(obs)} so long as that number is comprised between \code{100} 
and \code{5000}.
}
\section{Functions}{
\itemize{
\item \code{d.rel.conn.dists.func}: Returns a function that is PDF for relative
connectivity

\item \code{p.rel.conn.dists.func}: Returns a function that is cumulative
probability distribution for relative connectivity

\item \code{q.rel.conn.dists.func}: Returns a function that is quantile
function for relative connectivity
}}
\examples{
library(ConnMatTools)
data(damselfish.lods)

# Histograms of simulated LODs
l <- seq(-1,30,0.5)
h.in <- hist(damselfish.lods$in.group,breaks=l)
h.out <- hist(damselfish.lods$out.group,breaks=l)

# PDFs for marked and unmarked individuals based on simulations
d.marked <- stepfun.hist(h.in)
d.unmarked <- stepfun.hist(h.out)

# Fraction of adults genotyped at source site
p.adults <- 0.25

# prior.shape1=1 # Uniform prior
prior.shape1=0.5 # Jeffreys prior

# Fraction of eggs from one or more genotyped parents
p <- dual.mark.transmission(p.adults)$p

# PDF for relative connectivity
D <- d.rel.conn.dists.func(damselfish.lods$real.children,
                           d.unmarked,d.marked,p,
                           prior.shape1=prior.shape1)

# Estimate most probable value for relative connectivity
phi.mx <- optim.rel.conn.dists(damselfish.lods$real.children,
                                    d.unmarked,d.marked,p)$phi

# Estimate 95\% confidence interval for relative connectivity
Q <- q.rel.conn.dists.func(damselfish.lods$real.children,
                           d.unmarked,d.marked,p,
                           prior.shape1=prior.shape1)

# Plot it up
phi <- seq(0,1,0.001)
plot(phi,D(phi),type="l",
     xlim=c(0,0.1),
     main="PDF for relative connectivity",
     xlab=expression(phi),
     ylab="Probability density")

abline(v=phi.mx,col="green",lty="dashed")
abline(v=Q(c(0.025,0.975)),col="red",lty="dashed")
}
\author{
David M. Kaplan \email{dmkaplan2000@gmail.com}
}
\references{
Kaplan DM, Cuif M, Fauvelot C, Vigliola L, Nguyen-Huu T, Tiavouane J and Lett C 
  (in press) Uncertainty in empirical estimates of marine larval connectivity. 
  ICES Journal of Marine Science. doi:10.1093/icesjms/fsw182.
}
\seealso{
Other connectivity.estimation: \code{\link{d.rel.conn.beta.prior}},
  \code{\link{d.rel.conn.finite.settlement}},
  \code{\link{d.rel.conn.multinomial.unnorm}},
  \code{\link{d.rel.conn.multiple}},
  \code{\link{d.rel.conn.unif.prior}},
  \code{\link{dual.mark.transmission}},
  \code{\link{optim.rel.conn.dists}},
  \code{\link{r.marked.egg.fraction}}
}

