#' Reinsert Safely Removed Taxa Into A Tree
#'
#' @description
#'
#' Following Safe Taxonomic Reduction reinsert removed taxa.
#'
#' @param treefile.in A Newick-formatted tree file containing tree(s) without safely removed taxa.
#' @param treefile.out A file name where the newly generated trees will be written out to (required).
#' @param str.list The safe taxonomic reduction table as generated by \link{SafeTaxonomicReduction}.
#' @param multi.placements An optional for what to do with taxa that have more than one possible reinsertion position. Options are \code{"exclude"} (does not reinsert them; the default) or \code{"random"} (picks one of the possible positions and uses that; will vary stochastically if multiple trees exist).
#'
#' @details
#'
#' TEXT NEEDED!
#'
#' @return
#'
#' Nothing is returned, but a new file (\code{treefile.out}) is written.
#'
#' @author Graeme T. Lloyd \email{graemetlloyd@@gmail.com}
#'
#' @examples
#'
#' # Nothing yet
#'
#' @export SafeTaxonomicReinsertion
SafeTaxonomicReinsertion <- function(treefile.in, treefile.out, str.list, multi.placements = "exclude") {
  
  # Add some data checks!!!!!
  
  # Ensure str list is formatted as characters:
  str.list <- cbind(as.character(str.list[, "Junior"]), as.character(str.list[, "Senior"]), as.character(str.list[, "Rule"]))
  
  # Re-add column names:
  colnames(str.list) <- c("Junior", "Senior", "Rule")
  
  # Read in tree file as text:
  text <- readLines(treefile.in)
  
  # Remove spaces to ensure grep/gsub works properly later:
  text <- gsub(" ", "", text)
  
  # Resort str list by Junior taxon:
  str.list <- str.list[order(str.list[, "Junior"]), ]
  
  # Find number of seniors for each junior:
  names.and.numbers <- rle(as.character(str.list[, "Junior"]))
  
  # Make list of taxa that have a single senior:
  single.replacements <- names.and.numbers$values[grep(TRUE, names.and.numbers$lengths == 1)]
  
  # Collapse to just those not in a polytomy with other taxa in the str list:
  single.replacements <- single.replacements[is.na(match(single.replacements, str.list[, "Senior"]))]
  
  # For taxa that have a single replacement:
  if(length(single.replacements) > 0) {
    
    # For each single replacement taxon:
    for(i in 1:length(single.replacements)) {
      
      # Reinsert into tree next to its senior:
      text <- gsub(str.list[match(single.replacements[i], str.list[, "Junior"]), "Senior"], paste("(", paste(str.list[match(single.replacements[i], str.list[, "Junior"]), c("Junior", "Senior")], collapse = ","), ")", sep = ""), text)
      
    }
    
    # Remove single replacement taxa from str list:
    str.list <- str.list[-match(single.replacements, str.list[, "Junior"]), ]
    
    # Update names and numbers now taxa have been removed:
    names.and.numbers <- rle(as.character(sort(str.list[, "Junior"])))
    
  }
  
  # Only worth contuning if there are names still to reinsert:
  if(length(names.and.numbers$values) > 0) {
    
    # Vector to store taxa that only occur in a single polytomy:
    polytomy.taxa <- vector(mode = "character")
    
    # For each taxon:
    for(i in 1:length(names.and.numbers$values)) {
      
      # Get taxon name:
      taxon.name <- names.and.numbers$values[i]
      
      # Find its seniors:
      seniors <- str.list[grep(TRUE, str.list[, "Junior"] == taxon.name), "Senior"]
      
      # If its seniors, except for one, are all also juniors then record it (this finds taxa that only exist in a single polytomy in the original tree):
      if(length(grep(TRUE, is.na(match(seniors, str.list[, "Junior"])))) <= 1) polytomy.taxa <- c(polytomy.taxa, taxon.name)
      
    }
    
    # If there are taxa that only occur in a single polytomy:
    if(length(polytomy.taxa) > 0) {
      
      # Reorder from most seniors to least:
      taxa.to.delete <- polytomy.taxa <- polytomy.taxa[order(names.and.numbers$lengths[match(polytomy.taxa, names.and.numbers$values)], decreasing=TRUE)]
      
      # Whilst there are still polytomous taxa in the list:
      while(length(polytomy.taxa) > 0) {
        
        # Get taxon name with most juniors:
        taxon.name <- polytomy.taxa[1]
        
        # Find all of its seniors:
        seniors <- str.list[grep(TRUE, str.list[, "Junior"] == taxon.name), "Senior"]
        
        # Find senior taxon (already in tree):
        senior.taxon <- seniors[grep(TRUE, is.na(match(seniors, str.list[, "Junior"])))]
        
        # Replace senior with all juniors in polytomy:
        text <- gsub(senior.taxon, paste("(", paste(sort(c(taxon.name, seniors)), collapse = ","), ")", sep = ""), text)
        
        # Remove taxa just dealt with from polytomy.taxa:
        polytomy.taxa <- polytomy.taxa[-sort(match(c(taxon.name, seniors), polytomy.taxa))]
        
      }
      
      # Trims str list down to remaining taxa:
      for(i in 1:length(taxa.to.delete)) str.list <- str.list[-grep(TRUE, str.list[, "Junior"] == taxa.to.delete[i]), , drop = FALSE]
      
    }
    
  }
  
  # Only keep going if there are taxa still to reinsert:
  if(length(str.list[, 1]) > 0) {
    
    # If the user wishes to reinsert remaining taxa at random:
    if(multi.placements == "random") {
      
      # List unique juniors:
      unique.juniors <- rle(str.list[, "Junior"])$values[order(rle(str.list[, "Junior"])$lengths)]
      
      # For each junior taxon remaining:
      for(i in 1:length(unique.juniors)) {
        
        # Isolate junior taxon name:
        junior <- unique.juniors[i]
        
        # Get a senior for each tree:
        seniors <- sample(str.list[grep(TRUE, str.list[, "Junior"] == junior), "Senior"], length(text), replace = TRUE)
        
        # Remove junior from str list:
        str.list <- str.list[-grep(TRUE, str.list[, "Junior"] == junior), ]
        
        # Make replacements:
        replacements <- paste("(", paste(junior, seniors, sep = ","), ")", sep = "")
        
        # For each tree:
        for(j in 1:length(text)) {
          
          # Isolate tree:
          tree.text <- text[j]
          
          # Case if senior ends in a parenthesis:
          if(length(grep(paste(seniors[j], ")", sep = ""), tree.text)) > 0) {
            
            # Replace senior with combination of junior and senior:
            tree.text <- gsub(paste(seniors[j], ")", sep = ""), paste(replacements[j], ")", sep = ""), tree.text)
            
          }
          
          # Case if senior ends in a comma:
          if(length(grep(paste(seniors[j], ",", sep = ""), tree.text)) > 0) {
            
            # Replace senior with combination of junior and senior:
            tree.text <- gsub(paste(seniors[j], ",", sep = ""), paste(replacements[j], ",", sep = ""), tree.text)
            
          }
          
          # Update tree text:
          text[j] <- tree.text
          
        }
        
      }
      
      # Set uninserted list to null (i.e. empty):
      uninserted <- NULL
      
    }
    
    # If the user wishes to exclude remaining taxa:
    if(multi.placements == "exclude") {
      
      # Set uninserted list to remaining juniors:
      uninserted <- unique(str.list[, "Junior"])
      
    }
    
  # If taxa all have a single reinsertion position:
  } else {
    
    # Set uninserted list to null (i.e. empty):
    uninserted <- NULL
    
  }
  
  # Write out tree file:
  write(text, treefile.out)
  
  # Compile output:
  output <- list(uninserted)
  
  # Name output:
  names(output) <- "unreinserted.taxa"
  
  # Return output:
  return(output)
  
}
