\name{BiCopLambda}       
\alias{BiCopLambda}

\title{Lambda-function (plot) for bivariate copula data}

\description{
This function plots the lambda-function of given bivariate copula data.
}

\usage{
BiCopLambda(u1=NULL, u2=NULL, family="emp", par=0, par2=0, 
            PLOT=TRUE, ...)
}

\arguments{
  \item{u1,u2}{Data vectors of equal length with values in [0,1] (default: \code{u1} and \code{u2 = NULL}).}
  \item{family}{An integer defining the bivariate copula family or indicating the empirical lambda-function: \cr
    \code{"emp"} = empirical lambda-function (default) \cr
    \code{1} = Gaussian copula; the theoretical lambda-function is simulated (no closed formula available) \cr
    \code{2} = Student t copula (t-copula); the theoretical lambda-function is simulated (no closed formula available) \cr
    \code{3} = Clayton copula \cr
    \code{4} = Gumbel copula \cr
    \code{5} = Frank copula \cr
    \code{6} = Joe copula \cr 
    \code{7} = BB1 copula \cr
    \code{8} = BB6 copula \cr
    \code{9} = BB7 copula \cr
    \code{10} = BB8 copula}
  \item{par}{Copula parameter; if the empirical lambda-function is chosen, \code{par = NULL} or \code{0} (default).}
  \item{par2}{Second copula parameter for t-, BB1, BB6, BB7 and BB8 copulas (default: \code{par2 = 0}).}
  \item{PLOT}{Logical; whether the results are plotted. If \code{PLOT = FALSE}, the values \cr \code{empLambda} and/or \code{theoLambda} are returned (see below; default: \code{PLOT = TRUE}).}
  \item{...}{Additional plot arguments.}
}

\value{
	\item{empLambda}{If the empirical lambda-function is chosen and \code{PLOT=FALSE}, a vector of the empirical lambda's is returned.}
	\item{theoLambda}{If the theoretical lambda-function is chosen and \code{PLOT=FALSE}, a vector of the theoretical lambda's is returned.}
}


\note{
The \eqn{\lambda}-function is characteristic for each bivariate copula family and defined by Kendall's distribution function \eqn{K}:
\deqn{
\lambda(v,\theta) := v - K(v,\theta)
}
with
\deqn{
K(v,\theta) := P(C_{\theta}(U_1,U_2) \leq v),\ \ v\in [0,1].
}{
K(v,\theta) := P(C_{\theta}(U_1,U_2) <= v),  v \in [0,1].
}
For Archimedean copulas one has the following closed form expression in terms of the generator function \eqn{\varphi}{\phi} of the copula \eqn{C_{\theta}}:
\deqn{
\lambda(v,\theta) = \frac{\varphi(v)}{\varphi '(v)},
}{
\lambda(v,\theta) = \phi(v) / \phi'(v),
}
where \eqn{\varphi '}{\phi'} is the derivative of \eqn{\varphi}{\phi}.
For more details see Genest and Rivest (1993) or Schepsmeier (2010).

For the bivariate Gaussian and t-copula no closed form expression for the theoretical \eqn{\lambda}-function exists. 
Therefore it is simulated based on samples of size 1000.
For all other implemented copula families there are closed form expressions available.

The plot of the theoretical \eqn{\lambda}-function also shows the limits of the \eqn{\lambda}-function corresponding to Kendall's tau \eqn{=0} 
and Kendall's tau \eqn{=1} (\eqn{\lambda=0}).

For rotated bivariate copulas one has to transform the input arguments \code{u1} and/or \code{u2}.
In particular, for copulas rotated by 90 degrees \code{u1} has to be set to \code{1-u1},
for 270 degrees \code{u2} to \code{1-u2} and for survival copulas \code{u1} and \code{u2} to \code{1-u1} and \code{1-u2}, respectively.
Then \eqn{\lambda}-functions for the corresponding non-rotated copula families can be considered.
}

\author{Ulf Schepsmeier}

\references{
Genest, C. and L.-P. Rivest (1993).
Statistical inference procedures for bivariate Archimedean copulas.
Journal of the American Statistical Association, 88 (423), 1034-1043.

Schepsmeier, U. (2010).
Maximum likelihood estimation of C-vine pair-copula constructions based on bivariate copulas from different families.
Diploma thesis, Technische Universitaet Muenchen.\cr
\url{http://mediatum.ub.tum.de/doc/1079296/1079296.pdf}.
}

\seealso{\code{\link{BiCopMetaContour}}, \code{\link{BiCopKPlot}}, \code{\link{BiCopChiPlot}}}

\examples{
\dontrun{
# Clayton and rotated Clayton copulas
n = 1000
tau = 0.5

# simulate from Clayton copula
fam = 3	
theta = BiCopTau2Par(fam,tau)
dat = BiCopSim(n,fam,theta)

# create lambda-function plots
dev.new(width=16,height=5)
par(mfrow=c(1,3))
BiCopLambda(dat[,1],dat[,2])	# empirical lambda-function	
BiCopLambda(family=fam,par=theta)	# theoretical lambda-function
BiCopLambda(dat[,1],dat[,2],family=fam,par=theta)	# both

# simulate from rotated Clayton copula (90 degrees)
fam = 23  
theta = BiCopTau2Par(fam,-tau)
dat = BiCopSim(n,fam,theta)

# rotate the data to standard Clayton copula data
rot_dat = 1-dat[,1]

dev.new(width=16,height=5)
par(mfrow=c(1,3))
BiCopLambda(rot_dat,dat[,2])  # empirical lambda-function	
BiCopLambda(family=3,par=-theta)	# theoretical lambda-function
BiCopLambda(rot_dat,dat[,2],family=3,par=-theta)	# both
}
}

